;==============================================================================
; This module implements most of the facility extension words of the
; "Forth 2012 optional Facility word set", which are:
;   
;   BEGIN-STRUCTURE
;   END-STRUCTURE
;   +FIELD
;   FIELD:
;   CFIELD:
;   EKEY
;   EKEY?
;   EKEY>CHAR
;   EKEY>FKEY
;   EMIT?
;   K-DELETE
;   K-DOWN
;   K-UP
;   K-LEFT
;   K-RIGHT
;   K-END
;   K-HOME
;   K-INSERT
;   K-NEXT
;   K-PRIOR
;   K-F1
;   K-F2
;   K-F3
;   K-F4
;   K-F5
;   K-F6
;   K-F7
;   K-F8
;
; Assemble this module with the myca command
;   $ myca m4-facility.asm -o m4-facility.bin
;
; Upload the binary module to My4TH with the my4th tool
; (for example. See "my4th --help" for details):
;   $ my4th write /dev/ttyS0 binary 15 m4-facility.bin
;
; Load the module on My4TH with the BLOAD word:
;   15 BLOAD
;
; (15 is a block number in the EEPROM, please choose a suited number
; for your system to avoid overwriting important EEPROM content)
;==============================================================================

#include <my4th/binmod.hsm>


;------------------------------------------------------------------------------
; Exported Words
;------------------------------------------------------------------------------

EXPORTS:
          EXPORT  e_begstruct,  "begin-structure"
          EXPORT  e_cfieldc,    "cfield:"
          EXPORT  e_fieldc,     "field:"
          EXPORT  e_pfield,     "+field"
          EXPORT  e_endstruct,  "end-structure"
          EXPORT  e_kdelete,    "k-delete"
          EXPORT  e_kdown,      "k-down"
          EXPORT  e_kup,        "k-up"
          EXPORT  e_kleft,      "k-left"
          EXPORT  e_kright,     "k-right"
          EXPORT  e_kend,       "k-end"
          EXPORT  e_khome,      "k-home"
          EXPORT  e_kinsert,    "k-insert"
          EXPORT  e_knext,      "k-next"
          EXPORT  e_kprior,     "k-prior"
          EXPORT  e_f1,         "k-f1"
          EXPORT  e_f2,         "k-f2"
          EXPORT  e_f3,         "k-f3"
          EXPORT  e_f4,         "k-f4"
          EXPORT  e_f5,         "k-f5"
          EXPORT  e_f6,         "k-f6"
          EXPORT  e_f7,         "k-f7"
          EXPORT  e_f8,         "k-f8"
          EXPORT  c_key,        "ekey"
          EXPORT  c_keyq,       "ekey?"
          EXPORT  c_ekeychar,   "ekey>char"
          EXPORT  c_ekeyfkey,   "ekey>fkey"
          EXPORT  e_emitq,      "emit?"
          EEND


;------------------------------------------------------------------------------
; Imported Words
;------------------------------------------------------------------------------

IMPORTS:
c_create  IMPORT  "create"
c_over    IMPORT  "over"
c_swap    IMPORT  "swap"
c_here    IMPORT  "here"
c_dup     IMPORT  "dup"
c_comma   IMPORT  ","
c_does    IMPORT  "does>"
c_at      IMPORT  "@"
c_key     IMPORT  "key"
c_keyq    IMPORT  "key?"
c_add     IMPORT  "+"
c_store   IMPORT  "!"
          IEND


;------------------------------------------------------------------------------
; Code Section
;------------------------------------------------------------------------------

CODE:

init_module:
          ;Nothing to do here.
          RET


e_begstruct:
          ;Implementation of the Forth word BEGIN-STRUCTURE  ( -- addr 0 ; -- size )
          PHL
          JSR  c_create
          JSR  c_here
          LDA  #0
          JSR  push_accu
          JSR  c_dup
          JSR  c_comma
          JSR  c_does
          JSR  c_at
          RTS


e_endstruct:
          ;Implementation of the Forth word END-STRUCTURE  ( addr n -- )
          PHL
          JSR  c_swap
          JSR  c_store
          RTS


e_pfield:
          ;Implementation of the Forth word +FIELD
          PHL
_pfield   JSR  c_create
          JSR  c_over
          JSR  c_comma
          JSR  c_add
          JSR  c_does
          JSR  c_at
          JSR  c_add
          RTS


e_fieldc:
          ;Implementation of the Forth word FIELD:
          PHL
          LDA  #2
          JSR  push_accu
          JMP  _pfield


e_cfieldc:
          ;Implementation of the Forth word CFIELD:
          PHL
          LDA  #1
          JSR  push_accu
          JMP  _pfield


; Key codes of special keys
KCODE_DEL     SET 0x7F
KCODE_MIN     SET 0x8E

;0x80(128)~0x9F(159): Special keys, for vt100 and I2C keyboard
KCODE_PGUP    SET 0x8E  ; Shift+Up 			142
KCODE_PGDOWN  SET 0x8F  ; Shift+Down		143	
KCODE_F1      SET 0x90  ; Ctrl+1~Ctrl+8		144
KCODE_F2      SET 0x91	;					145
KCODE_F3      SET 0x92	;					146
KCODE_F4      SET 0x93	;					147
KCODE_F5      SET 0x94	;					148
KCODE_F6      SET 0x95	;					149
KCODE_F7      SET 0x96	;					150
KCODE_F8      SET 0x97	;					151
KCODE_INSERT  SET 0x98  ; Ctrl+/			152
KCODE_HOME    SET 0x99  ; Shift+Left		153
KCODE_END     SET 0x9A  ; Shift+Right		154

KCODE_UP      SET 0x9C  ;					156
KCODE_DOWN    SET 0x9D  ;					157
KCODE_RIGHT   SET 0x9E  ;					158
KCODE_LEFT    SET 0x9F  ;					159

KCODE_MAX     SET 0x9F


e_kdelete:
          ;Implementation of the Forth word K-DELETE
          LDA  #KCODE_DEL
          JMP  push_accu

e_kdown:
          ;Implementation of the Forth word K-DOWN
          LDA  #KCODE_DOWN
          JMP  push_accu

e_kup:
          ;Implementation of the Forth word K-UP
          LDA  #KCODE_UP
          JMP  push_accu

e_kleft:
          ;Implementation of the Forth word K-LEFT
          LDA  #KCODE_LEFT
          JMP  push_accu

e_kright:
          ;Implementation of the Forth word K-RIGHT
          LDA  #KCODE_RIGHT
          JMP  push_accu

e_kend:
          ;Implementation of the Forth word K-END
          LDA  #KCODE_END
          JMP  push_accu

e_khome:
          ;Implementation of the Forth word K-HOME
          LDA  #KCODE_HOME
          JMP  push_accu

e_kinsert:
          ;Implementation of the Forth word K-INSERT

          LDA  #KCODE_INSERT
          JMP  push_accu

e_knext:
          ;Implementation of the Forth word K-NEXT
          LDA  #KCODE_PGDOWN
          JMP  push_accu

e_kprior:
          ;Implementation of the Forth word K-PRIOR
          LDA  #KCODE_PGUP
          JMP  push_accu
  
e_f1:
          ;Implementation of the Forth word K-F1
          LDA  #KCODE_F1
          JMP  push_accu

e_f2:
          ;Implementation of the Forth word K-F2
          LDA  #KCODE_F2
          JMP  push_accu

e_f3:
          ;Implementation of the Forth word K-F3
          LDA  #KCODE_F3
          JMP  push_accu

e_f4:
          ;Implementation of the Forth word K-F4
          LDA  #KCODE_F4
          JMP  push_accu

e_f5:
          ;Implementation of the Forth word K-F5
          LDA  #KCODE_F5
          JMP  push_accu

e_f6:
          ;Implementation of the Forth word K-F6
          LDA  #KCODE_F6
          JMP  push_accu

e_f7:
          ;Implementation of the Forth word K-F7
          LDA  #KCODE_F7
          JMP  push_accu

e_f8:
          ;Implementation of the Forth word K-F8
          LDA  #KCODE_F8

push_accu:
          ;push the accumulator to the data stack
          STA  R4_L
          LD   R4_H,#0
          JMP  rom_push_data_R4


c_ekeychar:
          ;Implementation of the Forth word EKEY>CHAR  ( x -- x false | char true )
          PHL
          JSR  ekey_common
          LDA  #0
          JPF  _cekyc01
ret_true: LDA  #0xFF
_cekyc01  STA  R4_L
          STA  R4_H
          JSR  rom_push_data_R4
          RTS


c_ekeyfkey:
          ;Implementation of the Forth word EKEY>FKEY  ( x -- u flag )
          PHL
          JSR  ekey_common
          JPF  ret_true
          LDA  #0
          JMP  _cekyc01


ekey_common:
          PHL
          JSR  rom_pop_data_R4
          JSR  rom_push_data_R4
          LDA  R4_L
          ;check if special key
          CMP  #KCODE_DEL
          JPF  rts_opc  ; return FLAG=1
          SU   #KCODE_MIN
          JNF  rts_opc  ; return FLAG=0
          SU   #KCODE_MAX-KCODE_MIN+1
          INC  FLAG
          ;Flag = 1 if special key code
rts_opc   RTS


e_emitq:
          ;Implementation of the Forth word EMIT?  ( -- flag )
          PHL
          JMP  ret_true  ; always true, because UART / LCD is always ready for output


;------------------------------------------------------------------------------
END_MODULE
