;==============================================================================
; This module implements most of the floating point words of the
; "Forth 2012 The optional Floating-Point word set", which are:
;
;   >FLOAT         F@             FMAX           FABS           FSIN
;   D>F            FALIGN         FMIN           FACOS          FSQRT
;   F!             FALIGNED       FNEGATE        FASIN          FTAN
;   F*             FCONSTANT      FOVER          FATAN          FTRUNC
;   F+             FDEPTH         FROT           FCOS           FVALUE
;   F-             FDROP          FROUND         FE.            PRECISION
;   F/             FDUP           FSWAP          FEXP           S>F
;   F0<            FLITERAL       FVARIABLE      FFIELD:        SET-PRECISION
;   F0=            FLOAT+         REPRESENT      FLN
;   F<             FLOATS         F.             FLOG
;   F>D            FLOOR          F>S            FS.
;
;
; Notes:
;
;   - At least My4TH ROM version 1.2 is required to be able to run this module
;   - This module implements "FlexiFloat" from Jörg Völker (https://forth-ev.de)
;   - The width of the mantissa is 23 bit, 1 bit less than IEEE-754 floats have
;   - Floats on My4TH are really slow
;   - To speed up floating point operations, no rounding is implemented at all
;   - The number display (F., FE., FS. etc.) is limited to 7 significants
;   - The largest number that can be entered and printed is 8388607 (23 bit).
;   - Some mathematical functions (especially exp, ln, log, asin, acos, atan)
;     quickly become inaccurate when the numbers become larger
;   - sin, cos and tan are relatively accurate
;   - The word F** is not implemented, because it would be very inaccurate
;
;
; Assemble this module with the myca command
;   $ myca m4-float.asm -o m4-float.bin
;
; Upload the binary module to My4TH with the my4th tool
; (for example. See "my4th --help" for details):
;   $ my4th write /dev/ttyS0 binary 20 m4-float.bin
;
; Load the module on My4TH with the BLOAD word:
;   20 BLOAD
;
; (20 is a block number in the EEPROM, please choose a suited number
; for your system to avoid overwriting important EEPROM content)
;==============================================================================


#include <my4th/binmod.hsm>


;------------------------------------------------------------------------------
; Exported Words
;------------------------------------------------------------------------------

EXPORTS:
            EXPORT  fl_fplus,     "f+"
            EXPORT  fl_fminus,    "f-"
            EXPORT  fl_fmul,      "f*"
            EXPORT  fl_fdiv,      "f/"
            EXPORT  fl_fprint,    "f."
            EXPORT  fl_store,     "f!"
            EXPORT  fl_load,      "f@"
            EXPORT  fl_todouble,  "f>d"
            EXPORT  fl_tosingle,  "f>s"
            EXPORT  fl_stofloat,  "s>f"
            EXPORT  fl_dtofloat,  "d>f"
            EXPORT  fl_lesszero,  "f0<"
            EXPORT  fl_iszero,    "f0="
            EXPORT  fl_isless,    "f<"
            EXPORT  fl_gtfloat,   ">float"
            EXPORT  fl_fdepth,    "fdepth"
            EXPORT  fl_floatpl,   "float+"
            EXPORT  fl_floats,    "floats"
            EXPORT  fl_represent, "represent"
            EXPORT  fl_floor,     "floor"
            EXPORT  fl_fround,    "fround"
            EXPORT  fl_ftrunc,    "ftrunc"
            EXPORT  fl_feprint,   "fe."
            EXPORT  fl_fsprint,   "fs."
            EXPORT  fl_fabs,      "fabs"
            EXPORT  fl_fmin,      "fmin"
            EXPORT  fl_fmax,      "fmax"
            EXPORT  fl_fnegate,   "fnegate"
            EXPORT  fl_ffield,    "ffield:"
            EXPORT  fl_precision, "precision"
            EXPORT  fl_set_prec,  "set-precision"
            EXPORT  fl_falign,    "falign"
            EXPORT  fl_faligned,  "faligned"
            EXPORT  fl_fconstant, "fconstant"
            EXPORT  fl_fdrop,     "fdrop"
            EXPORT  fl_fdup,      "fdup"
            EXPORTI fl_fliteral,  "fliteral"
            EXPORT  fl_fover,     "fover"
            EXPORT  fl_frot,      "frot"
            EXPORT  fl_fswap,     "fswap"
            EXPORT  fl_fvariable, "fvariable"
            EXPORT  fl_fvalue,    "fvalue"
            EXPORT  fl_fsqrt,     "fsqrt"
            EXPORT  fl_fsin,      "fsin"
            EXPORT  fl_fcos,      "fcos"
            EXPORT  fl_ftan,      "ftan"
            EXPORT  fl_fasin,     "fasin"
            EXPORT  fl_facos,     "facos"
            EXPORT  fl_fatan,     "fatan"
            EXPORT  fl_fexp,      "fexp"
            EXPORT  fl_fln,       "fln"
            EXPORT  fl_flog,      "flog"
            EEND



;------------------------------------------------------------------------------
; Imported Words
;------------------------------------------------------------------------------

IMPORTS:

c_base      IMPORT "base"
c_quit      IMPORT "quit"
c_type      IMPORT "type"
c_depth     IMPORT "depth"
c_2slash    IMPORT "2/"
c_2constant IMPORT "2constant"
c_2drop     IMPORT "2drop"
c_2dup      IMPORT "2dup"
c_2literal  IMPORT "2literal"
c_2over     IMPORT "2over"
c_2rot      IMPORT "2rot"
c_2swap     IMPORT "2swap"
c_2variable IMPORT "2variable"
c_2value    IMPORT "2value"
c_create    IMPORT "create"
c_over      IMPORT "over"
c_comma     IMPORT ","
c_sum16     IMPORT "+"
c_doesgt    IMPORT "does>"
c_at        IMPORT "@"
c_2tor      IMPORT "2>r"
c_2rat      IMPORT "2r@"
c_2rfrom    IMPORT "2r>"
            IEND



;------------------------------------------------------------------------------
; Local Zero-Page Variables
;------------------------------------------------------------------------------

SEG_ZPBINMOD

F1M     DS  3  ;mantissa of floating point number 1 (operand 1)
F1E     DS  1  ;exponent of floating point number 1 (operand 1)
F2M     DS  3  ;mantissa of floating point number 2 (operand 2)
F2E     DS  1  ;exponent of floating point number 2 (operand 2)
F3M     DS  3  ;mantissa of floating point number 3 (result)
F3E     DS  1  ;exponent of floating point number 3 (result)
SPTR    DS  2  ;string pointer
SLEN    DS  1  ;length of the string
SFLG    DS  1  ;flag for number conversion
SIGN    DS  1  ;saved sign of a number
ROUND   DS  1  ;flag for rounding
ESIGN   DS  1  ;for extended functions: sign
ECNT    DS  1  ;for extended functions
EMODE   DS  1  ;for extended functions
ECCNT   DS  1  ;for extended functions: coefficients count
EPM     DS  1  ;for extended functions: plus/minus flag
ECPTR   DS  2  ;for extended functions: pointer to coefficients

SEG_PRGCODE



;------------------------------------------------------------------------------
; Data Section
;------------------------------------------------------------------------------

etxt_overflow   DB  "\n? overflow",0
etxt_precision  DB  "\n? precision",0
etxt_divzero    DB  "\n? division by zero",0
etxt_eover      DB  "OVER ",0
text_romver     DB  "Warning: ROM < v1.2\n",0
text_hello      DB  "\nInstalled: M4-Float v1.0\n",0

tab_10exp   DB  0x00,0x00,0x00,0x0A ; 10
            DB  0x00,0x00,0x00,0x64 ; 100
            DB  0x00,0x00,0x03,0xE8 ; 1000
            DB  0x04,0x00,0x02,0x71 ; 10000
            DB  0x04,0x00,0x18,0x6A ; 100000
            DB  0x04,0x00,0xF4,0x24 ; 1000000
            DB  0x04,0x09,0x89,0x68 ; 10000000
            DB  0x08,0x05,0xF5,0xE1 ; 100000000
            DB  0x08,0x3B,0x9A,0xCA ; 1000000000

tab_sinus   DB  0x8E,0x77,0x5C,0xD7 ; 1/362880
            DB  0x80,0x06,0x68,0xDD ; 1/5040
            DB  0x44,0x44,0x44,0xE3 ; 1/120
            DB  0x55,0x55,0x55,0xE7 ; 1/6

tab_asinus ;DB  0xCE,0xBB,0x44,0xE3 ; 654729075/78033715200
           ;DB  0x94,0xF7,0x4F,0xE3 ; 34459425/3530096640
           ;DB  0xE1,0xA1,0x5E,0xE3 ; 2027025/175472640
            DB  0x00,0x80,0x72,0xE3 ; 135135/9676800
            DB  0x00,0x00,0x48,0xE4 ; 10395/599040
            DB  0x00,0x00,0x5C,0xE4 ; 945/42240
            DB  0x00,0xA0,0x7C,0xE4 ; 105/3456
            DB  0x00,0x70,0x5B,0xE5 ; 15/336
            DB  0x00,0xCD,0x4C,0xE6 ; 3/40
            DB  0x00,0x56,0x55,0xE7 ; 1/6

tab_exp     DB  0x62,0xBB,0x47,0xCD ; 1/(12!)
            DB  0x14,0x99,0x6B,0xD0 ; 1/(11!)
            DB  0x3E,0xF9,0x49,0xD4 ; 1/(10!)
            DB  0x8E,0x77,0x5C,0xD7 ; 1/(9!)
            DB  0x80,0x06,0x68,0xDA ; 1/(8!)
            DB  0x80,0x06,0x68,0xDD ; 1/(7!)
            DB  0xB0,0x05,0x5B,0xE0 ; 1/(6!)
            DB  0x44,0x44,0x44,0xE3 ; 1/(5!)
            DB  0x55,0x55,0x55,0xE5 ; 1/(4!)
            DB  0x55,0x55,0x55,0xE7 ; 1/(3!)
            DB  0x01,0x00,0x00,0xFF ; 1/(2!)

tab_ln      DB  0x18,0x86,0x61,0xE5 ; 1/21
            DB  0x1A,0xCA,0x6B,0xE5 ; 1/19
            DB  0x79,0x78,0x78,0xE5 ; 1/17
            DB  0x45,0x44,0x44,0xE6 ; 1/15
            DB  0xED,0xC4,0x4E,0xE6 ; 1/13
            DB  0x45,0x17,0x5D,0xE6 ; 1/11
            DB  0x1C,0xC7,0x71,0xE6 ; 1/9
            DB  0x92,0x24,0x49,0xE7 ; 1/7
            DB  0x67,0x66,0x66,0xE7 ; 1/5
            DB  0x56,0x55,0x55,0xE8 ; 1/3

tab_pihalf  DB  0xEB,0x87,0x64,0xEA ; 1/2 pi
tab_095     DB  0x99,0x99,0x79,0xE9 ; 0.95
tab_1       DB  0x01,0x00,0x00,0x00 ; 1.0
tab_ln10    DB  0xC6,0xAE,0x49,0xEB ; ln(10)

max_prec    SET 7

sto_prec    DB  max_prec-1
floatstr    DS  20
numbuf      DS  8



;------------------------------------------------------------------------------
; Float module house keeping functions
;------------------------------------------------------------------------------

CODE:  ; marker for start of program code


init_module:
          PHL
          LD   PTR_L,#<text_hello
          LD   PTR_H,#>text_hello
          JSR  rom_print_str
          ;check ROM version
          LD   PTR_L,#0xFF
          LD   PTR_H,#0x7D
          LAP
          AD   #0x11
          SU   #0x12    ;ROM v1.2 or later required
          JPF  _inim01
          LD   PTR_L,#<text_romver
          LD   PTR_H,#>text_romver
          JSR  rom_print_str
_inim01   ;install vector for number parsing on command line
          LDA  #5
          JSR  rom_get_vector
          LD   PTR_L,#<(finpvect+1)
          LD   PTR_H,#>(finpvect+1)
          LDA  R4_L
          SAP
          JSR  IncPtr
          LDA  R4_H
          SAP
          LD   PTR_L,#<floatInputHandler
          LD   PTR_H,#>floatInputHandler
          LD   R4_L,PTR_L
          LD   R4_H,PTR_H
          LDA  #5
          JSR  rom_set_vector
_RTS      RTS


floatInputHandler:
          PHL
          JSR  string_to_float
          JNF  _RTS ;quit early, float number could be read
          ;call the old handler and try to read decimal or hex value
finpvect  JSR  _RTS
          RTS



;------------------------------------------------------------------------------
; Definitions of the new float words
;------------------------------------------------------------------------------

;re-use some double words:
fl_falign     SET _RET
fl_faligned   SET _RET
fl_fconstant  SET c_2constant
fl_fdrop      SET c_2drop    
fl_fdup       SET c_2dup     
fl_fliteral   SET c_2literal 
fl_fover      SET c_2over    
fl_frot       SET c_2rot     
fl_fswap      SET c_2swap    
fl_fvariable  SET c_2variable
fl_fvalue     SET c_2value   


fl_fplus:
          ; "F+" add two floats
          PHL
          JSR  stack_to_F1
          JSR  stack_to_F2
          JSR  float_add
          JMP  F1_to_stack_ret


fl_fminus:
          ; "F-" subtract two floats
          PHL
          JSR  stack_to_F2
          JSR  stack_to_F1
          JSR  float_sub
          JMP  F1_to_stack_ret


fl_fmul:
          ; "F*" multiply two floats
          PHL
          JSR  stack_to_F2
          JSR  stack_to_F1
          JSR  float_mul
          JMP  F1_to_stack_ret


fl_fdiv:
          ; "F/" divide two floats
          PHL
          JSR  stack_to_F2
          JSR  stack_to_F1
          JSR  float_div
          JMP  F1_to_stack_ret


float_to_double:
          ;convert float in F1 to a double number stored in F1M(lsb) and F1E(msb)
          PHL
          ;get the sign and do sign-extension into R1
          LD   R1,#0
          LDA  F1M+2
          ROL
          JNF  _ftd03
          DEC  R1
_ftd03    TST  F1E
          JPF  _ftd04  ;only sign extension required, we are done!
_ftd01    ;check if exponent is positive or negative
          LDA  F1E
          ROL
          JPF  _ftd02
          ;positive exponent, shift mantissa left
          LD   R0,F1E
_ftd05    RWL  F1M+0
          ROL  F1M+2
          ROL  R1
          JLP  _ftd05
_ftd04    LD   F1E,R1
          RTS
_ftd02    ;negative exponent, shift mantissa right
          LD   FLAG,R1
          ROR  F1M+2
          ROR  F1M+1
          ROR  F1M+0
          INC  F1E
          TST  F1E
          JNF  _ftd02
          JMP  _ftd04


fl_tosingle:
          ; "F>S"  convert float to single cell integer
          PHL
          JSR  fl_todouble
          JSR  rom_pop_data_R4
          RTS


fl_todouble:
          ; "F>D"  convert float to double
          PHL
          JSR  fl_fround
          JSR  stack_to_F1
          JSR  float_to_double
          JMP  F1_to_stack_ret


fl_stofloat:
          ; "S>F"  convert single to float
          PHL
          JSR  rom_pop_data_R4
          LDA  R4_H
          ROL
          LD   R5_L,#0
          JNF  _flsf01
          DEC  R5_L
_flsf01   LD   R5_H,R5_L
          JMP  _fldf01


fl_dtofloat:
          ; "D>F"  convert double to float
          PHL
          JSR  rom_pop_data_R5
          JSR  rom_pop_data_R4
_fldf01   ;convert double in R4/R5 to float in R4/R5
          LD   R0,#0
          ;shift double number right until bits22-31 are all 0 or all 1
          LDA  R5_H
          ROL
          LD   R1,#0  ;compare-value, 0x00 or 0xFF
          JNF  _dtf01
          DEC  R1
_dtf01    LDA  R5_H
          CMP  R1
          JPF  _dtf02
          ROL
          ROR  R5_H
          ROR  R5_L
          ROR  R4_H
          ROR  R4_L
          INC  R0
          JMP  _dtf01
_dtf02    ;ensure that bit 23 has the correct value
          LDA  R5_L
          XOR  R1
          ROL
          JNF  _dtf03
          ROR  R1
          ROR  R5_L
          ROR  R4_H
          ROR  R4_L
          INC  R0
_dtf03    ;save the exponent
          LD   R5_H,R0
          JMP  pushR4R5ret


fl_store:
          ; "F!"  store float
          PHL
          JSR  rom_pop_data_R4
          LD   PTR_L,R4_L
          LD   PTR_H,R4_H
          JSR  rom_pop_data_R5
          JSR  rom_pop_data_R4
          LDA  R4_L
          SAP
          JSR  IncPtr
          LDA  R4_H
          SAP
          JSR  IncPtr
          LDA  R5_L
          SAP
          JSR  IncPtr
          LDA  R5_H
          SAP
          RTS


fl_load:
          ; "F@"  load float
          PHL
          JSR  rom_pop_data_R4
          LD   PTR_L,R4_L
          LD   PTR_H,R4_H
          LAP
          STA  R4_L
          JSR  IncPtr
          LAP
          STA  R4_H
          JSR  IncPtr
          LAP
          STA  R5_L
          JSR  IncPtr
          LAP
          STA  R5_H
          JSR  rom_push_data_R4
          JSR  rom_push_data_R5
          RTS


fl_lesszero:
          ; "F0<"  return true if float is negative
          PHL
          JSR  rom_pop_data_R5
          JSR  rom_pop_data_R4
          ROL  R5_L
          JNF  retFalse
          JMP  retTrue


fl_iszero:
          ; "F0="  return true if float is zero
          PHL
          JSR  rom_pop_data_R5
          JSR  rom_pop_data_R4
          TST  R4_L
          JNF  retFalse
          TST  R4_H
          JNF  retFalse
          TST  R5_L
          JNF  retFalse
          JMP  retTrue


fl_isless:
          ; "F<"  return true if float1 is less then float2
          PHL
          JSR  stack_to_F2
          JSR  stack_to_F1
          JSR  float_sub
          ROL  F1M+2
          JNF  retFalse
          JMP  retTrue


minmax:
          ;common code for FMIN and FMAX
          PHL
          JSR  fl_fover
          JSR  fl_fover
          JSR  stack_to_F2
          JSR  stack_to_F1
          JSR  float_sub
          ROL  F1M+2
          RTS


fl_fmin:
          ; "FMIN"  return the lesser float
          PHL
          JSR  minmax
          JPF  _flfmi1
_flfmi2   JSR  fl_fswap
_flfmi1   JSR  fl_fdrop
          RTS


fl_fmax:
          ; "FMAX"  return the greater float
          PHL
          JSR  minmax
          JPF  _flfmi2
          JSR  fl_fdrop
          RTS


fl_fnegate:
          ; "FNEGATE"  negate float
          PHL
_fneg_rts JSR  stack_to_F1
          JSR  negate_F1
          JMP  F1_to_stack_ret
          

fl_fdepth:
          ; "FDEPTH"  return possible float numbers on the data stack
          PHL
          JSR  c_depth
          JSR  c_2slash
          RTS


fl_floatpl:
          ; "FLOAT+"  add size of float to address
          PHL
          JSR  rom_pop_data_R4
          LDA  #4
          AD   R4_L
          STA  R4_L
          JNF  pushR4ret
          INC  R4_H
          JMP  pushR4ret


fl_floats:
          ; "FLOATS"  return memory size for n floats
          PHL
          JSR  rom_pop_data_R4
          CLC
          RWL  R4
          RWL  R4
pushR4ret:
          JSR  rom_push_data_R4
          RTS


fl_ffield:
          ;Implementation of the Forth word FFIELD:
          PHL
          LD   R4_L,#4
          LD   R4_H,#0
          JSR  rom_push_data_R4
          JSR  c_create
          JSR  c_over
          JSR  c_comma
          JSR  c_sum16
          JSR  c_doesgt
          JSR  c_at
          JSR  c_sum16
          RTS


fround_prep:
          PHL
          JSR  stack_to_F1
          LDA  F1M+2
          STA  R7_H
          ROL
          RTS


fl_ftrunc:
          ; "FTRUNC"  round towards zero
          PHL
          JSR  fround_prep
          JNF  _flfl01  ;re-use code in fl_floor
          ;negative number, make it positive
          JSR  negate_F1
          JMP  _flfl01  ;re-use code in fl_floor


fl_fround:
          ; "FROUND"  round to nearest integer value
          PHL
          JSR  fround_prep
          JNF  _flfr01
          ;negative number, make it positive
          JSR  negate_F1
_flfr01   ;add 0.5 to the number
          LD   F2E,#0xFF
          LD   F2M+2,#0x00
          LD   F2M+1,#0x00
          LD   F2M+0,#0x01
          JMP  _flfl04 ;re-use code in fl_floor


fl_floor:
          ; "FLOOR"  floor toward negative value
          PHL
          JSR  fround_prep
          JNF  _flfl01
          ;negative number, make it positive
          JSR  negate_F1
          ;add 0.9999999 to it
          LD   F2E,#0xE9
          LD   F2M+2,#0x7F
          LD   F2M+1,#0xFF
          LD   F2M+0,#0xFF
_flfl04   JSR  float_add
_flfl01   ;test if exponent is negative
          LDA  F1E
          ROL
          JPF  _flfl05
          ;shift F1 left until exponent is zero
          TST  F1E
          JPF  _flfl02
          RWL  F1M+0
          ROL  F1M+2
          DEC  F1E
          JMP  _flfl01
_flfl05   ;shift F1 right until exponent is zero
          CLC
          ROR  F1M+2
          ROR  F1M+1
          ROR  F1M+0
          INC  F1E
          JMP  _flfl01
_flfl02   ;negate number if it was originally negativ
          ROL  R7_H
          JNF  _flfl03
          JSR  negate_F1
_flfl03   ;return
          JMP  F1_to_stack_ret


fl_fabs:
          ; "FABS"  return absolut value of float
          PHL
          JSR  stack_to_F1
          LDA  F1M+2
          ROL
          JNF  F1_to_stack_ret
          JSR  negate_F1
          JMP  F1_to_stack_ret


fl_precision:
          ; "PRECISION"  get actually set precision for number printing
          PHL
          JSR  load_precision
          STA  R4_L
          LD   R4_H,#0
          JSR  rom_push_data_R4
          RTS


fl_set_prec:
          ; "SET-PRECISION"  set precision for number printing
          PHL
          JSR  rom_pop_data_R4
          LD   PTR_L,#<sto_prec
          LD   PTR_H,#>sto_prec
          LDA  #max_prec
          SU   R4_L
          LDA  R4_L
          JPF  _flsp01
          LDA  #max_prec
_flsp01   TST
          JPF  _RTS
          SAP
          RTS



;------------------------------------------------------------------------------
; Arithmetic functions:  mul / div / add / sub
;------------------------------------------------------------------------------


muldiv_prepare:
          ;make numbers positive, compute final sign (R1.7)
          PHL
          LDA  F1M+2
          XOR  F2M+2
          STA  R1
          LDA  F1M+2
          ROL
          JNF  _mdp01
          JSR  negate_F1
_mdp01    LDA  F2M+2
          ROL
          JNF  _RTS
          JSR  negate_F2
          RTS


float_mul:
          ;multiply F1 with F2, place result in F1
          PHL
          JSR  muldiv_prepare
          ;Init: accu (F3) := 0
          LDA  #0
          STA  F3M+0
          STA  F3M+1
          STA  F3M+2
          ;Add exponents, save in F3E
          LDA  F1E
          AD   F2E
          STA  F3E
          ;start in step 4
          CLC
          JMP  _flm04
_flm01    ;1) add factor B (F2) to accu (F3)
          JSR  _flm20
_flm02    ;2) shift factor B left
          CLC
          RWL  F2M+0
          ROL  F2M+2
_flm03    ;3) if F2M+2.7 is set, continue in step 12
          LDA  F2M+2
          ROL
          JPF  _flm12
_flm04    ;4) shift factor A (F1) right
          ROR  F1M+2
          ROR  F1M+1
          ROR  F1M+0
_flm05    ;5) if carry is set, continue in step 1
          JPF  _flm01
_flm06    ;6) if factor A still nonzero, go to step 2
          TST  F1M+0
          JNF  _flm02
          TST  F1M+1
          JNF  _flm02
          TST  F1M+2
          JNF  _flm02
_flm07    ;7) if accu bit 23 (F3M+2.7) is set, shift accu right and increment the exponent
          LDA  F3M+2
          ROL
          JNF  _flm08
          CLC
          ROR  F3M+2
          ROR  F3M+1
          ROR  F3M+0
          INC  F3E
_flm08    ;8) finished, the result is in F3M
          JMP  _flm16
_flm09    ;9) add factor B (F2) to accu (F3) with carry output
          JSR  _flm20
_flm10    ;10) shift accu right, with carry input
          ROR  F3M+2
          ROR  F3M+1
          ROR  F3M+0
_flm11    ;11) increment the exponent of F3
          INC  F3E
          LDA  F3E
          CMP  #0x80
          JPF  overflow_error
_flm12    ;12) shift factor A (F1) right
          CLC
          ROR  F1M+2
          ROR  F1M+1
          ROR  F1M+0
_flm13    ;13) if carry is set, continue in step 9
          JPF  _flm09
_flm14    ;14) if factor A still nonzero, go to step 10 (with carry=0)
          TST  F1M+0
          JNF  _flm10
          TST  F1M+1
          JNF  _flm10
          TST  F1M+2
          JNF  _flm10
_flm15    ;15) if accu bit 23 (F3M+2.7) is set, go to step 10
          LDA  F3M+2
          ROL
          INC  FLAG
          JNF  _flm10
_flm16    ;16) finished, the result is in F3M
          ;    now finalize the number in F3M, set correct sign
          JSR  cp_F3_to_F1
_flm19    ROL  R1
          JNF  _flm21
          JSR  negate_F1
_flm21    ;if mantissa is zero, make exponent zero as well
          TST  F1M+0
          JNF  _flm22
          TST  F1M+1
          JNF  _flm22
          TST  F1M+2
          JNF  _flm22
          LD   F1E,#0
_flm22    ;if result value gets too small, make it zero
          LDA  F1E
          ROL
          JNF  _RTS
          LDA  #0xC1
          SU   F1E
          JNF  _RTS
          LDA  #0
          STA  F1M+0
          STA  F1M+1
          STA  F1M+2
          STA  F1E
          RTS
_flm20    ;add factor B (F2) to accu (F3)
          LDA  F2M+0
          AD   F3M+0
          STA  F3M+0
          LDA  F2M+1
          ADD  F3M+1
          STA  F3M+1
          LDA  F2M+2
          ADD  F3M+2
          STA  F3M+2
_RET      RET


float_div:
          ;divide F1 by F2, place result in F1
          PHL
          JSR  muldiv_prepare
          ;Subtract exponents, save in F1E
          LDA  F1E
          SU   F2E
          STA  F1E
          ;divide the mantissa F1M by F2M
          ;make sure F2M is not zero
          TST  F2M+0
          JNF  _fld09
          TST  F2M+1
          JNF  _fld09
          TST  F2M+2
          JPF  divzero_error
_fld09    ;test if F2M uses only 16 bit. if yes, use fast 32-bit division routine from ROM.
          LD   R4_L,F1M+0
          LD   R4_H,F1M+1
          LD   R5_L,F1M+2
          TST  F2M+2
          JNF  _fld01
          LD   R5_H,#0
          LD   R6_L,F2M+0
          LD   R6_H,F2M+1
          JSR  rom_div_u32u16
          LD   F1M+0,R4_L
          LD   F1M+1,R4_H
          LD   F1M+2,R5_L
          LD   R7_L,#0
          ;F1M is the quotient, R6_L,R6_H,R7_L is the reset
          JMP  _fld02
_fld01    ;do 24bit/24bit division
          ;dividend in R4/R5_L, divisor in F2M
          LDA  #0
          STA  R6_L
          STA  R6_H
          STA  R7_L
          LD   R0,#24
         ; CLC
_fld03    RWL  R4
          ROL  R5_L
          RWL  R6
          ROL  R7_L
          LDA  R6_L
          SU   F2M+0
          STA  PAR1
          LDA  R6_H
          SUB  F2M+1
          STA  PAR2
          LDA  R7_L
          SUB  F2M+2
          JNF  _fld06
          LD   R6_L,PAR1
          LD   R6_H,PAR2
          STA  R7_L
_fld06    RWL  F1M+0
          ROL  F1M+2
          JLP  _fld03
_fld02    ;F1M is the quotient, R6_L,R6_H,R7_L is the reset
          ;shift the reminder into the result
_fld05    ;break loop if quotient register is 'full'
          LDA  F1M+2
          ROL
          ROL
          JPF  _flm19  ;finished, invert result in F1 if required and return
          ;break loop if reminder is zero
          TST  R6_L
          JNF  _fld07
          TST  R6_H
          JNF  _fld07
          TST  R7_L
          JPF  _flm19  ;finished, invert result in F1 if required and return
_fld07    ;CLC
          RWL  R6
          ROL  R7_L
          LDA  R6_L
          SU   F2M+0
          STA  PAR1
          LDA  R6_H
          SUB  F2M+1
          STA  PAR2
          LDA  R7_L
          SUB  F2M+2
          JNF  _fld08
          LD   R6_L,PAR1
          LD   R6_H,PAR2
          STA  R7_L
_fld08    RWL  F1M+0
          ROL  F1M+2
          DEC  F1E
          JMP  _fld05


exp_equalize_F1F2:
          ;Shift numbers F1 and F2 so they have the same exponent
          LDA  F1E
          CMP  F2E
          JPF  _RET
          PHL
          ;load F1E and F2E to temp. registers and add offset 0x80
          LD   R2,F2E
          ROL  R2
          INC  FLAG
          ROR  R2
          LDA  F1E
          ROL
          INC  FLAG
          ROR
          ;compare exponents
          SU   R2
          JNF  _expe_f2   ; jump if F2E > F1E

_expe_f1  ;shift F1 left
          ;test if bits 22 and 23 are equal (the sign will not change when shifting)
          LD   R1,F1M+2
          LD   R2,#0
          RWL  R1   ; same like "RWL R1, RWL R2"
          LDA  #0
          ROL  R1
          ROL
          CMP  R2
          JNF  _expe02  ; overflow will happen, begin with shifting F2 right
          ;shift left
          CLC
          RWL  F1M+0
          ROL  F1M+2
          DEC  F1E
          LDA  F1E
          CMP  F2E
          JNF  _expe_f1
          RTS  ;finished
_expe02   ;shift F2 right
          LDA  F2M+2
          ROL
          ROR  F2M+2
          ROR  F2M+1
          ROR  F2M+0
          INC  F2E
          LDA  F2E
          CMP  F1E
          JNF  _expe02
          ;exponents are now equal
          RTS

_expe_f2  ;shift F2 left
          ;test if bits 22 and 23 are equal (the sign will not change when shifting)
          LD   R1,F2M+2
          LD   R2,#0
          RWL  R1   ; same like "RWL R1, RWL R2"
          LDA  #0
          ROL  R1
          ROL
          CMP  R2
          JNF  _expe03  ; overflow will happen, begin with shifting F1 right
          ;shift left
          CLC
          RWL  F2M+0
          ROL  F2M+2
          DEC  F2E
          LDA  F2E
          CMP  F1E
          JNF  _expe_f2
          RTS  ;finished
_expe03   ;shift F1 right
          LDA  F1M+2
          ROL
          ROR  F1M+2
          ROR  F1M+1
          ROR  F1M+0
          INC  F1E
          LDA  F1E
          CMP  F2E
          JNF  _expe03
          ;exponents are now equal
          RTS


float_add:
          ;add floating point numbers F1 and F2, place result in F1
          PHL
          JSR  exp_equalize_F1F2
          ;add mantissa
          LDA  F1M+0
          AD   F2M+0
          STA  F1M+0
          LDA  F1M+1
          ADD  F2M+1
          STA  F1M+1
          LDA  F1M+2
          ADD  F2M+2
          PSH  ACCU    ;delay storing of F1M+2
          ;calculate one extra bit for the mantissa (the 25th bit)
          ROR          ;load carry into accu.7
          XOR  F1M+2   ;add to sign of F1
          XOR  F2M+2   ;add to sign of F2
          POP  F1M+2   ;finally store F1M+2
_fla01    ;compare new sign bit with the msb of of the calculated mantissa
          STA  R0      ;save the calculated 25th bit (which is the new sign bit) to R0.7
          XOR  F1M+2
          ROL
          ;FLAG = 1 if the mantissa of 24 bit overflowed
          JNF  _RTS    ;all ok
          ;overflow, need to increase exponent and shift mantissa right
          INC  F1E
          LDA  F1E
          CMP  #0x80
          JPF  overflow_error
          LDA  R0
          ROL
          ROR  F1M+2
          ROR  F1M+1
          ROR  F1M+0
          RTS


float_sub:
          ;subtract floating point number F2 from F1, place result in F1
          PHL
          JSR  exp_equalize_F1F2
          ;sign-extend F1 and place extra bits 23-31 in R7_L
          LD   R7_L,#0
          LDA  F1M+2
          ROL
          JNF  _fls01
          DEC  R7_L
_fls01    ;sign-extend F2 and place extra bits 23-31 in R7_H
          LD   R7_H,#0
          LDA  F2M+2
          ROL
          JNF  _fls02
          DEC  R7_H
_fls02    ;subtract the 32-bit values
          LDA  F1M+0
          SU   F2M+0
          STA  F1M+0
          LDA  F1M+1
          SUB  F2M+1
          STA  F1M+1
          LDA  F1M+2
          SUB  F2M+2
          STA  F1M+2
          LDA  R7_L    ;here is much potential for speed optimization (like already done in the float_add function!)
          SUB  R7_H
          ;compare new sign bit with the msb of of the calculated 24-bit mantissa
          JMP  _fla01  ;re-use code from float_add



;------------------------------------------------------------------------------
; More complex mathematics functions
;------------------------------------------------------------------------------


fl_fsqrt: ;Calculate the square root of a floating point number with the well-known algorithm:
          ;  : 2fdup fover fover ;
          ;  : fnip fswap fdrop;
          ;  : fsqrt fdup begin 2fdup f/ fover f+ 2e f/ fswap fover f- fabs 0.002e f< until fnip ;
          PHL
          ;check if number is negative
          JSR  stack_to_F1
          LDA  F1M+2
          ROL
          JNF  _flfsq02
          ;error, result is "NaN", but we return zero here
          JSR  pushZero
          JSR  pushZero
          RTS
_flfsq02  ;fdup
          JSR  F1_to_stack
          JSR  F1_to_stack
_flfsq01  ;begin 2fdup f/ fover f+
          JSR  fl_fover
          JSR  fl_fover
          JSR  fl_fdiv
          JSR  fl_fover
          JSR  fl_fplus
          ;2e f/
          JSR  stack_to_F1
          DEC  F1E
          JSR  F1_to_stack
          ;fswap fover f- fabs
          JSR  fl_fswap
          JSR  fl_fover
          JSR  fl_fminus
          JSR  fl_fabs
          ;0.001953e f<
          JSR  stack_to_F1
          LD   F2M+0,#1
          LD   F2M+1,#0
          LD   F2M+2,#0
          LD   F2E,#0xF7 
          JSR  float_sub
          ;until
          ROL  F1M+2
          JNF  _flfsq01
          ;fnip
          JSR  fl_fswap
          JSR  fl_fdrop
          RTS


taylor123:
          ;Generic function for taylor series with exponents x^1, x^2, x^3, x^4, etc.
          ;Input:  PTR : pointer to coefficients
          ;        ECCNT : number of coefficients
          ;        EPM : plus/minus toggle flag:
          ;          0xFF : +,-,+,-  ; 0x00 : -,+,-,+
          ;          0xC0 : +,+,+,+  ; 0x40 : -,-,-,-
          PHL
          LD   ECPTR+0,PTR_L
          LD   ECPTR+1,PTR_H
          JSR  fl_fdup
          JSR  c_2tor
          PSH  ECCNT
          DEC  ECCNT
          JMP  _tay01


taylor135:
          ;Generic function for taylor series with exponents x^1, x^3, x^5, x^7, etc.
          ;Input:  PTR : pointer to coefficients
          ;        ECCNT : number of coefficients
          ;        EPM : plus/minus toggle flag:
          ;          0xFF : +,-,+,-  ; 0x00 : -,+,-,+
          ;          0xC0 : +,+,+,+  ; 0x40 : -,-,-,-
          PHL
          LD   ECPTR+0,PTR_L
          LD   ECPTR+1,PTR_H
          JSR  fl_fdup
          JSR  fl_fdup
          JSR  fl_fmul
_tay05    JSR  c_2tor
          PSH  ECCNT
          DEC  ECCNT
_tay01    JSR  fl_fdup
          JSR  c_2rat
          JSR  fl_fmul
          DEC  ECCNT
          TST  ECCNT
          JNF  _tay01
          POP  ECCNT
          JSR  fl_fdup
          JSR  c_2rfrom
          JSR  fl_fmul
          JSR  next_coeff
          JSR  fl_fmul
_tay02    JSR  fl_fswap
          JSR  next_coeff
          JSR  fl_fmul
          LDA  EPM
          ROL
          JNF  _tay03
          JSR  fl_fplus
          INC  EPM
          JMP  _tay04
_tay03    JSR  fl_fminus
          DEC  EPM
_tay04    TST  ECCNT
          JNF  _tay02
          JSR  fl_fplus
          RTS

next_coeff:
          PHL
          LD   PTR_L,ECPTR+0
          LD   PTR_H,ECPTR+1
          JSR  _nxco
          JSR  _nxco
          LD   ECPTR+0,PTR_L
          LD   ECPTR+1,PTR_H
          DEC  ECCNT
          RTS
_nxco:    PHL
_nxcr:    LAP
          STA  R4_L
          JSR  IncPtr
          LAP
          STA  R4_H
          JSR  IncPtr
          JSR  rom_push_data_R4
          RTS

half_pi:       ;constant PI
          LD   PTR_L,#<tab_pihalf
          LD   PTR_H,#>tab_pihalf
ldconst:  PHL
          JSR  _nxco
          JMP  _nxcr

const_095:     ;constant 0.95
          LD   PTR_L,#<tab_095
          LD   PTR_H,#>tab_095
          JMP  ldconst

const_1:       ;constant 1.0
          LD   PTR_L,#<tab_1
          LD   PTR_H,#>tab_1
          JMP  ldconst

const_ln10:    ;constant ln(10)
          LD   PTR_L,#<tab_ln10
          LD   PTR_H,#>tab_ln10
          JMP  ldconst


          ; The sinus function is implemented by using the taylor series
          ;  sin x = x - 1/(3!)*x^3 + 1/(5!)*x^5 - 1/(7!)*x^7 + ...
          ; Divisions are replaced by multiplications to achieve an increase in speed.
          ; Only the first half of the sine wave is calculated by the tailor series.
          ; Repetitive wave fragments are genereted by mirroring at pi/2 and the x and y axis.
fl_fcos:  LD   EMODE,#1 ;mode: cosinus
          JMP  _flfs03
fl_fsin:  LD   EMODE,#0 ;mode: sinus
_flfs03   PHL
          ;check the sign, make number positive
          LD   ESIGN,#0
          JSR  stack_to_F1
          LDA  F1M+2
          ROL
          JNF  _flfs01
          DEC  ESIGN
          JSR  negate_F1
_flfs01   JSR  F1_to_stack
          ;fdup half_pi f/ f>d
          JSR  F1_to_stack
          JSR  half_pi
          JSR  fl_fdiv
          JSR  stack_to_F1
          JSR  float_to_double
          JSR  F1_to_stack
          ;2dup, >ECNT
          LD   ECNT,F1M+0
          ;half_pi f* f-
          JSR  half_pi
          JSR  fl_fmul
          JSR  fl_fminus
          ;check for sinus/cosinus
          TST  EMODE
          JPF  _flfs04
          INC  ECNT
_flfs04   ;check ECNT. If ECNT is odd, do: half_pi fswap f-
          LDA  ECNT
          ROR
          JNF  _flfs02
          JSR  half_pi
          JSR  fl_fswap
          JSR  fl_fminus
_flfs02   ;now use the taylor series to calculate the sinus value between 0 and pi/2
          LD  PTR_L,#<tab_sinus
          LD  PTR_H,#>tab_sinus
          LD  ECCNT,#4
          LD  EPM,#0
          JSR  taylor135
          ;Check bit 1 of ECNT. If it is set, negate the result.
          ;For sinus, combine it with ESIGN, which also forces negation if set.
          LDA  ECNT
          ROR
          TST  EMODE
          JNF  _flfs05
          XOR  ESIGN
_flfs05   ;negate result if bit 0 is set
          ROR
          JPF  _fneg_rts
          RTS


fl_ftan:
          PHL
          JSR  fl_fdup
          JSR  fl_fsin
          JSR  fl_fswap
          JSR  fl_fcos
          JSR  fl_fdiv
          RTS


fl_fasin:
          ; The arcus sinus function is implemented by using the taylor series
          ;  arcsin x = x + (1/2)*(x^3)/3 + ((1*3)/(2*4))*(x^5)/5 + ((1*3*5)/(2*4*6))*(x^7)/7 + ...
          ; Divisions are replaced by multiplications to achieve an increase in speed.
          ; This function gets very inaccurate for input values > 0.9
          PHL
          JSR  fl_fdup
          LD   PTR_L,#<tab_asinus
          LD   PTR_H,#>tab_asinus
          LD   ECCNT,#7 ;10
          LD   EPM,#0xC0
          JSR  taylor135
          ;"correct" result for input values 0.95 to 1.0:
          JSR  fl_fswap
          JSR  const_095
          JSR  fl_fminus
          JSR  stack_to_F1
          LDA  F1M+2
          ROL
          JPF  _RTS
          JSR  F1_to_stack
          JSR  fl_fdup
          JSR  fl_fplus
          JSR  fl_fdup
          JSR  fl_fplus
          JSR  fl_fplus
          RTS


fl_facos:
          PHL
          JSR  half_pi
          JSR  fl_fswap
          JSR  fl_fasin
          JSR  fl_fminus
          RTS


fl_fatan:
          PHL
          JSR  fl_fdup
          JSR  fl_fdup
          JSR  fl_fmul
          JSR  const_1
          JSR  fl_fplus
          JSR  fl_fsqrt
          JSR  fl_fdiv
          JSR  fl_fasin
          RTS


fl_fexp:
          ; The exponentiation function is implemented with the taylor series
          ;   e^x = 1 + x + x^2/(2!) + x^3/(3!) + x^4/(4!) + ...
          ; Divisions are replaced by multiplications to achieve an increase in speed.
          PHL
          ;select precision:
          ;0 <= x < 1  : 6 coefficients
          ;1 <= x < 2  : 9 coefficients
          ;2 <= x < oo : 11 coefficients
          ;-oo <= x < 0: 11 coefficients
          JSR  stack_to_F1
          JSR  F1_to_stack
          JSR  float_to_double
          TST  F1M+0
          JPF  _fexp01
          DEC  F1M+0
          TST  F1M+0
          JPF  _fexp02
          LD   PTR_L,#<tab_exp
          LD   PTR_H,#>tab_exp
          LD   ECCNT,#11
          JMP  _fexp03
_fexp02   LD   PTR_L,#<(tab_exp+8)
          LD   PTR_H,#>(tab_exp+8)
          LD   ECCNT,#9
          JMP  _fexp03
_fexp01   LD   PTR_L,#<(tab_exp+20)
          LD   PTR_H,#>(tab_exp+20)
          LD   ECCNT,#6
_fexp03   ;calculate taylor series
          LD   EPM,#0xC0
          JSR  taylor123
          JSR  const_1
          JSR  fl_fplus
          RTS


fl_fln:
          ; The natural logarithm function is implemented with the taylor series
          ;  ln(x) = 2 * ( y + (1/3)*y^3 + (1/5)*y^5 + (1/7)*y^7 + ... )
          ;  with y = (x-1)/(x+1)
          PHL
          JSR  fl_fdup
          ;prepare input value for taylor series:
          ;y = (x-1)/(x+1)
          JSR  fl_fdup
          JSR  const_1
          JSR  fl_fminus
          JSR  fl_fswap
          JSR  const_1
          JSR  fl_fplus
          JSR  fl_fdiv
          ;check which precision is needed
          JSR  fl_fswap
          JSR  stack_to_F1
          INC  F1E
          JSR  float_to_double
          TST  F1M+1
          JNF  _flfn4
          TST  F1M+0
          JPF  _flfn4
          LDA  #9-1
          SU   F1M+0
          JNF  _flfn4
          LDA  #6-1
          SU   F1M+0
          JNF  _flfn3
          LDA  #4-1
          SU   F1M+0
          JNF  _flfn2
          ;calculate taylor series with 4 coefficients
          LD   PTR_L,#<(tab_ln+24)
          LD   PTR_H,#>(tab_ln+24)
          LD   ECCNT,#4
          JMP  _flfn1
_flfn2    ;calculate taylor series with 6 coefficients
          LD   PTR_L,#<(tab_ln+16)
          LD   PTR_H,#>(tab_ln+16)
          LD   ECCNT,#6
          JMP  _flfn1
_flfn3    ;calculate taylor series with 8 coefficients
          LD   PTR_L,#<(tab_ln+8)
          LD   PTR_H,#>(tab_ln+8)
          LD   ECCNT,#8
          JMP  _flfn1
_flfn4    ;calculate taylor series with 10 coefficients
          LD   PTR_L,#<tab_ln
          LD   PTR_H,#>tab_ln
          LD   ECCNT,#10
_flfn1    LD   EPM,#0xC0
          JSR  taylor135
          ;multiply result by 2
          JSR  stack_to_F1
          INC  F1E
          JMP  F1_to_stack_ret


fl_flog:
          PHL
          JSR  fl_fln
          JSR  const_ln10
          JSR  fl_fdiv
          RTS



;------------------------------------------------------------------------------
; Print functions
;------------------------------------------------------------------------------


load_precision:
          LD   PTR_L,#<sto_prec
          LD   PTR_H,#>sto_prec
          LAP
          RET


fl_fprint:
          ; "F." print a float
          PHL
          JSR  stack_to_F1
          JSR  float_print
          JSR  c_type
          JSR  pr_space
          RTS

pr_space  LDA  #0x20
          JMP  rom_print_char


fl_fsprint:
          ; "FS." print a float
          LD   R7_H,#0
          JMP  _flfe08


fl_feprint:
          ; "FE." print a float
          LD   R7_H,#1
_flfe08   PHL
          JSR  load_precision
          STA  SLEN
          LD   PTR_L,#<numbuf
          LD   PTR_H,#>numbuf
          LD   SPTR+0,PTR_L
          LD   SPTR+1,PTR_H
          PSH  R7_H
          JSR  represent
          POP  R7_H
          JSR  rom_pop_data_R4  ; get flag2 (overflow flag)
          LD   R3,R4_L
          JSR  rom_pop_data_R4  ; get flag1 (sign)
          LD   R2,R4_L
          JSR  rom_pop_data_R4  ; get exponent
          LD   R0,R4_L
          ;test for overflow error
          TST  R3
          JNF  _flfe01
          LD   PTR_L,#<etxt_eover
          LD   PTR_H,#>etxt_eover
          JSR  rom_print_str
          RTS
_flfe01   ;print the sign
          TST  R2
          JPF  _flfe02
          LDA  #'-'
          JSR  rom_print_char
_flfe02   ;print the number
          JSR  load_precision
          STA  R1
          LD   PTR_L,#<numbuf
          LD   PTR_H,#>numbuf
_flfe03   JSR  lapIncPtrDecR1
          JSR  rom_print_char
          DEC  R0
          TST  R1
          JPF  _flfe04
          TST  R7_H
          JPF  _flfe05
          LDA  R0
          CMP  #9
          JPF  _flfe05
          CMP  #6
          JPF  _flfe05
          CMP  #3
          JPF  _flfe05
          TST
          JPF  _flfe05
          CMP  #0xFD ;-3
          JPF  _flfe05
          CMP  #0xFA ;-6
          JNF  _flfe03
_flfe05   LDA  #'.'
          JSR  rom_print_char
          ;print rest of the number
_flfe06   JSR  lapIncPtrDecR1
          JSR  rom_print_char
          TST  R1
          JNF  _flfe06
_flfe04   ;end of number, print exponent
          LDA  #'E'
          JSR  rom_print_char
          LDA  R0
          ROL
          LDA  R0
          JNF  _flfe07
          LDA  #'-'
          JSR  rom_print_char
          LDA  #0
          SU   R0
_flfe07   OR   #0x30
          JSR  rom_print_char
          JSR  pr_space
          RTS


fl_represent:
          ; "REPRESENT"  ( r c-addr u -- n flag1 flag2 ) 
          ; convert float to string in exponential representation
          PHL
          JSR  rom_pop_data_R4 ; get u
          LD   SLEN,R4_L       ; max number of digits in the string to be generated
          JSR  rom_pop_data_R4 ; get c-addr
          LD   SPTR+0,R4_L     ; ptr to string buffer
          LD   SPTR+1,R4_H
          DB   OPCODE_LDAI  ; this skips the following PHL instruction
represent PHL
          JSR  stack_to_F1     ; get r
          JSR  _float_pr   ;float_print  -- call float_print, but without rounding
          JSR  rom_pop_data_R5 ; get length of printed number into R5_L
          JSR  rom_pop_data_R4 ; get ptr to number buffer
          ;R4 = buffer / begin of printed number
          ;R5_L = length of printed number
          ;test for overflow
          LD   PTR_L,R4_L
          LD   PTR_H,R4_H
          LAP
          CMP  #'O'
          JPF  _flre03      ;overflow
          ;get the sign
          LD   R1,R5_L
          JSR  getSign
          LD   R5_L,R1
          STA  SIGN         ;sign of the number
_flre23   LD   R4_L,PTR_L
          LD   R4_H,PTR_H
          ;skip leading zeros in the buffer
          LD   R1,R5_L
_flre05   TST  R1
          JPF  _flre14
          LAP
          CMP  #'.'
          JPF  _flre06
          CMP  #'0'
          JNF  _flre14
_flre06   JSR  IncPtrDecR1
          JMP  _flre05
_flre14   ;R1 = length, PTR = start of string (first digit)
          ;SLEN = max. number of digits to represent

          ;round the number 'up'
          LD   R7_L,PTR_L
          LD   R7_H,PTR_H
          ;walk to last digit (represent + 1)
          LD   R2,R1
          LD   R3,SLEN
          INC  R3
          ;R2 = remaining characters in the string
_flre15   TST  R2
          JPF  _flre16  ; string too short
          LAP
          JSR  IncPtr
          DEC  R2
          CMP  #'.'
          JPF  _flre15
          DEC  R3
          TST  R3
          JNF  _flre15
          JSR  DecPtr
_flre17   ;ptr points now to the last digit to represent+1, this digit is in the accu
          SU   #0x35
          JNF  _flre16  ; jump if no rounding is required
          ;add '1' to the number
          LD   R2,SLEN
_flre18   JSR  DecPtr
          TST  R2
          JPF  _flre19  ; jump if no more digits. need to prefix a '1' to the number
          DEC  R2
          LAP
          CMP  #'.'
          JNF  _flre20
          JSR  DecPtr
          LAP
_flre20   INC
          SAP
          CMP  #0x3A
          JNF  _flre16 ; finished with rounding
          LDA  #0x30
          SAP
          JMP  _flre18
_flre19   LAP
          CMP  #'.'
          JNF  _flre22
          JSR  DecPtr
_flre22   LDA  #0x31
          SAP
          JMP  _flre23  ;re-start with new number
_flre16   ;rounding finished
          LD   PTR_H,R7_H
          LD   PTR_L,R7_L

          ;copy SLEN digits to destination buffer, pad with zeros
_flre01   TST  SLEN
          JPF  _flre02
          TST  R1
          JPF  _flre07
          JSR  lapIncPtrDecR1
          CMP  #'.'
          JPF  _flre01
          PSH  PTR_L
          PSH  PTR_H
          LD   PTR_L,SPTR+0
          LD   PTR_H,SPTR+1
          SAP
          JSR  IncPtr
          LD   SPTR+0,PTR_L
          LD   SPTR+1,PTR_H
          DEC  SLEN
          POP  PTR_H
          POP  PTR_L
          JMP  _flre01
_flre07   ;pad buffer with zeros
          LD   PTR_L,SPTR+0
          LD   PTR_H,SPTR+1
_flre08   TST  SLEN
          JPF  _flre02
          LDA  #'0'
          SAP
          JSR  IncPtr
          DEC  SLEN
          JMP  _flre08
_flre02   ;calculate the exponent
          LD   F1E,#0  ;exponent
          ;fist skip a leading zero
          LD   PTR_L,R4_L
          LD   PTR_H,R4_H
          LD   R1,R5_L
          LAP
          CMP  #'0'
          JNF  _flre09
          JSR  IncPtrDecR1
_flre09   ;read all numbers before the dot, increase exponent
          TST  R1
          JPF  _flre10
          JSR  lapIncPtrDecR1
          CMP  #'.'
          JPF  _flre11
          INC  F1E
          JMP  _flre09
_flre11   ;dot found, continue behind the dot
          TST  F1E
          JNF  _flre10
_flre12   TST  R1
          JPF  _flre10
          JSR  lapIncPtrDecR1
          CMP  #'0'
          JNF  _flre10
          DEC  F1E
          JMP  _flre12
_flre10   ;push exponent
          LD   R4_L,F1E
          LD   R4_H,#0
          ROL  F1E
          JNF  _flre04
          DEC  R4_H
_flre04   JSR  rom_push_data_R4
          ;push flag1 (sign) and flag2
          LD   R4_L,SIGN
          LD   R4_H,SIGN
          JSR  rom_push_data_R4  ;sign
          JMP  retTrue   ;flag2 = true
_flre03   ;overflow
          JSR  pushZero
          JSR  pushZero
          JMP  retFalse  ;flag2 = false


_float_pr:
          ;same as float_print, but with max. precision
          PHL
          LDA  #max_prec
          JMP  _flpr30

float_print:
          ;print floating point number in F1
          ;output string address and length on data stack
          ;parameters:   F1 ( -- caddr u )
          PHL
          JSR  load_precision
_flpr30   STA  R1
          LD   ROUND,#0
          CMP  #max_prec
          JPF  _flpr00
          INC  R1
          INC  ROUND
_flpr00   DEC  R1 ;R1 = remaining digit count -1
          LD   PTR_L,#<(floatstr+10)
          LD   PTR_H,#>(floatstr+10)
          LDA  #0
          STA  R2  ;string length
          STA  F2M+0
          STA  F2M+1
          STA  F2M+2
          STA  F2E
          ;check the sign of the number, print sign and invert number if required
          LDA  F1M+2
          STA  SIGN  ; save the sign
          ROL
          JNF  _flpr01
          JSR  negate_F1
_flpr01   ;check exponent
          LDA  F1E
          ROL
          JPF  _flpr02 ;jump if exponent is negative
_flpr10   TST  F1E
          JPF  _flpr11
          ;the exponent is positive, try to shift mantissa left
          ;and decrease exponent until it gets zero
          DEC  F1E
          RWL  F1M+0
          ROL  F1M+2
          ROL  F2E
          JNF  _flpr10
          ;32-bit overflow
          LD   R7_L,PTR_L
          LD   R7_H,PTR_H
          LDA  #'O'
          JSR  emitcf
          LDA  #'V'
          JSR  emitcf
          LDA  #'R'
          JSR  emitcf
          JMP  _flpr18
_flpr11   ;test if number is <= 24 bit wide
          TST  F2E
          JPF  _flpr04
          ;print a 32-bit wide integer number and exit
          JSR  _pr_u32
_flpr18   LDA  #'.'
          JSR  emitcf
          JMP  _flpr23
_flpr02   ;shift mantissa right and increase exponent until exponent is zero
          ;this is to get the integer part of the (unsigned) number
          TST  F1E
          JPF  _flpr04
          INC  F1E
          ROR  F1M+2
          ROR  F1M+1
          ROR  F1M+0
          ROR  F2M+2
          ROR  F2M+1
          ROR  F2M+0
          JMP  _flpr02
_flpr04   ;F1M is now the integer part, and F2M is the fractional part
          ;print the integer part
          JSR  _pr_u32
          ;print the decimal dot
          LDA  #'.'
          JSR  emitcf
          ;check for early end
          LDA  R1
          ROL
          JNF  _flpr05
          TST  ROUND
          JPF  _flpr09
          ;print reminder in F2M until F2M is zero
_flpr05   TST  F2M+0
          JNF  _flpr06
          TST  F2M+1
          JNF  _flpr06
          TST  F2M+2
          JPF  _flpr09
_flpr06   ;multiply F2M by 10, let the result overflow into F1M+0
          ;F1M,F2M := F1M * 10
          ;copy F2M to F3M for later addition
          LD   F3M+0,F2M+0
          LD   F3M+1,F2M+1
          LD   F3M+2,F2M+2
          ;F1M,F2M := F2M*2*2
          LD   F1M+0,#0
          JSR  _flpr07
          JSR  _flpr07
          ;F1M,F2M := F1M,F2M + F3M
          LDA  F2M+0
          AD   F3M+0
          STA  F2M+0
          LDA  F2M+1
          ADD  F3M+1
          STA  F2M+1
          LDA  F2M+2
          ADD  F3M+2
          STA  F2M+2
          JNF  _flpr08
          INC  F1M+0
_flpr08   ;F1M,F2M := F1M,F2M * 2
          JSR  _flpr07
          ;print the number in F1M+0
          LDA  #0x30
          OR   F1M+0
          JSR  emitdf
          ;continue with next number
          LDA  R1
          ROL
          JNF  _flpr05
_flpr09   
          ;check if rounding is required/requested
          TST  ROUND
          JPF  _flpr03
          LDA  R1
          ROL
          JNF  _flpr03
          ;fetch last digit, remove it from string
          JSR  DecPtr
          LAP
          CMP  #'.'
          JPF  _flpr23 ;it's an integer number, can't round here
          DEC  R2
          PSH  PTR_L
          PSH  PTR_H
          LD   R1,R2  ;R1=string length
          ;check if number must be rounded up
          LDA  R1
          CMP  #max_prec
          LD   PAR2,#0x34
          JPF  _flpr29
          LD   PAR2,#0x35
_flpr29   LAP
          CMP  #'.'
          JPF  _flpr26
          SU   PAR2 ;#0x34
          JNF  _flpr26
          ;loop over the string, increment digit with carry
_flpr25   JSR  DecPtr
          DEC  R1
          LAP
          CMP  #'.'
          JPF  _flpr25
          INC
          SAP
          CMP  #0x3A
          JNF  _flpr26
          LDA  #0x30
          SAP
          TST  R1
          JNF  _flpr25
          ;prefix the string with a '1'
          JSR  DecPtr
          LDA  #0x31
          SAP
          LD   R7_L,PTR_L
          LD   R7_H,PTR_H
_flpr26   ;restore pointer
          POP  PTR_H
          POP  PTR_L

_flpr03   ;finished, remove trailing zeros  
          ;R7 = ptr to start of string, R2 = string length
          LDA  R7_L
          AD   R2
          STA  PTR_L
          LD   PTR_H,R7_H
          JNF  _flpr27
          INC  PTR_H
_flpr27   JSR  DecPtr
          LAP
          CMP  #0x30
          JNF  _flpr23
          DEC  R2
          JMP  _flpr27

_flpr23   ;add sign to the string if number is negative
          LD   PTR_L,R7_L
          LD   PTR_H,R7_H
_flpr28   ROL  SIGN
          JNF  _flpr24
          LDA  #'-'
          JSR  emitcr
_flpr24   ;finalize string and exit
          LD   R4_L,PTR_L
          LD   R4_H,PTR_H
          LD   R5_L,R2
          LD   R5_H,#0
          JSR  rom_push_data_R4
          JSR  rom_push_data_R5
          RTS

_flpr07   ;shift F2M left with overflow to F1M+0
          CLC
          RWL  F2M+0
          ROL  F2M+2
          ROL  F1M+0
          RET

emitdf:   ;emit a character to the number string buffer, increase buffer position
          DEC  R1
emitcf:   INC  R2
          SAP
          INC  PTR_L
          TST  PTR_L
          JNF  _RET
          INC  PTR_H
          RET

emitdr:   ;decrease buffer position, emit a character to the number string buffer
          DEC  R1
emitcr:   INC  R2
          TST  PTR_L
          DEC  PTR_L
          JNF  _emcr1
          DEC  PTR_H
_emcr1    SAP
          RET


_pr_u32   ;print unsigned 32-bit decimal number in F1M and F2E
          PHL
          ;check if number is zero
          TST  F1M+0
          JNF  _flpr19
          LDA  F1M+1
          OR   F1M+2
          OR   F2E
          TST
          JNF  _flpr19
          ;print zero, but do not decrease R1
          LDA  #'0'
          JSR  emitcr
          JMP  _flpr22
_flpr19   LD   R4_L,F1M+0
          LD   R4_H,F1M+1
          LD   R5_L,F1M+2
          LD   R5_H,F2E
_flpr20   LD   R6_L,#10
          LD   R6_H,#0
          PSH  R1
          PSH  R2
          JSR  rom_div_u32u16
          POP  R2
          POP  R1
          LDA  #0x30
          OR   R6_L
          JSR  emitdr  ; note: the number is printed "reversed"
          TST  R4_L
          JNF  _flpr20
          TST  R4_H
          JNF  _flpr20
          TST  R5_L
          JNF  _flpr20
          TST  R5_H
          JNF  _flpr20
_flpr22   ;remember start of string, reset pointer
          LD   R7_L,PTR_L
          LD   R7_H,PTR_H
          LD   PTR_L,#<(floatstr+10)
          LD   PTR_H,#>(floatstr+10)
          RTS



;------------------------------------------------------------------------------
; String-to-float conversion functions
;------------------------------------------------------------------------------


fl_gtfloat:
          ;convert string to float: ">FLOAT"  ( c-addr u -- r true | false ) 
          PHL
          JSR  rom_pop_data_R5
          JSR  rom_pop_data_R4
          LD   PTR_L,R4_L
          LD   PTR_H,R4_H
          LD   R1,R5_L
_fl2f01   TST  R1
          JPF  _fl2f02
          LAP
          CMP  #0x20
          JNF  _fl2f03
          JSR  IncPtrDecR1
          JMP  _fl2f01
_fl2f03   LD   SFLG,#1
          JSR  str_to_f
          JPF  retFalse
          JSR  rom_push_data_R4
          JSR  rom_push_data_R5
retTrue:  LD   R4_L,#0xFF
retTrueFalse:
          LD   R4_H,R4_L
          JSR  rom_push_data_R4
          RTS
pushZero: PHL
retFalse: LD   R4_L,#0
          JMP  retTrueFalse
_fl2f02   LD   R4_L,#0
          LD   R4_H,#0
          JSR  rom_push_data_R4
          JSR  rom_push_data_R4
          JMP  retTrue


string_to_float:
          ;Input: PTR = pointer to string, R1 = string length
          ;Return: FLAG=0 if number could be read, number in R4/R5, PTR points behind the number string
          ;        FLAG=1 if string points not to a decimal number, PTR is unchanged
          LD   SFLG,#0
str_to_f: PHL
          ;save pointer
          LD   SPTR+0,PTR_L
          LD   SPTR+1,PTR_H
          LD   SLEN,R1
          ;check base, it must be 10
          TST  SFLG
          JNF  _sttf22
          JSR  c_base
          JSR  c_at
          JSR  rom_pop_data_R4
          LDA  #10
          CMP  R4_L
          JNF  strtof_ret_err
          LD   PTR_L,SPTR+0
          LD   PTR_H,SPTR+1
_sttf22   ;test for dot
          LD   R3,#0
          LAP
          CMP  #'.'
          JPF  _sttf19
          ;test for the sign and get it
          JSR  getSign
          JPF  strtof_ret_err
          STA  R3
          TST  R1
          JPF  strtof_ret_err  ; error, early end of number
          LAP
          JSR  test_digit
          JNF  _sttf19
          CMP  #'.'
          JNF  strtof_ret_err  ; error, not a digit or dot
_sttf19   ;read all numbers until 'E', 'e', '+' or '-'
          ;count the numbers behind the dot in R7_L
          LDA  #0
          STA  F1M+0
          STA  F1M+1
          STA  F1M+2
          STA  F1E
          STA  R7_L
_sttf01   TST  R1
          JPF  strtof_ret_err  ; error, early end of number
          JSR  lapIncPtrDecR1
          STA  R0
          CMP  #'E'
          JPF  _sttf02
          CMP  #'e'
          JPF  _sttf02
          CMP  #'+'
          JPF  _sttf02
          CMP  #'-'
          JNF  _sttf20
          JSR  DecPtr
          INC  R1
          JMP  _sttf02
_sttf20   CMP  #'.'
          JNF  _sttf03
          ;dot found, drop it and set R7_L to 1
          TST  R7_L
          JNF  strtof_ret_err  ; error, 2nd dot found
          INC  R7_L
          JMP  _sttf01
_sttf03   JSR  test_digit
          JPF  strtof_ret_err  ; error, not a digit
          ;found digit (in R0)
          ;increas count of numbers after the dot
          TST  R7_L
          JPF  _sttf05
          INC  R7_L
_sttf05   ;convert digit to binary
          ;multiply M1 by 10:
          ;copy F1M to F2M
          JSR  cp_F1_to_F2
          ;F1 *= 4
          JSR  _sttf04
          JPF  strtof_ret_err  ; overflow error, treat number like an unknown word
          JSR  _sttf04
          JPF  strtof_ret_err  ; overflow error, treat number like an unknown word
          ;F1 += F2
          LDA  F1M+0
          AD   F2M+0
          STA  F1M+0
          LDA  F1M+1
          ADD  F2M+1
          STA  F1M+1
          LDA  F1M+2
          ADD  F2M+2
          STA  F1M+2
          LDA  F1E
          ADD  F2E
          STA  F1E
          ;F1 *= 2
          JSR  _sttf04
          JPF  strtof_ret_err  ; overflow error, treat number like an unknown word
          ;add digit to F1
          LDA  #0x0F
          AND  R0
          AD   F1M+0
          STA  F1M+0
          JNF  _sttf01
          INC  F1M+1
          TST  F1M+1
          JNF  _sttf01
          INC  F1M+2
          TST  F1M+2
          JNF  _sttf01
          INC  F1E
          TST  F1E
          JNF  _sttf01
          JMP  strtof_ret_err ;error
_sttf02   ;now read the exponent
          LD   R7_H,#0  ;exponent to base10
          TST  R1
          JPF  _sttf06 ;no exponent, or end of exponent
          JSR  getSign
          JNF  _sttf21
          LAP
          CMP  #0x20
          JNF  strtof_ret_err
          LDA  #0
_sttf21   STA  R0
_sttf07   ;multiply R7_H by 10
          TST  R1
          JPF  _sttf08 ;end of exponent
          LAP
          CMP  #0x20
          JPF  _sttf08 ;end of exponent
          LDA  R7_H
          CLC
          ROL  R7_H
          ROL  R7_H
          AD   R7_H
          ROL
          STA  R7_H
          JSR  lapIncPtrDecR1
          JSR  test_digit
          JPF  strtof_ret_err ;error
          AND  #0x0F
          AD   R7_H
          STA  R7_H
          JMP  _sttf07
_sttf08   TST  R0
          JPF  _sttf06
          LDA  #0xFF
          XOR  R7_H
          INC
          STA  R7_H
_sttf06   ;correct nbr of digits behind the dot
          TST  R7_L
          JPF  _sttf09
          DEC  R7_L
_sttf09   ;adjust the exponent
          LDA  R7_H
          SU   R7_L
          STA  R7_H
          ;finished parsing the string.
          ;update the string ponter:
          LD   SPTR+0,PTR_L
          LD   SPTR+1,PTR_H
          LD   SLEN,#0
          ;here we have:
          ; big integer number in F1M,F1E
          ; sign of the number in R3, the base10 exponent in R7_H
          ;shift mantissa right until right most bit is set to 1
          LD    R1,#0 ;new base2 exponent
_sttf17   TST   F1M+0
          JNF   _sttf16
          TST   F1M+1
          JNF   _sttf16
          TST   F1M+2
          JNF   _sttf16
          TST   F1E
          JPF   _sttf18
_sttf16   LDA   F1M+0
          ROR
          JPF   _sttf18
          INC   R1
          CLC
          ROR   F1E
          ROR   F1M+2
          ROR   F1M+1
          ROR   F1M+0
          JMP   _sttf16
_sttf18   TST   F1E
          JNF   strtof_ret_overflow
          LDA   F1M+2
          ROL
          JPF   strtof_ret_overflow
          LD    F1E,R1
          ;check if the exponent is zero
          TST  R7_H
          JPF  _sttf10  ;yes, we are done now
          ;check if exponent is positive
          LDA  R7_H
          ROL
          JPF  _sttf13
          ;the mantissa must be multiplied by 10^exponent
          JSR  _sttf14
          JSR  float_mul
          JMP  _sttf10
_sttf13   ;the exponent is negative
          ;the mantissa must be divided by 10*exponent
          LDA  #0
          SU   R7_H
          STA  R7_H
          JSR  _sttf14
          PSH  R3
          JSR  float_div
          POP  R3
_sttf10   ;negate mantissa if flag in R3 is set
          TST  R3
          JPF  _sttf11
          JSR  negate_F1
_sttf11   ;return the 32-bit floating point number in R4 and R5
          LD   R4_L,F1M+0
          LD   R4_H,F1M+1
          LD   R5_L,F1M+2
          LD   R5_H,F1E
          CLC
          LD   R3,#0xFF ; flag for a 32-bit number
          JMP  strtof_ret
_sttf14   ;check exponent for overflow
          LDA  #9
          SU   R7_H
          JNF  strtof_ret_overflow
          ;get value from table
          PHL
          LDA  R7_H
          DEC
          CLC
          ROL
          ROL
          LD   PTR_L,#<tab_10exp
          LD   PTR_H,#>tab_10exp
          AD   PTR_L
          STA  PTR_L
          JNF  _sttf12
          INC  PTR_H
_sttf12   JSR  lapIncPtrDecR1
          STA  F2E
          JSR  lapIncPtrDecR1
          STA  F2M+2
          JSR  lapIncPtrDecR1
          STA  F2M+1
          LAP
          STA  F2M+0
          RTS

_sttf04   ;shift F1M left
          CLC
          RWL  F1M+0 ;and F1M+1
          RWL  F1M+2 ;and F1E
          RET

strtof_ret_overflow:
          TST  SFLG
          JPF  precision_error

strtof_ret_err:
          SEC
strtof_ret:
          LD   PTR_L,SPTR+0
          LD   PTR_H,SPTR+1
          LD   R1,SLEN
          RTS



;------------------------------------------------------------------------------
; Helper functions
;------------------------------------------------------------------------------


stack_to_F1:
          ;Load float from datastack into F1
          PHL
          JSR  rom_pop_data_R5
          JSR  rom_pop_data_R4
          LD   F1M+0,R4_L
          LD   F1M+1,R4_H
          LD   F1M+2,R5_L
          LD   F1E,R5_H
          RTS

stack_to_F2:
          ;Load float from datastack into F2
          PHL
          JSR  rom_pop_data_R5
          JSR  rom_pop_data_R4
          LD   F2M+0,R4_L
          LD   F2M+1,R4_H
          LD   F2M+2,R5_L
          LD   F2E,R5_H
          RTS

F1_to_stack:
          ;Push floating point number F1 to datastack
          PHL
F1_to_stack_ret:
          LD   R4_L,F1M+0
          LD   R4_H,F1M+1
          LD   R5_L,F1M+2
          LD   R5_H,F1E
pushR4R5ret:
          JSR  rom_push_data_R4
pushR5ret:
          JSR  rom_push_data_R5
          RTS


cp_F1_to_F2:
          LD   F2M+0,F1M+0
          LD   F2M+1,F1M+1
          LD   F2M+2,F1M+2
          LD   F2E,F1E
          RET


cp_F3_to_F1:
          LD   F1M+0,F3M+0
          LD   F1M+1,F3M+1
          LD   F1M+2,F3M+2
          LD   F1E,F3E
          RET


negate_F1:
          LDA  #0
          SU   F1M+0
          STA  F1M+0
          LDA  #0
          SUB  F1M+1
          STA  F1M+1
          LDA  #0
          SUB  F1M+2
          STA  F1M+2
          RET


negate_F2:
          LDA  #0
          SU   F2M+0
          STA  F2M+0
          LDA  #0
          SUB  F2M+1
          STA  F2M+1
          LDA  #0
          SUB  F2M+2
          STA  F2M+2
          RET


getSign:  ;Get sign from string, drop the sign.
          ;Returns Flag=0, Accu = 0: positive number, Accu = 0xFF: negative number.
          ;Returns Flag=1: error
          LAP
          CMP  #'+'
          JPF  _gsgn1
          CMP  #'-'
          JPF  _gsgn2
          PHL
          JSR  test_digit
          JPF  _RTS
          LDA  #0
          RTS
_gsgn1    LDA  #0
          JMP  IncPtrDecR1
_gsgn2    LDA  #0xFF
          JMP  IncPtrDecR1


lapIncPtrDecR1:
          LAP
IncPtrDecR1:
          DEC  R1
IncPtr:   INC  PTR_L
          TST  PTR_L
          JNF  _RET
          INC  PTR_H
          CLC
          RET  ;return always with FLAG=0

DecPtr:   TST  PTR_L
          DEC  PTR_L
          JNF  _RET
          DEC  PTR_H
          RET


test_digit:
          ; Check if ACCU contains a valid digit ('0' - '9')
          ; In:  ACCU = ASCII character
          ; Out: Returns with FLAG=0 when ACCU contains a digit.
          PSH  ACCU
          SU   #'0'
          JNF  _tstdi1
          SUB  #10
          POP  ACCU
          RET
_tstdi1   SEC
          POP  ACCU
          RET


overflow_error:
          ;print overflow error message and abort program execution
          LD   PTR_L,#<etxt_overflow
          LD   PTR_H,#>etxt_overflow
_pr_errq  JSR  rom_print_str
          JMP  c_quit

precision_error:
          ;print precision error message and abort program execution
          LD   PTR_L,#<etxt_precision
          LD   PTR_H,#>etxt_precision
          JMP  _pr_errq

divzero_error:
          ;print division by zero error message and abort program execution
          LD   PTR_L,#<etxt_divzero
          LD   PTR_H,#>etxt_divzero
          JMP  _pr_errq


;------------------------------------------------------------------------------
END_MODULE
