;==============================================================================
; This module adds support for the Forth Deck to the My4TH-nfd ROM.
;
; Assemble this module with the myca command
;   $ myca m4-forthdeck.asm -o m4-forthdeck.bin
;
; Upload the binary module to My4TH with the my4th tool
; (for example. See "my4th --help" for details):
;   $ my4th write /dev/ttyS0 binary 60 m4-forthdeck.bin
;
; Configure your Forth Deck to auto-start this driver. In screen 0,
; write the number '1' at the beginning of the first line.
; Then edit screen 1, clear it completely and write this into line 1:
;   $2D1E execute 60 bload
;
;==============================================================================

#include <my4th/binmod.hsm>


;------------------------------------------------------------------------------
; Jumplist for incomming jumps
;------------------------------------------------------------------------------

CODEOFS   SET  0x30
          DB   0,0  ; reserved bytes, used for alignment

fdvect:   JMP  CODEOFS + lcd_init
          JMP  CODEOFS + lcd_clear
          JMP  CODEOFS + lcd_gotoxy
          JMP  CODEOFS + lcd_cursor_on
          JMP  CODEOFS + lcd_cursor_off
          JMP  CODEOFS + lcd_print
          JMP  CODEOFS + lcd_out_ch
          JMP  CODEOFS + lcd_out_fast
          JMP  CODEOFS + lcd_load_charset
          JMP  CODEOFS + keyb_readkey
          JMP  CODEOFS + keyb_inputstring
          JMP  0x0E2F                ; magic number
          JMP  CODEOFS + init_module ; initialization vector for load-image



;------------------------------------------------------------------------------
; Exported Words
;------------------------------------------------------------------------------

EXPORTS:  ;List of exported words.
          EXPORT  w_lcd,      "lcd"
          EXPORT  w_terminal, "terminal"
          EXPORT  w_keyq,     "key?"
          EXPORT  w_edit,     "edit"
          EXPORT  w_list,     "list"
          EEND



;------------------------------------------------------------------------------
; Imported Words
;------------------------------------------------------------------------------

IMPORTS:  ;List of imported words.
c_list    IMPORT  "list"
c_edit    IMPORT  "edit"
c_key     IMPORT  "key"
c_dup     IMPORT  "dup"
c_block   IMPORT  "block"
c_update  IMPORT  "update"
c_savebuf IMPORT  "save-buffers"
c_emptybu IMPORT  "empty-buffers"
c_fill    IMPORT  "fill"
          IEND



;------------------------------------------------------------------------------
; Definitions
;------------------------------------------------------------------------------

UART_TXD_DEFAULT  SET 0x01   ; TXD is high by default

LCDENA1   SET  0x12|UART_TXD_DEFAULT  ;bit0 = UART, bit1 (sda) = 1, bit2 (scl) = 0, bit3 (E) = 0, bit4/5 = lcd enable
LCDENA2   SET  0x22|UART_TXD_DEFAULT  ;bit0 = UART, bit1 (sda) = 1, bit2 (scl) = 0, bit3 (E) = 0, bit4/5 = lcd enable
LCDDTMD   SET  0x06|UART_TXD_DEFAULT  ;bit0 = UART, bit1 (sda) = 1, bit2 (scl) = 1, bit3 (E) = 0, bit4/5 = don't care
LCDDTMR   SET  0x02|UART_TXD_DEFAULT  ;bit0 = UART, bit1 (sda) = 1, bit2 (scl) = 0, bit3 (E) = 0, bit4/5 = don't care

CRSR_ON_CMD  SET 0x0F  ;0x0E = underline, 0x0F = blinking cursor

;zeropage variables
LIB_BUF       SET  0x20
LIB_BIGKEYB   SET  0x34
LIB_KEYREPC   SET  0x35
LIB_DISPMAXX  SET  0x37
LIB_DISPMAXY  SET  0x38
LIB_LCDMAXX   SET  0x39
LIB_LCDOUT    SET  0x3A
LIB_LCDENAS   SET  0x3B
LIB_LCDCRSR   SET  0x3C
LIB_KBNOKEYB  SET  0x3D
LIB_LCDPOSX   SET  0x3E
LIB_LCDPOSY   SET  0x3F
LIB_LCDBUFP   SET  0x40
LIB_NUMPAGES  SET  0x4A
NEXTINCHR     SET  0x8C
RANDVAR       SET  0xA4

LCD_BUFFER    SET  0x8200



;------------------------------------------------------------------------------
; Data Section
;------------------------------------------------------------------------------

text_hello1   DB "[Forth_Deck] ROM_v",0
text_hello2   DB " ",0
text_hello3   DB "_MHz ",0
text_hello4   DB "_blocks\n",0
text_errfd    DB "no Forth Deck\n",0
text_errrom   DB "wrong ROM version\n",0
text_errload  DB "load error\n",0
text_saving2  DB "saving...",0
text_loading  DB "\nloading screen...",0
text_ed_head  DB "[save] [save+quit] [quit] [clear]",0
text_useterm  DB "Please use the RS232 terminal",0
NUM_BUF       DB 0,0,0,0,0,0,0

tab_lcd_chr DB  0x00,0x00,0x09,0x16,0x00,0x00,0x00,0x00  ;[0]:~
            DB  0x00,0x10,0x08,0x04,0x02,0x01,0x00,0x00  ;[1]:\

ed2_chars   DB  0x00,0x17,0x15,0x15,0x15,0x15,0x17,0x00  ;[2]:10
            DB  0x00,0x04,0x04,0x08,0x10,0x08,0x04,0x04  ;[3]:<
            DB  0x00,0x04,0x04,0x02,0x01,0x02,0x04,0x04  ;[4]:>

KCODE_BS      SET 0x08  ;back space
KCODE_DEL     SET 0x7F
KCODE_PGUP    SET 0x0E
KCODE_PGDOWN  SET 0x0F
KCODE_F1      SET 0x10
KCODE_F2      SET 0x11
KCODE_F3      SET 0x12
KCODE_F4      SET 0x13
KCODE_F5      SET 0x14
KCODE_F6      SET 0x15
KCODE_F7      SET 0x16
KCODE_F8      SET 0x17
KCODE_INSERT  SET 0x18
KCODE_HOME    SET 0x19
KCODE_END     SET 0x1A
KCODE_UP      SET 0x1C
KCODE_DOWN    SET 0x1D
KCODE_RIGHT   SET 0x1E
KCODE_LEFT    SET 0x1F

TAB_KEYNORM
    DB  KCODE_DOWN,KCODE_UP,KCODE_RIGHT,0x0D,0x5C,KCODE_BS, 0,0  ; M4
    DB  0x5D,0x3D,KCODE_LEFT,0x27,0x5B,0x2D, 0,0  ; M3
    DB  0x2E,0x6C,0x2F,0x3B,0x70,0x30, 0,0  ; M2
    DB  0x6F,0x39,0x2C,0x6B,0x69,0x38, 0,0  ; M1
    DB  0x6E,0x68,0x6D,0x6A,0x75,0x37, 0,0  ; M0
    DB  0x79,0x36,0x62,0x67,0x74,0x35, 0,0  ; OUT7
    DB  0x63,0x64,0x76,0x66,0x72,0x34, 0,0  ; OUT6
    DB  0x65,0x33,0x78,0x73,0x77,0x32, 0,0  ; OUT5
    DB  0x20,0x1B,0x7A,0x61,0x71,0x31, 0,0  ; OUT4

TAB_KEYSHIFT
    DB  KCODE_DOWN,KCODE_UP,KCODE_RIGHT,0x0D,0x7C,KCODE_DEL, 0,0  ; M4
    DB  0x7D,0x2B,KCODE_LEFT,0x22,0x7B,0x5F, 0,0  ; M3
    DB  0x3E,0x4C,0x3F,0x3A,0x50,0x29, 0,0  ; M2
    DB  0x4F,0x28,0x3C,0x4B,0x49,0x2A, 0,0  ; M1
    DB  0x4E,0x48,0x4D,0x4A,0x55,0x26, 0,0  ; M0
    DB  0x59,0x5E,0x42,0x47,0x54,0x25, 0,0  ; OUT7
    DB  0x43,0x44,0x56,0x46,0x52,0x24, 0,0  ; OUT6
    DB  0x45,0x23,0x58,0x53,0x57,0x40, 0,0  ; OUT5
    DB  0x20,0x7E,0x5A,0x41,0x51,0x21, 0,0  ; OUT4



;------------------------------------------------------------------------------
; Code Section. This section must be placed last, after all other sections.
;------------------------------------------------------------------------------

CODE:

init_module:
          ;check if our environment is ok
          PHL
          LD   PTR_L,#<(rom_lcd_init+2)
          LD   PTR_H,#>(rom_lcd_init+2)
          LAP
          ROL
_inim01   LD   PTR_L,#<text_errrom
          LD   PTR_H,#>text_errrom
          JNF  _inim04
          LD   PTR_L,#<fdvect
          LD   PTR_H,#>fdvect
          LDA  #0x40
          CMP  PTR_L
          JNF  _inim02
          LDA  #0x84
          CMP  PTR_H
          JPF  _inim03
_inim02   LD   PTR_L,#<text_errload
          LD   PTR_H,#>text_errload
_inim04   JSR  rom_print_str
          RTS
_inim03   ;initialize the keyboard
          JSR  keyboard_init
          LD   PTR_L,#<text_errfd
          LD   PTR_H,#>text_errfd
          TST  LIB_BIGKEYB
          JPF  _inim04
          ;initialize the LCD
          LD   R7_L,#0
          JSR  lcd_init
          LDA  #0x9C
          CMP  R7_L
          JNF  _inim01
_inim05   ;get ROM version and CPU speed
          LD   PTR_L,#0xFE
          LD   PTR_H,#0x7D
          JSR  lap_inc_ptr
          STA  R7_L  ; CPU speed
          LAP
          AD   #0x11
          STA  R7_H  ; ROM version
          ;decode the speed, set best typematic rate for the speed
          LDA  R7_L
          LD   R1,#70
          LD   R4_L,#14
          LD   R4_H,#2
          CMP  #14
          JPF  _inim06
          LD   R1,#60
          LD   R4_L,#195
          LD   R4_H,#1
          CMP  #12
          JPF  _inim06
          LD   R1,#50
          LD   R4_L,#120
          LD   R4_H,#1
_inim06   LD   PTR_L,#<(_rdbk10+2)
          LD   PTR_H,#>(_rdbk10+2)
          LDA  R1
          SAP
          LD   PTR_L,#<(_rdbk05+2)
          LD   PTR_H,#>(_rdbk05+2)
          LDA  R4_L
          SAP
          LD   PTR_L,#<(_rdbk05+5)
          LD   PTR_H,#>(_rdbk05+5)
          LDA  R4_H
          SAP
          ;print hello message
          LD   PTR_L,#<text_hello1
          LD   PTR_H,#>text_hello1
          JSR  rom_print_str
          ;print version
          LDA  R7_H
          CLC
          ROR
          CLC
          ROR
          CLC
          ROR
          CLC
          ROR
          STA  R4_L
          LD   R4_H,#0
          JSR  rom_print_decword
          LDA  #'.'
          JSR  rom_print_char
          LDA  #0x0F
          AND  R7_H
          STA  R4_L
          LD   R4_H,#0
          JSR  rom_print_decword
          LD   PTR_L,#<text_hello2
          LD   PTR_H,#>text_hello2
          JSR  rom_print_str
          ;print speed
          LD   R4_L,R7_L
          LD   R4_H,#0
          JSR  rom_print_decword
          LD   PTR_L,#<text_hello3
          LD   PTR_H,#>text_hello3
          JSR  rom_print_str
          ;print blocks
          LD   R4_L,LIB_NUMPAGES+0
          LD   R4_H,LIB_NUMPAGES+1
          JSR  rom_print_decword
          LD   PTR_L,#<text_hello4
          LD   PTR_H,#>text_hello4
          JSR  rom_print_str
          ;ready
rts_opc   RTS



;------------------------------------------------------------------------------
; Library Functions
;------------------------------------------------------------------------------

lcdWait2ms:
          LDA  #2
delay_ms: ;wait some milliseconds
          LD   R0,#70  ;calibrated for 14 MHz CPU clock
_dlms01   JLP  _dlms01
          CLC
          DEC
          TST
          JNF  delay_ms
          RET

sap_inc_ptr:
          SAP
          JMP  inc_ptr

lap_inc_ptr:
          LAP
inc_ptr:  INC  PTR_L
          TST  PTR_L
          JNF  ret_opc
          INC  PTR_H
ret_opc:  RET

dec_ptr:  TST  PTR_L
          JNF  _decpt1
          DEC  PTR_H
_decpt1   DEC  PTR_L
          RET

ld_ptr_r4:
          LD   PTR_L,R4_L
          LD   PTR_H,R4_H
          RET

print_nl:
          LDA  #10
          JMP  rom_print_char

return_flag_1:
          SEC
          RTS

is_alpha:
          PSH  ACCU
          SU   #'A'
          JNF  _isalp01
          SUB  #'Z'+1-'A'
          JNF  _isalp02
          SUB  #'a'-('Z'+1)
          JNF  _isalp01
          SUB  #'z'+1-'a'
_isalp02  POP  ACCU
          RET
_isalp01  SEC
          POP  ACCU
          RET



;------------------------------------------------------------------------------
; LCD Functions
;------------------------------------------------------------------------------

lcd_init:
          ; Initialize LCD display.
          PHL
          JSR  _lcdinit1
          JSR  _lcdinit1  ; call init1 two times, sometimes it is needed
          LD   LIB_LCDOUT,#1
          LD   R7_L,#0x9C ;flag
          RTS

_lcdinit1: ;initialize both chips of the LCD
          PHL
          LD   LIB_LCDENAS,#LCDENA1  ; start with first half of the LCD
          JSR  _lcdinit2
          LD   LIB_LCDENAS,#LCDENA2  ; then initialize the second half of the LCD
          JSR  _lcdinit2
          LD   LIB_LCDCRSR,#1
          LD   LIB_LCDMAXX,#40
          LD   LIB_DISPMAXX,LIB_LCDMAXX
          LD   LIB_DISPMAXY,#4
          JSR  lcd_clear
          ;finished
          RTS

_lcdinit2: ;initialize the currently selected LCD (E1/E2)
          PHL
          ;configure interface (set to 4-bit)
          JSR  lcdSetCmdReg
          OUT  OUTP
          LDA  #0
          JSR  lcdSendByte
          LDA  #3
          JSR  lcdSendByte
          JSR  lcdWait2ms
          LDA  #3
          JSR  lcdSendByte
          JSR  lcdWait2ms
          LDA  #3
          JSR  lcdSendByte
          JSR  lcdWait2ms
          LDA  #2
          JSR  lcdSendByte
          JSR  lcdWait2ms
          LDA  #0x28        ;4-bit mode
          JSR  lcdSendByte
          LDA  #0x0C        ;switch display on (and cursor off)
          JSR  lcdSendByte
          LDA  #0x01        ;clear the LCD
          JSR  lcdSendByte
          JSR  lcdWait2ms
          LDA  #0x06        ;entry mode: increment, no shift
          JSR  lcdSendByte
          JSR  lcdWait2ms
          LD   R0,#0
          LD   R1,#2
          LD   PTR_L,#<tab_lcd_chr
          LD   PTR_H,#>tab_lcd_chr
          JSR  lcd_load_charset
          RTS


lcdSetDataReg:
          LDA  LIB_LCDENAS
          OUT
          OR   #0x04 ; SCL high, clock in the E-bits
          OUT
          LD   OUTP,#LCDDTMD
          RET


lcdSetCmdReg:
          LDA  LIB_LCDENAS
          OUT
          OR   #0x04 ; SCL high, clock in the E-bits
          OUT
          LD   OUTP,#LCDDTMR
          RET


lcdSendNibble:
          ; Send a data nibble to the LCD
          ; Input: ACCU = data nibble in bits 4-7
          OR   OUTP
          OUT  ACCU
          STA  PAR2
          LD   PAR1,#0x08
          OR   PAR1
          OUT  ACCU   ; E=1
          OUT  PAR2   ; E=0
          RET


lcdSendByte:
          ; Send a data byte to the LCD in 4-bit mode
          ; Input: ACCU = data byte
          ; Changes: ACCU, R1
          STA  R1
lcdSendR1:
          LDA  #0xF0
          AND  R1
          OR   OUTP
          OUT  ACCU
          STA  PAR2
          LD   PAR1,#0x08
          OR   PAR1
          OUT  ACCU   ; E=1
          OUT  PAR2   ; E=0
          LDA  R1
          CLC
          ROL
          CLC
          ROL
          CLC
          ROL
          CLC
          ROL
          OR   OUTP
          OUT  ACCU
          STA  PAR2
          OR   PAR1
          OUT  ACCU   ; E=1
          OUT  PAR2   ; E=0
          RET


lcd_clear:
          ; Clear the LCD display.
          ; The cursor will be set to upper left edge.
          ; Changes: ACCU, R0, R1
          PHL
          ;send clear-display-command
          LDA  #0x01
          JSR  sendCmdToAllLCD
          JSR  lcdWait2ms
          LD   R0,#4*40
          LD   PTR_L,#<LCD_BUFFER
          LDA  #0x20 ;OP-code order changed to prevent loader from changing the lcd buffer address
          LD   PTR_H,#>LCD_BUFFER
_lcdcl02  SAP
          INC  PTR_L
          TST  PTR_L
          JNF  _lcdcl01
          INC  PTR_H
_lcdcl01  JLP  _lcdcl02
          ;reset cursor position
          LD   LIB_LCDPOSX,#0
          LD   LIB_LCDPOSY,#0
          JSR  lcd_rst_crsr
          RTS


lcd_cursor_on:
          ; Switch cursor on
          ; Changes: ACCU, R0, R1
          LD   LIB_LCDCRSR,#1
          JMP  lcd_rst_crsr


lcd_cursor_off:
          ; Switch cursor off
          ; Changes: ACCU, R0, R1
          LD   LIB_LCDCRSR,#0

sendCrsrOffCmd:
          LDA  #0x0C

sendCmdToAllLCD:
          ; Send command in ACCU to all available LCD controller chips
          PHL
          PSH
          LD   LIB_LCDENAS,#LCDENA1
          JSR  lcdSetCmdReg
          POP
          PSH
          JSR  lcdSendByte
          LD   LIB_LCDENAS,#LCDENA2
          JSR  lcdSetCmdReg
          POP
          JSR  lcdSendByte
          RTS


lcd_shift:
          ; Shift LCD one line up, clear last line
          PHL
          PSH  R0
          PSH  R1
          PSH  R3
          PSH  R4
          PSH  R5
          PSH  PTR_L
          PSH  PTR_H
          ;print out all lines, shift buffer one line up
          LD   R3,#0  ; line counter (destination line)
          LD   R4,#<(LCD_BUFFER+0)  ; destination address in memory
          LD   R5,#<(LCD_BUFFER+40) ; source address in memory
_lcdsft7  ;setup first/next line to output
          LD   R0,#0
          LD   R1,R3
          JSR  lcd_gotoxy
          ;copy/output one row
          JSR  lcdSetDataReg
          LD   R0,LIB_LCDMAXX ; row counter, counts down to zero
_lcdsft6  LD   PTR_L,R5
          LD   PTR_H,#>LCD_BUFFER
          LAP
          LD   PTR_L,R4
          SAP
          JSR  lcdSendByte
          INC  R4
          INC  R5
          JLP  _lcdsft6
_lcdsft8  ;copy next line
          INC  R3
          LDA  #3
          CMP  R3
          JNF  _lcdsft7
          ;clear last line
          LD   PTR_L,R4
          LD   R0,#0
          LD   R1,#3
          JSR  lcd_gotoxy
          JSR  lcdSetDataReg
          LD   R0,LIB_LCDMAXX
_lcdsft9  LDA  #20h
          SAP
          INC  PTR_L
          JSR  lcdSendByte
          JLP  _lcdsft9
          ;set cursor to beginning of last line
          LD   R0,#0
          LD   R1,#3
          JSR  lcd_gotoxy
          POP  PTR_H
          POP  PTR_L
          POP  R5
          POP  R4
          POP  R3
          POP  R1
          POP  R0
          RTS


lcd_print:
          ; Print a character on the LCD
          ; In: ACCU = character
          ; Changes: ACCU
          STA  PAR2
          AND  #0xE0
          TST
          JPF  _lcdpr1  ; value is below < 0x20
_lcdpr0   TST  LIB_LCDCRSR
          JPF  _lcdpr7
          LDA  LIB_LCDPOSX
          INC
          CMP  LIB_LCDMAXX
          JNF  _lcdpr7
          ;next printed character will result
          ;in moving the cursor outside the view area:
          ;the cursor must then be set to the next
          ;valid position
          PHL
          LDA  PAR2
          JSR  lcd_out_ch
          PSH  R0
          PSH  R1
          LDA  LIB_LCDPOSY
          CMP  #3       ;last line, last row? If yes:
          JPF  _lcdpr8  ;switch cursor off, because it is behind the scene
          LD   LIB_LCDPOSX,#0
          INC  LIB_LCDPOSY
          JSR  lcd_rst_crsr
          POP  R1
          POP  R0
          RTS
_lcdpr8   JSR  sendCrsrOffCmd
          POP  R1
          POP  R0
          RTS

_lcdpr7   ;continue with outputting the character
          LDA  LIB_LCDPOSX
          CMP  LIB_LCDMAXX
          LDA  PAR2
          JNF  lcd_out_ch
          ;wrap around into next LCD line
          PHL
          PSH  PAR2
          LD   LIB_LCDPOSX,#0
          LDA  LIB_LCDPOSY
          CMP  #3
          JPF  _lcdpr5
          INC  LIB_LCDPOSY
          PSH  R0
          PSH  R1
          JSR  lcd_rst_crsr
          POP  R1
          POP  R0
          JMP  _lcdpr6
_lcdpr5   ;shift content of display one line up, and clear last line
          JSR  lcd_shift
_lcdpr6   POP  ACCU
          JMP  _lcdout

_lcdpr1   ;test for special characters
          LDA  PAR2
          CMP  #0x0D  ;CR '\r'
          JPF  _lcdpr2
          CMP  #0x0A  ;LF '\n'
          JPF  _lcdpr4
          CMP  #0x08  ;Backspace
          JNF  _lcdpr0
          ;Backspace
          PHL
          TST  LIB_LCDPOSX
          JPF  rts_opc
          PSH  R0
          PSH  R1
          DEC  LIB_LCDPOSX
          DEC  LIB_LCDBUFP
          JSR  lcd_rst_crsr
          JMP  _lcdpr3
_lcdpr4   ;LF
          PHL
          PSH  R0
          PSH  R1
          JSR  lcd_newline
_lcdpr3   POP  R1
          POP  R0
          RTS
_lcdpr2   ;CR
          PHL
          PSH  R0
          PSH  R1
          JSR  lcd_return
          JMP  _lcdpr3

_lcdpr90  ;suppress printing the first "ok"
          TST  LIB_LCDOUT
          JPF  ret_opc
          CMP  #0x0A
          JNF  ret_opc
          LD   LIB_LCDOUT,#1
          RET
          

lcd_out_ch:
          ;Print character in accu to the LCD.
          ;Write chacter also to the backlog buffer.
          ;Correct character codes before printing (~ and \)
          ; This function does no safety checkings and
          ; does not support special control characters.
          PHL
_lcdout   CMP  #'~'
          JNF  _lcdoch1
          LDA  #0
_lcdoch1  CMP  #'\\'
          JNF  _lcdoch2
          LDA  #1
_lcdoch2  PSH  R1
          STA  R1
          PSH  PTR_L
          PSH  PTR_H
          LD   PTR_L,LIB_LCDBUFP
          LD   PTR_H,#>LCD_BUFFER
          SAP
          POP  PTR_H
          POP  PTR_L
          INC  LIB_LCDBUFP
          INC  LIB_LCDPOSX
          ; code inlined: (JSR  lcdSetDataReg)
          LDA  LIB_LCDENAS
          OUT
          OR   #0x04 ; SCL high, clock in the E-bits
          OUT
          LD   OUTP,#LCDDTMD
          ;
          JSR  lcdSendR1
          POP  R1
          RTS


lcd_out_fch:
          ; Same like lcd_out_fast, but corrects special characters
          CMP  #'~'
          JNF  _lcdofc1
          LDA  #0
_lcdofc1  CMP  #'\\'
          JNF  lcd_out_fast
          LDA  #1

lcd_out_fast:
          ; Print a character on the LCD *fast*.
          ; This function does no safety checkings and
          ; does not support special control characters.
          ; It also does not write to the LCD backlog
          ; buffer, so scrolling the LCD up is not possible.
          ; In: ACCU = character
          ; Changes: ACCU, R1
          STA  R1
          INC  LIB_LCDPOSX
          LDA  LIB_LCDENAS
          OUT
          OR   #0x04 ; SCL high, clock in the E-bits
          OUT
          LD   OUTP,#LCDDTMD
          JMP  lcdSendR1


lcd_rst_crsr:
          ; Reset cursor position to position stored in LIB_LCDPOSX/Y
          LD   R0,LIB_LCDPOSX
          LD   R1,LIB_LCDPOSY
          JMP  lcd_gotoxy


lcd_return:
          ; Move cursor to beginning of current line.
          ; Changes: ACCU, R0, R1
          LD   R1,LIB_LCDPOSY
          LD   R0,#0
          JMP  lcd_gotoxy


lcd_newline:
          ; Move cursor to beginning of next line.
          ; Changes: ACCU, R0, R1
          LDA  LIB_LCDPOSY
          CMP  #3
          JPF  lcd_shift
          LD   R0,#0
          LD   R1,LIB_LCDPOSY
          INC  R1

lcd_gotoxy:
          ; Set the cursor to the next position to print to.
          ; In: R0 = X-position, R1 = Y-position
          ; Changes: ACCU, R1
          PHL
          LD   LIB_LCDPOSX,R0
          LD   LIB_LCDPOSY,R1
          ;gotoxy on a big 4x40 LCD
          TST  LIB_LCDCRSR
          JPF  _lgt12
          JSR  sendCrsrOffCmd
          LD   R1,LIB_LCDPOSY
_lgt12    LD   LIB_LCDENAS,#LCDENA1
          LDA  #0x80
          LD   LIB_LCDBUFP,#<(LCD_BUFFER+0*40)
          TST  R1
          JPF  _lgt11
          DEC  R1
          LDA  #0xC0
          LD   LIB_LCDBUFP,#<(LCD_BUFFER+1*40)
          TST  R1
          JPF  _lgt11
          DEC  R1
          LD   LIB_LCDENAS,#LCDENA2
          LDA  #0x80
          LD   LIB_LCDBUFP,#<(LCD_BUFFER+2*40)
          TST  R1
          JPF  _lgt11
          LDA  #0xC0
          LD   LIB_LCDBUFP,#<(LCD_BUFFER+3*40)
_lgt11    AD   R0
          STA  R1
          JSR  lcdSetCmdReg
          JSR  lcdSendR1
          LDA  LIB_LCDBUFP
          AD   R0
          STA  LIB_LCDBUFP
          TST  LIB_LCDCRSR
          JPF  rts_opc
          LDA  #CRSR_ON_CMD
          JSR  lcdSendByte
          RTS


lcd_load_charset:
          ; Load a custom character set (up to 8 characters)
          ; into the LCD. Custom characters can be displayed
          ; with character codes 0-7.
          ; In: R0 = ID of first character to load (0-7),
          ;     R1 = count of characters to load (1-8),
          ;     PTR = pointer to bitmap data (8 byte per character),
          ;     note: set R0=0, R1=8 to load all 8 characters at once
          ; Changes: ACCU, R0, R1, PTR
          PHL
          PSH  R0
          PSH  R1
          PSH  PTR_L
          PSH  PTR_H
          LD   LIB_LCDENAS,#LCDENA1
          JSR  _lcdlcs3
          POP  PTR_H
          POP  PTR_L
          POP  R1
          POP  R0
          LD   LIB_LCDENAS,#LCDENA2
          JSR  _lcdlcs3
          RTS
_lcdlcs3  PHL
          PSH  R1
          LDA  R0
          CLC
          ROL
          ROL
          ROL
          OR   #0x40
          STA  R1
          JSR  lcdSetCmdReg
          JSR  lcdSendR1
          JSR  lcdSetDataReg
          POP  R0
_lcdlcs2  PSH  R0
          LD   R0,#8
_lcdlcs1  JSR  lap_inc_ptr
          JSR  lcdSendByte
          JLP  _lcdlcs1
          POP  R0
          JLP  _lcdlcs2
          JSR  lcd_rst_crsr
          RTS



;------------------------------------------------------------------------------
; Keyboard Functions
;------------------------------------------------------------------------------

keyboard_init:
          ; Detect the installed keyboard
          PHL
          LD   LIB_BIGKEYB,#0
          LD   R4_L,#0xF0
          LD   R4_H,#0x3F
          JSR  output_m
          IN
          ROL
          JPF  rts_opc
          LD   R4_H,#0xBF
          JSR  output_m
          IN
          ROL
          JNF  rts_opc
          ;successfully detected the big keyboard with 4x40 LCD
          LD   LIB_BIGKEYB,#1
          RTS


keyboard_term:
          ;small or no keyboard installed
          LD   LIB_BIGKEYB,#0
          RET


output_m:
          ; Output the eight M-bits on the additional 74HC574 on the keyboard
          ; and output OUT3-OUT7 on the output port.
          ; In: R4_L bits 3-7 = OUT3-OUT7, R4_H bits 0-7 = M-bits
          ; Changes: ACCU
          LDA  #0xF0
          AND  R4_H
          STA  PAR1
          LDA  #0x0B  ;SCL = low 
          AND  OUTP
          STA  OUTP
          OR   PAR1
          OUT
          OR   #0x04  ;SCL = high
          OUT
          LDA  R4_H
          CLC
          ROL
          CLC
          ROL
          CLC
          ROL
          CLC
          ROL
          OR   OUTP   ;SCL = low 
          OUT
          OR   #0x04  ;SCL = high
          OUT
          LDA  R4_L
          OR   OUTP   ;SCL = low 
          OUT
          STA  OUTP
          RET


keyb_readkey:
          ; Read the keyboard (if any is installed).
          ; Out: ACCU = key code or 0 if no button was pressed.
          ;      See KC_xxx - defines in keydefs.hsm for details.
          ; Changes: ACCU, R0, PTR
          INC  RANDVAR

          ;read the big keyboard here
          PHL
          PSH  R4_L
          PSH  R4_H
          ;Test if any key is pressed. Exit early if not.
          LD   R4_L,#0x00
          LD   R4_H,#0xA0
          JSR  output_m
          IN
          LD   PAR2,#0xFC
          AND  PAR2
          CMP  #0xFC
          JPF  _rdbk05  ;no key pressed
          ;test if no key was pressed within last 8 polls (debounce keyboard)
          TST  LIB_KBNOKEYB
          LD   LIB_KBNOKEYB,#0xFF
          JPF  _rdbk08
          ;a key is still pressed,
          ;maintain the key-repeat-counter
          TST  LIB_KEYREPC+0
          JNF  _rdbk09
          TST  LIB_KEYREPC+1
          JPF  _rdbk10 
          DEC  LIB_KEYREPC+1
_rdbk09   DEC  LIB_KEYREPC+0
          JMP  _rdbk06 ;some key was already pressed, ignore this keystroke and re-start debouncing
_rdbk10   LD   LIB_KEYREPC+0,#100
_rdbk08   ;scan all key rows, begin with OUT4
          LD   R4_L,#0xE0
          LD   R4_H,#0xBF
          LD   R0,#9
_rdbk01   JSR  output_m
          IN
          AND  PAR2
          CMP  #0xFC
          JNF  _rdbk02  ;some key is pressed
          ;scan next row
          CLC
          RWL  R4
          LDA  #0x10
          OR   R4_L
          STA  R4_L
          LDA  #0xC0
          XOR  R4_H
          STA  R4_H
          JLP  _rdbk01
          ;no key pressed, may be shift was pressed, but ignore it.
          LD   LIB_KBNOKEYB,#0
          JMP  _rdbk06
_rdbk02   DEC  R0
          ;R0 = row
          ;find the column now and
          ;add column-number (0 to 5) to (row number * 8)
          STA  PAR2
          ROL  R0
          ROL  R0
          ROL  R0
          DEC  R0
_rdbk03   INC  R0
          ROL  PAR2
          JPF  _rdbk03
          ;check the shift-key status, choose other table if shift is pressed
          LD   R4_L,#0xF0
          LD   R4_H,#0xFF
          JSR  output_m
          IN
          ROL
          LD   PTR_L,#<TAB_KEYNORM
          LD   PTR_H,#>TAB_KEYNORM
          JPF  _rdbk04
          LD   PTR_L,#<TAB_KEYSHIFT
          LD   PTR_H,#>TAB_KEYSHIFT
_rdbk04   ;translate the code
          LDA  R0
          AD   PTR_L
          STA  PTR_L
          JNF  _rdbk11
          INC  PTR_H
_rdbk11   LAP
          ;return key-code in ACCU
          JMP  _rdbk07
_rdbk05   LD   LIB_KEYREPC+0,#100
          LD   LIB_KEYREPC+1,#2
          CLC
          ROL  LIB_KBNOKEYB
_rdbk06   LDA  #0
_rdbk07   POP  R4_H
          POP  R4_L
          RTS


keyb_inputstring:
          ;input a string on the LCD
          ;max. line length is 40 character
          ;Input:
          ;  R0 = buffer size
          ;  R4 = ptr to beginning of the buffer
          PHL
          LD   R2,#0      ; R2 = entered string length
          DEC  R0
          LD   R7_L,LIB_LCDPOSX ; R7_L = left x-position of input field
          LDA  #39   ;right most screen position, this is 39 on a 40x4 LCD
          SU   R7_L
          STA  R7_H  ;R7_H = maximum length of the input string
          SU   R0
          JNF  _inpl01
          LD   R7_H,R0
_inpl01   ;R7_L = left x-position of input field
          ;R7_H = max. buffer length / input length
          ;R4 = ptr to beginning of the buffer
          ;R2 = cursor position
          ;R3 = current length of the string
          JSR  ld_ptr_r4
          LDA  #0
          ;STA  R2   ;already set to zero in the code above (input_string)
          STA  R3
          SAP
         
_inpl02   ;print input buffer and position the cursor
          LDA  R7_L
          JSR  _inpl16
          JSR  ld_ptr_r4
_inpl04   JSR  lap_inc_ptr
          TST
          JPF  _inpl03
          JSR  lcd_out_ch
          JMP  _inpl04
_inpl03   LDA  #0x20
          JSR  lcd_out_ch

_inpl12   ;position the cursor
          LDA  R2
          JSR  _inpl15

_inpl05   ;read next key
          PSH  R7_L
          PSH  R7_H
          JSR  rom_readkey
          POP  R7_H
          POP  R7_L
          
          CMP  #KCODE_LEFT
          JPF  _inpl06
          CMP  #KCODE_RIGHT
          JPF  _inpl07
          CMP  #13
          JPF  _inpl08
          CMP  #8
          JPF  _inpl09
          
          ;check if character is printable
          STA  R1
          LDA  #0xE0
          AND  R1
          TST
          JPF  _inpl05
          ;print it
          ;1. check if there is "space" at the end of the line
          LDA  R3
          CMP  R7_H
          JPF  _inpl05  ;no space, ignore this keypress
          ;2. move rest of the line to the right to insert space for the new character
          LDA  R3
          AD   R4_L
          STA  PTR_L
          LD   PTR_H,R4_H
          JNF  _inpl10
          INC  PTR_H
_inpl10   LDA  R3
          SU   R2
          INC   ;move also the terminating zero at the end of the string
          STA  R0
_inpl11   JSR  lap_inc_ptr
          SAP
          JSR  dec_ptr
          JSR  dec_ptr
          JLP  _inpl11
          ;3. write character into memory and print out the new line
          JSR  inc_ptr
          LDA  R1
          SAP
          INC  R2
          INC  R3
          JMP  _inpl02

_inpl06   ;cursor left
          TST  R2
          JPF  _inpl05
          DEC  R2
          JMP  _inpl12

_inpl07   ;cursor right
          LDA  R2
          CMP  R3
          JPF  _inpl05
          INC  R2
          JMP  _inpl12

_inpl09   ;delete
          TST  R2
          JPF  _inpl05
          LDA  R2
          AD   R4_L
          STA  PTR_L
          LD   PTR_H,R4_H
          JNF  _inpl13
          INC  PTR_H
_inpl13   LDA  R3
          SU   R2
          STA  R0
          INC  R0  ;move also the terminating zero at the end of the string
_inpl14   LAP
          JSR  dec_ptr
          JSR  sap_inc_ptr
          JSR  inc_ptr
          JLP  _inpl14
          DEC  R2
          DEC  R3
          JMP  _inpl02

_inpl15   ;set cursor to offset in accu
          AD   R7_L
_inpl16   STA  R0
          LD   R1,LIB_LCDPOSY
          JMP  lcd_gotoxy

_inpl08   ;return
          LDA  R3
          JSR  _inpl15
          RTS



;------------------------------------------------------------------------------
; Word List
;------------------------------------------------------------------------------

ED2_WINX    SET LIB_BUF+0  ; first row of a line shown on the LCD (0-26) (LCD window top-left X-coordinate)
ED2_WINY    SET LIB_BUF+1  ; first line shown on the LCD (0 - 17) (LCD window top-left Y-coordinate)
ED2_LINE    SET LIB_BUF+2  ; current line, can be 0 - 17, 0 and 17 = head and bottom line, others = screen line
ED2_XPOS    SET LIB_BUF+3  ; position within a line (0 to 63, or 0 to 3 in head and bottom line)
ED2_SCRNUM  SET LIB_BUF+4  ; block/screen number (2 bytes)
ED2_SCRBUF  SET LIB_BUF+6  ; ptr to block buffer (2 bytes)
ED2_SXPOS   SET LIB_BUF+8  ; saved value of ED2_XPOS before activation of the menu line
ED2_LCDUPD  SET LIB_BUF+9  ; LCD update flag: 0=set only cursor, 0x80=update current line, others: update whole display,
ED2_TOUCH   SET LIB_BUF+10 ; Flag: nonzero when screen was modified

CHAR_NBR10  SET 2
CHAR_ALEFT  SET 3
CHAR_ARIGHT SET 4


w_lcd:    ; LCD  ( -- )
          PHL
          TST  LIB_LCDOUT
          JNF  rts_opc
          JSR  keyboard_init
          JSR  lcd_init
          RTS

            
w_terminal: ; TERMINAL ( -- )
          PHL
          TST  LIB_LCDOUT
          JPF  rts_opc
          JSR  lcd_clear
          LD   PTR_L,#<text_useterm
          LD   PTR_H,#>text_useterm
          JSR  rom_print_str
          JSR  keyboard_term
          JSR  lcd_cursor_off
          JSR  rom_lcd_term  ;vector for "terminal_init"
          LD   LIB_LCDOUT,#0
          RTS


w_keyq:   ; KEY?  ( -- flag )  If a character is available, return true.  Otherwise, return false.
          TST  LIB_LCDOUT
          JPF  c_key
          PHL
          TST  NEXTINCHR
          JNF  ret_true
          JSR  keyb_readkey
          STA  NEXTINCHR
          TST
          JPF  ret_false
ret_true  LD   R4_L,#0xFF
          JMP  _rettf
ret_false LD   R4_L,#0
_rettf    LD   R4_H,R4_L
          JSR  rom_push_data_R4
          RTS


w_list    ; LIST ( n -- )
          TST  LIB_LCDOUT
          JPF  c_list

w_edit    ; EDIT ( n -- )
          TST  LIB_LCDOUT
          JPF  c_edit
          PHL
          LD   PTR_L,#<text_loading
          LD   PTR_H,#>text_loading
          JSR  rom_print_str
          JSR  c_emptybu
          JSR  c_dup
          JSR  c_block
          JSR  c_dup
          JSR  mkprintable
          ;on stack: blocknmbr, buffer
          
          JSR  rom_pop_data_R4
          LD   ED2_SCRBUF+0,R4_L
          LD   ED2_SCRBUF+1,R4_H
          JSR  rom_pop_data_R4
          LD   ED2_SCRNUM+0,R4_L
          LD   ED2_SCRNUM+1,R4_H
          
          ;initialize the LCD
          LD   PTR_L,#<ed2_chars
          LD   PTR_H,#>ed2_chars
          LD   R0,#CHAR_NBR10
          LD   R1,#3
          JSR  lcd_load_charset
          JSR  lcd_clear
          
          ;make string from screen number
          LD   R4_L,ED2_SCRNUM+0
          LD   R4_H,ED2_SCRNUM+1
          JSR  convert_16bit_to_decimal
          LD   PTR_L,#<NUM_BUF
          LD   PTR_H,#>NUM_BUF
          LDA  #'S'
          JSR  sap_inc_ptr
          LDA  #'C'
          JSR  sap_inc_ptr
          LDA  #'R'
          JSR  sap_inc_ptr
          LDA  LIB_BUF+2
          JSR  sap_inc_ptr
          LDA  LIB_BUF+3
          JSR  sap_inc_ptr
          LDA  LIB_BUF+4
          JSR  sap_inc_ptr
          LDA  #' '
          JSR  sap_inc_ptr
          LDA  #0
          SAP
          
          ;initialize the editor
          ;LDA  #0  -> accu is already 0
          STA  ED2_WINX
          STA  ED2_WINY
          STA  ED2_XPOS
          STA  ED2_TOUCH
          LDA  #1
          STA  ED2_LINE
          STA  ED2_LCDUPD
          
          ;Main loop
          ;update display and set the cursor
          ;depending on the state of the ED2_LCDUPD-flag
_cedit201 INC  ED2_LCDUPD
_cedit200 TST  ED2_LCDUPD
          JPF  _cedit219
          LDA  ED2_LCDUPD
          CMP  #0x80
          JNF  _cedit218
          LD   R0,ED2_LINE
          LD   R1,LIB_LCDPOSY
          JSR  _c_ed2_prline
          JMP  _cedit229
_cedit218 JSR  _c_ed2_updDisplay
_cedit229 LD   ED2_LCDUPD,#0
_cedit219 JSR  _c_ed2_setCursor
_cedit202 ;wait for keypress and get the key code
          JSR  rom_readkey
          STA  R3 ;save a copy of the key code in R3
          CMP  #KCODE_LEFT
          JPF  _cedit203
          CMP  #KCODE_RIGHT
          JPF  _cedit204
          CMP  #KCODE_UP
          JPF  _cedit205
          CMP  #KCODE_DOWN
          JPF  _cedit206
          JSR  _c_ed2_isMenuLine
          LDA  R3
          JPF  _cedit217
          CMP  #13
          JPF  _cedit215
          CMP  #8
          JPF  _cedit222
          
          ;check if character is printable
          LDA  #0xE0
          AND  R3
          TST
          JPF  _cedit202
          ;print it
          ;1. check if there is "space" at the end of the line
          LD   R1,#63
          JSR  _c_ed_calcCurMemLinePtr
          LAP
          CMP  #0x20
          JNF  _cedit202  ;no space, ignore this keypress
          ROR  ED2_TOUCH  ;set flag
          ;2. move rest of the line to the right to insert space for the new character
          LDA  #63
          SU   ED2_XPOS
          TST
          JPF  _cedit220
          STA  R0
_cedit221 JSR  dec_ptr
          JSR  lap_inc_ptr
          SAP
          JSR  dec_ptr
          JLP  _cedit221
          ;3. write character into memory
_cedit220 LDA  R3
          SAP
          ;update at least the current line in the display
          LD   ED2_LCDUPD,#0x80
          ;move cursor to the right
          JMP  _cedit204
          
_cedit217 ;handle the MENU
          CMP  #13
          JNF  _cedit202
          LDA  ED2_XPOS
          TST
          JPF  _cedit225
          CMP  #1
          JPF  _cedit226
          CMP  #2
          JPF  _cedit227
          
          ;clear screen
          LD   R4_L,ED2_SCRBUF+0
          LD   R4_H,ED2_SCRBUF+1
          JSR  rom_push_data_R4
          JSR _c_clearScreen
          INC  ED2_TOUCH
          JMP  _cedit218
          
_cedit225 ;save
          JSR  _cedit228
          JMP  _cedit218
          
_cedit226 ;save + quit
          JSR  _cedit228
          JMP  _cedit300
          
_cedit228 ;save screen and return
          PHL
          JSR  lcd_clear
          LD   PTR_L,#<text_saving2
          LD   PTR_H,#>text_saving2
          JSR  rom_print_str
          JSR  print_nl
          JSR  c_update
          JSR  c_savebuf
          RTS
          
_cedit222 ;handle deletion of a character
          TST  ED2_XPOS
          JPF  _cedit202  ;nothing to delete, ignore this keypress
          LD   ED2_LCDUPD,#0x80
          LD   R1,ED2_XPOS
          JSR  _c_ed_calcCurMemLinePtr
          LDA  #63
          STA  ED2_TOUCH
          SU   ED2_XPOS
          STA  R0
          TST  R0
          INC  R0
          JNF  _cedit224
          LAP
          CMP  #0x20
          JPF  _cedit224
          LDA  #0x20
          SAP
          JMP  _cedit200
_cedit223 JSR  inc_ptr
_cedit224 LAP
          JSR  dec_ptr
          JSR  sap_inc_ptr
          JLP  _cedit223
          LDA  #0x20
          SAP
          
_cedit203 ;cursor left
          TST  ED2_XPOS
          JPF  _cedit202
          JSR  _c_ed2_isMenuLine
          LDA  ED2_XPOS
          DEC  ED2_XPOS
          JPF  _cedit201
          CMP  ED2_WINX
          JNF  _cedit200
          DEC  ED2_WINX
          JMP  _cedit201
          
_cedit204 ;cursor right
          JSR  _c_ed2_isMenuLine
          LDA  ED2_XPOS
          JNF  _cedit209
          CMP  #3
          JPF  _cedit202
          INC  ED2_XPOS
          JMP  _cedit201
_cedit209 CMP  #63
          JPF  _cedit200 ;print entered character without scrolling to the right
          INC  ED2_XPOS
          LDA  ED2_XPOS
          SU   #36
          JNF  _cedit200
          INC  ED2_WINX
          CMP  ED2_WINX
          JPF  _cedit201
          DEC  ED2_WINX
          JMP  _cedit200
          
_cedit205 ;cursor up
          TST  ED2_LINE
          JPF  _cedit202
          JSR  _c_ed2_isMenuLine
          LDA  ED2_LINE
          DEC  ED2_LINE
          JNF  _cedit212
_cedit213 LD   ED2_XPOS,ED2_SXPOS
          JMP  _cedit201
_cedit212 CMP  ED2_WINY
          JNF  _cedit230 ;_cedit200
          DEC  ED2_WINY
_cedit211 JSR  _cedit214
          JMP  _cedit201
          
_cedit214 ;enter menu-line, return with FLAG=1 when display must be updated
          PHL
          JSR  _c_ed2_isMenuLine
          JNF  rts_opc
          LD   ED2_SXPOS,ED2_XPOS
          LD   ED2_XPOS,#1
          TST  ED2_TOUCH
          JNF  return_flag_1
          INC  ED2_XPOS
          RTS
          
_cedit215 ;return
          LD   ED2_XPOS,#0
          LD   ED2_WINX,#0
          LDA  ED2_LINE
          CMP  #16
          JPF  _cedit201
          
_cedit206 ;cursor down
          LDA  ED2_LINE
          CMP  #17
          JPF  _cedit202
          TST  ED2_LINE
          INC  ED2_LINE
          JPF  _cedit213
          SU   #3
          CMP  ED2_WINY
          JNF  _cedit216
          INC  ED2_WINY
          JMP  _cedit211
_cedit216 LDA  #13
          CMP  R3
          JPF  _cedit211
_cedit230 JSR  _cedit214
          JPF  _cedit201
          JMP  _cedit200


_c_ed2_setCursor:
          ;set cursor to current position
          PHL
          JSR  _c_ed2_isMenuLine
          JNF  _ced2sc01
          LD   R0,R4_L
          INC  R0
          JMP  _ced2sc02
_ced2sc01 LDA  ED2_XPOS
          SU   ED2_WINX
          INC
          INC
          STA  R0
_ced2sc02 LDA  ED2_LINE
          SU   ED2_WINY
          STA  R1
          JSR  lcd_gotoxy
          RTS

_c_ed2_updDisplay:
          ;update the display, print out all lines
          PHL
          LD   R5_L,#0
          LD   R5_H,ED2_WINY
_ced2upd1 LD   R1,R5_L
          LD   R0,R5_H
          JSR  _c_ed2_prline
          INC  R5_H
          INC  R5_L
          LDA  R5_L
          CMP  #4
          JNF  _ced2upd1
          RTS

_c_ed2_isMenuLine:
          LDA  ED2_LINE
_c_ed2_isMLaccu:
          ;Check if the line numbe in ACCU is line 0 or line 17.
          TST
          JPF  ret_opc
          CMP  #17
          RET

_c_ed_calcCurMemLinePtr:
          LDA  ED2_LINE
_c_ed_calcMemLinePtr:  ;shared code
          DEC
          STA  PTR_L
          LD   PTR_H,#0
          LD   R0,#6
          CLC
_ced2pl16 RWL  PTR_L
          JLP  _ced2pl16
          LDA  R1
          AD   PTR_L
          AD   ED2_SCRBUF+0
          STA  PTR_L
          LDA  PTR_H
          ADD  ED2_SCRBUF+1
          STA  PTR_H
          RET
            
_c_ed2_prline:
          ;Print a text line to the screen
          ;In: R0 = line 0-17, R1 = LCD Y-position 0-3
          ;Out: In line 0 and line 17, R4_L is the position of the selected '<'
          PHL
          LD   R2,R0
          LD   R0,#0
          JSR  lcd_gotoxy
          LDA  R2
          JSR  _c_ed2_isMLaccu
          JPF  _ced2pl01
          ;print text line
          ;1. print line number (hex 0x1-0x10)
          LDA  #9
          SU   R2
          JNF  _ced2pl11
          LDA  #0x30
          OR   R2
          JMP  _ced2pl10
_ced2pl11 LDA  #15
          SU   R2
          JNF  _ced2pl12
          LDA  #'A'-10
          AD   R2
          JMP  _ced2pl10
_ced2pl12 LDA  #CHAR_NBR10 ;special character "10"
_ced2pl10 JSR  lcd_out_fch
          ;2. print | or <-
          LDA  #'|'
          TST  ED2_WINX
          JPF  _ced2pl13
          LDA  #CHAR_ALEFT
_ced2pl13 JSR  lcd_out_fch
          ;3. print the line
          LDA  R2
          LD   R1,ED2_WINX
          JSR  _c_ed_calcMemLinePtr
          LD   R0,#37
_ced2pl17 LAP
          INC  PTR_L
          TST  PTR_L
          JNF  _ced2pl18
          INC  PTR_H
_ced2pl18 JSR  lcd_out_fch
          JLP  _ced2pl17
          ;print last character '->' only if required 
          LDA  #27
          CMP  ED2_WINX
          LDA  #'|'
          JPF  _ced2pl15
          LDA  #CHAR_ARIGHT
_ced2pl15 JSR  lcd_out_fch
          RTS
_ced2pl01 ;print head/bottom-line
          LD   R3,#7
          LD   R2,ED2_XPOS
          JSR  _c_ed2_isMenuLine
          JPF  _ced2pl09
          LD   R2,#0x80
_ced2pl09 LD   PTR_L,#<NUM_BUF
          LD   PTR_H,#>NUM_BUF
          JSR  rom_print_str
          LD   PTR_L,#<text_ed_head
          LD   PTR_H,#>text_ed_head
_ced2pl02 JSR  lap_inc_ptr
          TST
          JPF  rts_opc
          TST  R2
          JNF  _ced2pl03
          JSR  is_alpha
          JNF  _ced2pl05  ;convert abc to ABC
          CMP  #'['
          JPF  _ced2pl08
          CMP  #']'
          JNF  _ced2pl06
          DEC  R2
          JMP  _ced2pl07
_ced2pl08 LD   R4_L,R3
_ced2pl07 SU   #0x1F      ;convert [/] to </>
          JMP  _ced2pl03
_ced2pl05 AND  #0xDF
_ced2pl03 CMP  #']'
          JNF  _ced2pl06
          DEC  R2
_ced2pl06 JSR  lcd_out_fch
          INC  R3
          JMP  _ced2pl02

_cedit227 ;quit
          JSR  lcd_clear
_cedit300 JSR  c_emptybu
          RTS

_c_clearScreen  ; ( bufaddr -- )
          ;fill screen buffer with spaces
          PHL
          LD   R4_L,#0
          LD   R4_H,#4
          JSR  rom_push_data_R4
          LD   R4_L,#32
          LD   R4_H,#0
          JSR  rom_push_data_R4
          JSR  c_fill
          RTS

; convert non-printable characters to '?'
mkprintable ;input: buffer-ptr on stack
          PHL
          JSR  rom_pop_data_R4
          JSR  ld_ptr_r4
          LD   R1,#4
          LD   R0,#0
          LD   PAR2,#0x20
_mkpr01   LAP
          INC
          STA  R2
          ROL
          JPF  _mkpr03
          LDA  R2
          SUB  PAR2
          JPF  _mkpr04
_mkpr03   LDA  #'?'
          SAP
_mkpr04   INC  PTR_L
          TST  PTR_L
          JNF  _mkpr02
          INC  PTR_H
_mkpr02   JLP  _mkpr01
          DEC  R1
          TST  R1
          JNF  _mkpr01
          RTS

convert_16bit_to_decimal:
          PHL
          LD   PTR_H,#>(LIB_BUF+5)
          LD   PTR_L,#<(LIB_BUF+5)
          LD   R0,#5
_cnvd01   LD   R5_L,#10
          LD   R5_H,#0
          PSH  LIB_BUF+0
          PSH  LIB_BUF+1
          PSH  LIB_BUF+2
          PSH  LIB_BUF+3
          JSR  divide_u
          POP  LIB_BUF+3
          POP  LIB_BUF+2
          POP  LIB_BUF+1
          POP  LIB_BUF+0
          LDA  #0x30
          OR   R5_L
          JSR  dec_ptr
          SAP
          JLP  _cnvd01
          RTS

divide_u:
          ; Divide a 16-bit unsigned number by an other 16-bit unsigned number.
          ; In : R4 = divident, R5 = divisor
          ; Out: R4 = 16-bit result, R5 = 16-bit reminder
          ; Changes: ACCU, R4, R5
          LD   LIB_BUF+0,#0
          LD   LIB_BUF+1,#0
          PSH  R0
          CLC
          LD   R0,#16
_divi01   RWL  R4
          RWL  LIB_BUF+0
          LDA  LIB_BUF+0
          SU   R5_L
          STA  PAR1
          LDA  LIB_BUF+1
          SUB  R5_H
          JNF  _divi02
          LD   LIB_BUF+0,PAR1
          STA  LIB_BUF+1
_divi02   RWL  LIB_BUF+2
          JLP  _divi01
          POP  R0
          LD   R4_L,LIB_BUF+2
          LD   R4_H,LIB_BUF+3
          LD   R5_L,LIB_BUF+0
          LD   R5_H,LIB_BUF+1
          RET


;------------------------------------------------------------------------------
END_MODULE  ; This line terminates the code section of this binary module
