;==============================================================================
; This module implements the basic string words of the
; "Forth 2012 optional String word set", which are:
;   
;   -TRAILING
;   /STRING
;   BLANK
;   CMOVE
;   CMOVE>
;   COMPARE
;   SEARCH
;   SLITERAL
;   
; Assemble this module with the myca command
;   $ myca m4-strings.asm -o m4-strings.bin
;
; Upload the binary module to My4TH with the my4th tool
; (for example. See "my4th --help" for details):
;   $ my4th write /dev/ttyS0 binary 15 m4-strings.bin
;
; Load the module on My4TH with the BLOAD word:
;   15 BLOAD
;
; (15 is a block number in the EEPROM, please choose a suited number
; for your system to avoid overwriting important EEPROM content)
;==============================================================================

#include <my4th/binmod.hsm>


;------------------------------------------------------------------------------
; Exported Words
;------------------------------------------------------------------------------

EXPORTS:
          EXPORTI e_sliteral,   "sliteral"   ; export an immediate word
          EXPORT  e_blank,      "blank"
          EXPORT  c_move,       "cmove"
          EXPORT  e_cmovegt,    "cmove>"
          EXPORT  e_compare,    "compare"
          EXPORT  e_sstring,    "/string"
          EXPORT  e_dtrailing,  "-trailing"
          EXPORT  e_search,     "search"
          EEND


;------------------------------------------------------------------------------
; Imported Words
;------------------------------------------------------------------------------

IMPORTS:
c_swap    IMPORT  "swap"
c_over    IMPORT  "over"
c_dup     IMPORT  "dup"
c_rot     IMPORT  "rot"
c_mrot    IMPORT  "-rot"
c_tuck    IMPORT  "tuck"
c_here    IMPORT  "here"
c_1plus   IMPORT  "1+"
c_add     IMPORT  "+"
c_sub     IMPORT  "-"
c_allot   IMPORT  "allot"
c_move    IMPORT  "move"
c_fill    IMPORT  "fill"
c_bl      IMPORT  "bl"
c_to_r    IMPORT  ">r"
c_r_from  IMPORT  "r>"
c_literal IMPORT  "literal"
          IEND


;------------------------------------------------------------------------------
; Code Section
;------------------------------------------------------------------------------

CODE:

init_module:
          ;Nothing to do here.
          RET


e_sliteral:
          ;Implementation of the Forth word SLITERAL
          ;( c-addr u -- ) 
          PHL
          JSR  c_here
          JSR  c_3plus
          JSR  c_dup
          JSR  c_to_r
          ;( c-addr u [here+3] -- ) 
          JSR  c_over
          JSR  c_add
          ;( c-addr u [here+u+3] ) 
          JSR  c_over
          JSR  c_3plus
          ;( c-addr u [here+u+3] [u+3] ) 
          JSR  c_here
          JSR  c_swap
          ;( c-addr u [here+u+3] here [u+3] )
          ;allocate memory for a JMP instruction and the string
          JSR  c_allot
          ;( c-addr u [here+u+3] here )
          ;insert a JMP instruction to jump
          ;over the following embedded string
          JSR  rom_pop_data_R5  ;dest.ptr
          JSR  rom_pop_data_R4  ;jump target
          LD   PTR_L,R5_L
          LD   PTR_H,R5_H
          LDA  #OPCODE_JMP
          JSR  sap_inc_ptr
          LDA  R4_L
          JSR  sap_inc_ptr
          LDA  R4_H
          SAP
          ;( c-addr u )
          JSR  c_swap
          ;( u c-addr )
          JSR  c_over
          ;( u c-addr u )
          JSR  c_r_from
          JSR  c_dup
          JSR  c_to_r
          ;( u c-addr u [here+3] )
          ;embed the string into the current definition
          JSR  c_swap
          JSR  c_move
          ;( u )
          JSR  c_r_from
          ;( u [here+3] )
          ;add the two literals: ( -- c-addr u )
          JSR  c_literal
          JSR  c_literal
          RTS

c_3plus:  ;Implementation of custom word "3+"
          PHL
          JSR  c_1plus
          JSR  c_1plus
          JSR  c_1plus
          RTS


e_blank:
          ;Implementation of the Forth word BLANK ( c-addr u -- )
          PHL
          JSR  c_bl
          JSR  c_fill
rts_opc:  RTS


e_cmovegt:
          ;Implementation of the Forth word CMOVE> ( c-addr1 c-addr2 u -- )
          PHL
          JSR  rom_pop_data_R4
          PSH  R4_L
          PSH  R4_H
          JSR  rom_push_data_R4
          JSR  rom_push_data_R4
          JSR  c_rot
          JSR  c_add
          JSR  c_mrot
          JSR  c_add
          JSR  rom_pop_data_R4
          JSR  rom_pop_data_R5
          POP  R6_H
          POP  R6_L
_cmvgt01  TST  R6_L
          JNF  _cmvgt02
          TST  R6_H
          JPF  rts_opc
_cmvgt02  JSR  dec_r6
          JSR  dec_r5
          JSR  dec_r4
          LD   PTR_L,R4_L
          LD   PTR_H,R4_H
          LAP
          LD   PTR_L,R5_L
          LD   PTR_H,R5_H
          SAP
          JMP  _cmvgt01


e_compare:
          ;Implementation of the Forth word COMPARE ( c-addr1 u1 c-addr2 u2 -- n )
          PHL
          JSR  rom_pop_data_R5
          LD   R3,R5_L
          JSR  rom_pop_data_R5
          JSR  rom_pop_data_R4
          LD   R2,R4_L
          JSR  rom_pop_data_R4
          ;string 1: R4(addr) / R2(len)
          ;string 2: R5(addr) / R3(len)
_comp01   ;loop
          TST  R2
          JPF  _comp02
          TST  R3
          JPF  ret_1    ;the strings are identical, but u1 > u2, thus return 1
          DEC  R2
          DEC  R3
          JSR  loadChrR4
          STA  R0
          LD   PTR_L,R5_L
          LD   PTR_H,R5_H
          LAP
          JSR  inc_r4
          JSR  inc_r5
          CMP  R0
          JPF  _comp01
          ;characters are not equal, return 1 or -1
          SU   R0
          JNF  ret_1
          JMP  ret_m1
_comp02   TST  R3
          JPF  ret_0    ;the strings are identical, return flag=0
          ;the strings are identical, but u1 < u2, thus return -1
ret_m1    ;return -1
          LDA  #0xFF
          JMP  _retf1
ret_1     ;return 1
          LD   R4_L,#1
          LDA  #0
          JMP  _retf2
ret_0     ;return 0
          LDA  #0
_retf1    STA  R4_L
_retf2    STA  R4_H
          JSR  rom_push_data_R4
          RTS


e_sstring:
          ;Implementation of the Forth word /STRING ( c-addr1 u1 n -- c-addr2 u2 )
          PHL
          JSR  c_tuck
          JSR  c_sub
          JSR  c_mrot
          JSR  c_add
          JSR  c_swap
          RTS


e_dtrailing:
          ;Implementation of the Forth word -TRAILING ( c-addr u1 -- c-addr u2 )
          PHL
          JSR  c_over
          JSR  c_over
          JSR  c_add
          JSR  rom_pop_data_R4
          ;R4 = last address + 1
          JSR  rom_pop_data_R5
          ;R5_L = remaining characters
_dtrai02  TST  R5_L
          JPF  _dtrai01
          JSR  dec_r4
          JSR  loadChrR4
          CMP  #0x20
          JNF  _dtrai01
          DEC  R5_L
          JMP  _dtrai02
_dtrai01  JSR  rom_push_data_R5
          RTS
          

e_search:
          ;Implementation of the Forth word SEARCH ( c-addr1 u1 c-addr2 u2 -- c-addr3 u3 flag )
          PHL
          JSR  rom_pop_data_R4
          LD   R3,R4_L  ; u2
          JSR  rom_pop_data_R4
          TST  R3
          JPF  ret_m1       ; return true because search string is empty
          LD   R7_L,R4_L
          LD   R7_H,R4_H    ; c-addr2
          JSR  rom_pop_data_R4
          JSR  rom_pop_data_R5
          JSR  rom_push_data_R5  ; c-addr1
          JSR  rom_push_data_R4  ; u1
          ;now on stack: ( c-addr1 u1 ) for the case the string will not be found
          LDA  R4_L
          SU   R3
          JNF  ret_0  ; error, u2 > u1
          STA  R2     ; max. offset for c-addr1
          ;loop over all possible start positions in string at c-addr1
_sstr01   LD   R6_L,R7_L
          LD   R6_H,R7_H
          LD   R4_L,R5_L
          LD   R4_H,R5_H
          LD   R0,R3  ; u2
_sstr02   ;compare the string
          JSR  loadChrR4
          STA  R1
          LD   PTR_L,R6_L
          LD   PTR_H,R6_H
          LAP
          JSR  inc_r4
          JSR  inc_r6
          CMP  R1
          JNF  _sstr03
          JLP  _sstr02
          ;found, drop c-addr1 and u1
          JSR  rom_pop_data_R4
          JSR  rom_pop_data_R4
          ;push c-addr3 and u3
          JSR  rom_push_data_R5
          LDA  R3
          AD   R2
          STA  R4_L
          LD   R4_H,#0
          JSR  rom_push_data_R4
          JMP  ret_m1     ;return true
_sstr03   ; c-addr1 is max. R2 times incremented
          JSR  inc_r5
          TST  R2
          DEC  R2
          JNF  _sstr01
          JMP  ret_0  ; string not found


loadChrR4:
          ;load the characters pointed by pointer in R4 into accu
          LD   PTR_L,R4_L
          LD   PTR_H,R4_H
          LAP
          RET

sap_inc_ptr: ;store accu and increment ptr
          SAP
          INC  PTR_L
          TST  PTR_L
          JNF  ret_opc
          INC  PTR_H
ret_opc   RET

inc_r4:   ;increment R4
          INC  R4_L
          TST  R4_L
          JNF  ret_opc
          INC  R4_H
          RET

inc_r5:   ;increment R5
          INC  R5_L
          TST  R5_L
          JNF  ret_opc
          INC  R5_H
          RET

inc_r6:   ;increment R6
          INC  R6_L
          TST  R6_L
          JNF  ret_opc
          INC  R6_H
          RET

dec_r4:   ;decrement R4
          TST  R4_L
          JNF  _decr4a
          DEC  R4_H
_decr4a   DEC  R4_L
          RET

dec_r5:   ;decrement R5
          TST  R5_L
          JNF  _decr5a
          DEC  R5_H
_decr5a   DEC  R5_L
          RET

dec_r6:   ;decrement R6
          TST  R6_L
          JNF  _decr6a
          DEC  R6_H
_decr6a   DEC  R6_L
          RET


;------------------------------------------------------------------------------
END_MODULE
