;------------------------------------------------------------------------------
; EEPROM Functions
;------------------------------------------------------------------------------
; 
; Possible board configurations:
;
; EEPROM Socket 1:
;     24LC512 : 0xA0
;     24LC1025: 0xA0 / 0xA8
;     24LC1026: 0xA0 / 0xA2
; 
; EEPROM Socket 2:
;     24LC512 : 0xA6
;     24LC1025: 0xA6 / 0xAE
;     24LC1026: 0xA4 / 0xA6
;
; If the board is populated with two 24LC512, and two more 24LC512 shall be
; connected, the these 24LC512 must have the addresses 0xAA and 0xAC.
;

SEG_LOWCODE

eeprom_init:
          ; Initialize the EEPROM library.
          ; Find all EEPROMs attached to the system.
          PHL
          LD   PTR_L,#LIB_EEADDRS
          LD   PTR_H,#REGPAGE_HI
          LD   R0,#8
          JSR  memclr
          STA  LIB_NUMPAGES+0
          STA  LIB_NUMPAGES+1
#if !defined (PLATFORM_MYNOR) && !defined(PLATFORM_XS) && !defined(ALTERNATIVE_EEPROM_SEARCH)
          ;test if an EEPROM is in socket 1
          LD   R1,#0
          LDA  #0xA0
          JSR  eeprom_test
          JPF  _eepin01
          ;check if it is an 24LC512/24LC1025/24LC1026
          LD   LIB_EEADDRS+0,#0xA0
          LDA  #0xA8
          JSR  eeprom_test
          JPF  _eepin02
          ;it is an 24LC1025
          LD   LIB_EEADDRS+1,#0xA8
          LD   R1,#2
          JMP  _eepin01
_eepin02  ;check for 24LC1026
          LDA  #0xA2
          JSR  eeprom_test
          LD   R1,#1
          JPF  _eepin01  ;it is an 24LC512
          LD   LIB_EEADDRS+1,#0xA2
          LD   R1,#3     ;it is an 24LC1026
_eepin01  ;test socket 2
          LD   R2,#0
          LDA  #0xA6
          JSR  eeprom_test
          JPF  _eepin03
          ;check if it is an 24LC512/24LC1025/24LC1026
          LD   LIB_EEADDRS+2,#0xA6
          LDA  #0xAE
          JSR  eeprom_test
          JPF  _eepin04
          ;it is an 24LC1025
          LD   LIB_EEADDRS+3,#0xAE
          LD   R2,#2
          JMP  _eepin03
_eepin04  ;check for 24LC1026
          LDA  #0xA4
          JSR  eeprom_test
          LD   R2,#1
          JPF  _eepin03  ;it is an 24LC512
          LD   LIB_EEADDRS+2,#0xA4
          LD   LIB_EEADDRS+3,#0xA6
          LD   R2,#3     ;it is an 24LC1026
_eepin03  ;all detected. Tighten the list of EEPROMs:
          TST  LIB_EEADDRS+0
          JNF  _eepin07
          ;socket 1 is empty
          LD   LIB_EEADDRS+0,LIB_EEADDRS+2
          LD   LIB_EEADDRS+1,LIB_EEADDRS+3
          LD   LIB_EEADDRS+2,#0
          LD   LIB_EEADDRS+3,#0
          JMP  _eepin08
_eepin07  TST  LIB_EEADDRS+1
          JNF  _eepin08
          ;socket 1 has an 24LC512
          LD   LIB_EEADDRS+1,LIB_EEADDRS+2
          LD   LIB_EEADDRS+2,LIB_EEADDRS+3
          LD   LIB_EEADDRS+3,#0
#endif

_eepin08  ;count the just found EEPROMs
          LD   PTR_L,#LIB_EEADDRS
          LD   PTR_H,#REGPAGE_HI
#if !defined(ALTERNATIVE_EEPROM_SEARCH)
          DEC  LIB_NUMPAGES
_eepin10  INC  LIB_NUMPAGES
          JSR  lap_inc_ptr
          TST
          JNF  _eepin10
          JSR  dec_ptr
#endif
          ;now search for EEPROMs at the still untested addresses
          ;in the range 0xA0 - 0xAF
          ;PTR points to the end of the list of valid EEPROM addresses
          LD   R3,#0
          LD   R4,#0xA0
_eepin12  LDA  R4
          CMP  LIB_EEADDRS+0
          JPF  _eepin09
          CMP  LIB_EEADDRS+1
          JPF  _eepin09
          CMP  LIB_EEADDRS+2
          JPF  _eepin09
          CMP  LIB_EEADDRS+3
          JPF  _eepin09
          JSR  eeprom_test
          JPF  _eepin09
          LDA  R4
          JSR  sap_inc_ptr
          INC  LIB_NUMPAGES
          INC  R3
_eepin09  INC  R4
          INC  R4
          LDA  R4
          CMP  #0xB0
          JNF  _eepin12
          ;calculate overall size
          CLC
          LD   R0,#6
_eepin11  RWL  LIB_NUMPAGES
          JLP  _eepin11
#ifdef PLATFORM_XS
          JSR  i2c_finish
#endif
          ;print list of detected EEPROMs
          PRINT text_eeproms1
#if defined(PLATFORM_XS) || defined(ALTERNATIVE_EEPROM_SEARCH)
          LD   R4_L,LIB_NUMPAGES+0
          LD   R4_H,LIB_NUMPAGES+1
          JSR  print_decword
          PRINT text_eeproms3
          RTS
#else
#ifndef PLATFORM_MYNOR
          TST  R3
          JPF  _eepin13
#endif
          JSR  _eepin14
          PRINT text_eeproms3
          RTS
_eepin13  LD   R0,R1
          JSR  printEepromName
          LDA  #','
          JSR  print_char
          LD   R0,R2
          JSR  printEepromName
          LDA  #' '
          JSR  print_char
          LDA  #'('
          JSR  print_char
          JSR  _eepin14
          PRINT text_eeproms2
          RTS
_eepin14  LD   R4_L,LIB_NUMPAGES+0
          LD   R4_H,LIB_NUMPAGES+1
          JMP  print_decword
          

printEepromName:
          ;R0 = index to the list of EEPROM names
          ; IDs for EEPROMs:  1 = 24LC512, 2 = 24LC1025, 3 = 24LC1026
          PHL
          LD   PTR_L,#<text_eeplist
          LD   PTR_H,#>text_eeplist
          TST  R0
          JPF  _peen01
_peen02   JSR  lap_inc_ptr
          TST
          JNF  _peen02
          JLP  _peen02
_peen01   JSR  print_str
          RTS
#endif
          

fblockToEepromAddr:
          ;Convert a Forth Block Number to an
          ;EEPROM device- and memory address
          ;Input:  R4 = Forth Block Number
          ;Output: ACCU/LIB_EEPADDR = EEPROM device address, R4 = memory address, Flag = 0
          ;Changes: R0, R4, PTR
          LDA  R4_L
          SU   LIB_NUMPAGES
          LDA  R4_H
          SUB  LIB_NUMPAGES+1
          JPF  ret_opc
          LDA  #0
          LD   R0,#10
_fbtea01  RWL  R4
          ROL  ACCU
          JLP  _fbtea01
          AD   #LIB_EEADDRS
          STA  PTR_L
          LD   PTR_H,#REGPAGE_HI
          LAP
          STA  LIB_EEPADDR
          RET  ;return with flag = 0


eeprom_test:
          ; Test if an EEPROM is present.
          ; In : ACCU = EEPROM address
          ; Out: FLAG = 0 when eeprom is present
          PHL
          STA  R0
          PSH  R1
          JSR  i2c_start_addr
          JSR  i2c_stop_xs
          POP  R1
          RTS
         

eeprom_start_write:
          ;@API EEPROM
          ; Start a write-transfer.
          ; In : R4 : 16-bit EEPROM write address
          ; Out: FLAG = 0 on success
          ; Changes: ACCU, R0, R1
          PHL
          LD   R0,LIB_EEPADDR
          JSR  i2c_start_addr
          JPF  _eepstw2  ; error
_eepstw3  LD   R1,R4_H
          JSR  i2c_send
          JPF  _eepstw2  ; error
          LD   R1,R4_L
          JSR  i2c_send
          JNF  return
_eepstw2  JSR  i2c_stop
          SEC
          RTS


eeprom_start_write_ackpoll:
          ;@API EEPROM
          ; Start a write-transfer, but wait until
          ; previous write has been completed in the EEPROM.
          ; In : R4 : 16-bit EEPROM write address
          ; Out: FLAG = 0 on success
          ; Changes: ACCU, R0, R1
          PHL
_eepstw4  JSR  i2c_start
          JNF  return_flag_1
          LD   R1,LIB_EEPADDR
          JSR  i2c_send
          JNF  _eepstw3  ; ok, device is ready again
          JSR  i2c_stop_xs
          JMP  _eepstw4


eeprom_start_read:
          ;@API EEPROM
          ; Start a read-transfer.
          ; In : R4 : 16-bit EEPROM read address
          ; Out: R6/R7 contain prepared data for the I2C transfer
          ;      FLAG = 0 on success
          ; Changes: ACCU, R0, R1, R6, R7
          PHL
          JSR  eeprom_start_write
          JPF  return ;error
          LD   R0,LIB_EEPADDR
          INC  R0
          JSR  i2c_start_addr
          RTS          


eeprom_read_block:
          ;@API EEPROM
          ; Read a block of data into memory.
          ; In: PTR = ptr to destination memory
          ;     R5 = number of bytes to read (16 bit)
          ; Changes: ACCU, R0, R1, R5, PTR
          PHL
          LD   R1,#0
          TST  R5_L
          JNF  _eeprb04
          TST  R5_H
          JPF  return
          DEC  R5_H
_eeprb04  DEC  R5_L
          ;load n-1 bytes
_eeprb01  TST  R5_L
          JNF  _eeprb02
          TST  R5_H
          JPF  _eeprb03
          DEC  R5_H
_eeprb02  DEC  R5_L
          JSR  i2c_recv
          SAP
#ifdef FAST_PTR
          IPT
#else
          INC  PTR_L
          TST  PTR_L
          JNF  _eeprb01
          INC  PTR_H
#endif
          JMP  _eeprb01
_eeprb03  ;last byte
          LD   R1,#1
          JSR  i2c_recv
          JSR  sap_inc_ptr
          RTS


eeprom_write_block:
          ; Write a block of data to EEPROM.
          ; NOTE: Do not write over page-boundaries of 128 byte!
          ;       After calling this function (or when the end
          ;       of a page was reached), eeprom_stop must be called!
          ; In : PTR = ptr to source memory
          ;      R2  = number of bytes to write (max 128!)
          ; Out: FLAG = 0 on success
          ; Changes: ACCU, R0, R1, R2, PTR
          PHL
_eepwb01  LAP
#ifdef FAST_PTR
          IPT
#else
          INC  PTR_L
          TST  PTR_L
          JNF  _eepwb02
          INC  PTR_H
#endif
_eepwb02  STA  R1
          JSR  i2c_send
          JPF  return  ; error
          DEC  R2
          TST  R2
          JNF  _eepwb01
          CLC
          RTS


eeprom_stop:
          ;@API EEPROM
          ; Stop a EEPROM transaction
          ; Changes: ACCU
          JMP  i2c_stop


eeprom_stop_ret:
          JSR  eeprom_stop
          RTS


eeprom_read_forth_block:
          ;Read a Forth block from EEPROM to memory
          ;Input:  R4 = Forth Block Number, R5 = memory destination address
          ;Output: FLAG = 0 on success
          ;Changes: All registers
          PHL
          JSR  fblockToEepromAddr
          JPF  return
          JSR  eeprom_start_read
          JPF  return
          JSR  ld_ptr_r5
          LD   R5_L,#0
          LD   R5_H,#4
          JSR  eeprom_read_block
          JSR  eeprom_stop
          CLC
          RTS


eeprom_write_forth_block:
          ;Write a Forth block from memory to EEPROM
          ;Input:  R4 = Forth Block Number, R5 = memory source address
          ;Output: FLAG = 0 on success
          ;Changes: All registers
          PHL
          JSR  fblockToEepromAddr
          JPF  _ewfb03
          JSR  ld_ptr_r5
          LD   R5_L,#8   ; 8 blocks of 128 bytes = 1024 bytes to write
          ;R4 = 16-bit EEPROM address
_ewfb01   JSR  eeprom_start_write_ackpoll
          JPF  _ewfb03  ; error
          LD   R2,#128
          JSR  eeprom_write_block
          PSH  FLAG
          JSR  i2c_stop_xs
          POP  FLAG
          JPF  _ewfb03  ; error
          LDA  #128
          AD   R4_L
          STA  R4_L
          JNF  _ewfb02
          INC  R4_H
_ewfb02   DEC  R5_L
          TST  R5_L
          JNF  _ewfb01
          CLC
_ewfb03
#ifdef PLATFORM_XS
          JSR  i2c_finish
#endif
          RTS

SEG_PRGCODE
