;------------------------------------------------------------------------------
; This is a simple Forth implementation for My4TH
;------------------------------------------------------------------------------
; This software is provided "as is". It is licensed to you under the
; Creative Commons Attribution-ShareAlike 4.0 International License.
; Written in 2022 by Dennis Kuschel, dennis_k@freenet.de, www.mynor.org
;------------------------------------------------------------------------------

SEG_LOWCODE

;Constants

#if defined(PLATFORM_MYNOR) || defined(PLATFORM_XS)  ; check if system has only 8kB RAM
LPG           SET 0x9000        ; start address of last 4kB memory page: MyNOR has only 8kB RAM
#else
LPG           SET 0xF000        ; start address of last 4kB memory page
#endif

FORTH_START   SET 0x8400        ; Start of forth memory region. The region 0x8400 to 0x842E is used by the Forth binary program header.
RSVD_BYTE     SET 0x842F        ; 1 unused byte
HEAP_START    SET 0x8430        ; start of free memory (for the dictionary)
HEAP_END      SET LPG+0x0CFF    ; last byte of the heap
STACK_SPLIT   SET 0x80          ; The lower half of the call stack is used for the data stack.
                                ; That means, the data stack has a size of 128 bytes, like the call stack.

DATA_STACK_SZ SET STACK_SPLIT   ; size of the data stack
DATA_STACK    SET 0x8100        ; start of the data stack. It grows upwards.
                                ; the data stack is at 0x8100 - 0x817F, the callstack is at 0x8180 - 0x81FF
USER_MEMORY   SET LPG+0x0D00    ; 256 bytes of memory for the USER
EDIT_STRUCT   SET LPG+0x0E00    ; 6 bytes for a data structure used by the EDITor
SYSDATASTACK  SET LPG+0x0E06    ; 42 bytes for the system data stack
SYS_BUF       SET LPG+0x0E30    ; 112 bytes for forth system internal use (ACCEPT word and EDITor), also used for NUM_BUF and WORD_BUF
PAD_MEMORY    SET LPG+0x0EA0    ; 96 bytes for the PAD transient area (must be at least 84 bytes)
PAD_SIZE      SET 0x60
EVALUATE_BUF  SET LPG+0x0F00    ; 128 bytes for the EVALUATE word (to store a zero-terminated string)
CCTLF_STK_SZ  SET 0x40
CCTLFLOWSTACK SET LPG+0x0F80    ; 32 words for the control flow stack for conditional jumps (allows up to 32 nested IFs or other constructs)
LCTLF_STK_SZ  SET 0x40
LCTLFLOWSTACK SET LPG+0x0FC0    ; 32 words for the loop control stack (allows up to 16 nested loops)
RETURNSTACK   SET CCTLFLOWSTACK ; the return stack shares the space with the control flow stack, but it increases from bottom to top

NUM_BUF       SET SYS_BUF       ; use system buffer also for pictured numeric output string
NUMBUF_SZ     SET 112           ; size of the system buffer

WORD_BUF      SET SYS_BUF+64    ; transient region for the word "WORD"

FTH_HDR_FLAG  SET 0x14          ; file header flag for FORTH memory images saved via "save-forth"
BIN_HDR_FLAG  SET 0x1B          ; file header flag for binary driver images
FL_IMMEDIATE  SET 0x80          ; marks a word as "IMMEDIATE", must be bit 7 so it is faster to evaluate


;------------------------------------------------------------------------------
; Data Stack Definitions
;------------------------------------------------------------------------------

#ifdef FAST_DSTACK
  #define JSR__push_data_R4         PHD  R4
  #define JSR__push_data_R5         PHD  R5
  #define JSR__pop_data_R4          PLD  R4
  #define JSR__pop_data_R5          PLD  R5
  #define JSR__pop_data_R4_R5       PLD2 R4,R5
  #define JSR__pop_data_R5_R4       PLD2 R5,R4
  #define JMP__push_data_R4_ret     PHDX R4,OPCODE_RTS
  #define JMP__push_data_R5_ret     PHDX R5,OPCODE_RTS
  #define JMP__push_data_R4_R5_ret  PHD2X R4,R5,OPCODE_RTS
  #define JMP__push_data_R4         PHDX R4,OPCODE_RET
  
  #define JSR__push_data_PTR		PHD  PTR_L
  #define JMP__push_data_PTR		PHDX PTR_L,OPCODE_RET
  #define JMP__push_data_PTR_ret	PHDX PTR_L,OPCODE_RTS 
  #define JMP__push_data_PTR_R5_ret PHD2X PTR_L,R5,OPCODE_RTS   
#else
  #define JSR__push_data_R4         JSR  push_data_R4
  #define JSR__push_data_R5         JSR  push_data_R5
  #define JSR__pop_data_R4          JSR  pop_data_R4
  #define JSR__pop_data_R5          JSR  pop_data_R5
  #define JSR__pop_data_R4_R5       JSR  pop_data_R4_R5
  #define JSR__pop_data_R5_R4       JSR  pop_data_R5_R4
  #define JMP__push_data_R4_ret     JMP  push_data_R4_ret
  #define JMP__push_data_R5_ret     JMP  push_data_R5_ret
  #define JMP__push_data_R4_R5_ret  JMP  push_data_R4_R5_ret
  #define JMP__push_data_R4         JMP  push_data_R4

  ;if no FAST_STACK, add routines for push_data_PTR
  
#endif


;------------------------------------------------------------------------------
; The "main program":
;------------------------------------------------------------------------------


init_vectors:
            ;initialize vectors and a first set of Forth variables
            ;this routine gets called when MyNOR is reset
            LDA  #OPCODE_JMP
            STA  VECT_INPUT+0
            STA  VECT_OUTPUT+0
            STA  VECT_OK+0
            STA  VECT_PROMPT+0
            STA  VECT_INPSTR+0
            STA  VECT_GETNBR+0
            LDA  #0
            STA  VECT_INPUT+1
            STA  VECT_INPUT+2
            STA  VECT_OUTPUT+1
            STA  VECT_OUTPUT+2
            LD   VECT_OK+1,#<output_ok
            LD   VECT_OK+2,#>output_ok
            LD   VECT_PROMPT+1,#<_prompt
            LD   VECT_PROMPT+2,#>_prompt
            LD   VECT_INPSTR+1,#<input_str_loop
            LD   VECT_INPSTR+2,#>input_str_loop
            LD   VECT_GETNBR+1,#<get_number_fn
            LD   VECT_GETNBR+2,#>get_number_fn
            RET


set_vector:
            ;Set a vector to a new address
            ;In: accu = vector number, R4 = new address
            PHL
            JSR  getVectAdr
            LDA  R4_L
            JSR__sap_inc_ptr
            LDA  R4_H
            SAP
            RTS

get_vector:
            ;Get the current address of a vector
            ;In: accu = vector number, Out: R4 = address
            PHL
            JSR  getVectAdr
            JSR  lap_inc_ptr
            STA  R4_L
            LAP
            STA  R4_H
            RTS

getVectAdr:
            STA  PAR1
            AD   PAR1
            AD   PAR1
            AD   #VECTOR1+1
            STA  PTR_L
            LD   PTR_H,#REGPAGE_HI
            RET


init_forth_1:
            ;initialize the Forth system, part 1
            LD   R4_L,#<tab_fthVarInit
            LD   R4_H,#>tab_fthVarInit
            LD   R5_L,#<VAR_INIT_START
            LD   R5_H,#REGPAGE_HI
            LD   R6_L,#VAR_INIT_END-VAR_INIT_START
            JMP  memcpy_short


init_forth_2:
            ;initialize the Forth system, part 2
            LD   SP,#STACK_SPLIT
            LD   SYSSP,#<SYSDATASTACK                  ; stack grows from bottom to top
            LD   fPSP,#<DATA_STACK                     ; stack grows from bottom to top
            LD   RETSP,#<RETURNSTACK                   ; stack grows from bottom to top
            LD   CCFSP,#<(CCTLFLOWSTACK+CCTLF_STK_SZ)  ; stack grows from top to bottom
            LD   LCFSP,#<(LCTLFLOWSTACK+LCTLF_STK_SZ)  ; stack grows from top to bottom
            LDA  #0
            STA  STATE+0
            STA  STATE+1
            STA  NEXTINCHR
#ifndef NO_BLOCK_WORDS
            STA  BLK_L
            STA  BLK_H
#endif
            RET


forth_start:
#ifdef INCLUDE_FLOAT_WORDS
            JSR  init_float_words
#endif
#ifdef INCLUDE_IOBOARD_WORDS
            JSR  ioboards_init
#endif

            ;Start the FORTH system (with autostart of a FORTH program)
            JSR  init_vectors
            JSR  init_forth_1
            JSR  init_forth_2

#ifndef NO_BLOCK_WORDS
            ;read screen 0, and get the autostart block number
            LD   R4_L,#0
            LD   R4_H,#0
            JSR  fblockToEepromAddr
            JPF  _fosta01
            JSR  eeprom_start_read
            JPF  _fosta01
            LD   R4_L,#0
            LD   R4_H,#0
            LD   R2,#5
_fosta02    LD   R1,#0
            JSR  i2c_recv
            JSR  is_digit
            JPF  _fosta03
            JSR  mul10
            LDA  #0x0F
            AND  R0
            AD   R4_L
            STA  R4_L
            JNF  _fosta04
            INC  R4_H
_fosta04    DEC  R2
            TST  R2
            JNF  _fosta02
            ;number too long
_fosta06    JSR  stop_i2cread
            JMP  _fosta01
_fosta03    ;finished with reading the autostart block number
            CMP  #0x20
            JNF  _fosta06
            JSR  stop_i2cread
            TST  R4_L
            JNF  _fosta07
            TST  R4_H
            JPF  _fosta01
_fosta07    ;print autostart message, wait if user interrupts autostart
            PRINT text_autostart1
#ifdef NO_LCD
AUTOWAITTM  SET  42*CPUCLOCK
#else
AUTOWAITTM  SET  37*CPUCLOCK
#endif
            LD   R5_L,#<(0xFFFF-AUTOWAITTM)
            LD   R5_H,#>(0xFFFF-AUTOWAITTM)
_fosta08    JSR  test_for_io_activity
            JNF  _fosta01
            JSR  inc_r5
            TST  R5_L
            JNF  _fosta08
            TST  R5_H
            JNF  _fosta08
            ;load the autostart block
            PRINT text_autostart2
            JSR  print_decword
            JSR  print_nl
            JSR__push_data_R4
            JSR  c_load
            JMP  abort_restart
_fosta01    ;no autostart configured, start the FORTH prompt
#endif

restart_forth:
            ;this is like a hard-reset of FORTH
            LD   SP,#STACK_SPLIT
            JSR  init_vectors
            PRINT text_heading1
#ifndef PLATFORM_XS
            LD   R4_L,#CPUCLOCK
            LD   R4_H,#0
            JSR  print_decword
            PRINT text_heading2
#endif

forth_reinit:
            JSR  init_forth_1

abort_restart:
#ifndef NO_I2C
            JSR  stop_eeprom_transfer
#endif
            JSR  init_forth_2
            LDA  INPUTDEV
            CMP  #INDEV_REMOTE
            JNF  prompt
#ifndef NO_BLOCK_WORDS
            JSR  txferr_errorback
#endif

prompt:     ;this is the input prompt
            LD   RETSP,#<RETURNSTACK
            LD   INPUTDEV,#INDEV_LOCAL ;required because this prevents sending also COK after CERR over UART
            LD   R0,#TIBUF_SZ
            LD   R4_L,#<TIBUF
            LD   R4_H,#>TIBUF
            JSR  PROMPT_VECTOR  ; usually calls _prompt
            JMP  prompt
_prompt     ;the Forth promt (R0 is the max. length of the input buffer
            ;including the terminating zero, and R4 points to the input buffer)
            LD   RETSP,#<RETURNSTACK
            JSR  input_string   ; this function may not return, but will jump to execRemoteString directly
            JSR  print_space
            JSR  ld_ptr_tibuf
            LDA  #INDEV_LOCAL
            JSR  input_parser
            JMP  prompt

#ifndef NO_BLOCK_WORDS
execRemoteString:
            ;execute a string that was received via UART
            ;PTR must point to the string
            LD   SP,#STACK_SPLIT
            LDA  #INDEV_REMOTE
            JSR  input_parser
            JSR  txferr_okback
#endif
forth_restart_prompt:
            LD   SP,#STACK_SPLIT
            JMP  prompt



;------------------------------------------------------------------------------
;  Forth program header in memory at 0x8400 (FORTH_START):
;
;   0x8400 : 0x14  - marking for Forth program (FTH_HDR_FLAG)
;   0x8401 : rver  - ROM version number
;   0x8402 : rsvd  - reserved byte
;   0x8403 : rsvd  - reserved byte
;   0x8404 : 6 system vectors
;   0x8416 : here  - pointer (2 bytes)
;   0x8418 : 8 word pointers for dictionary
;   0x8428 : UP user pointer
;   0x842A : BLK structure
;   0x8430 : start of dictionary
;
;   0x8461 - 0x8464 : In case of the Forth Deck driver, this bytes
;                     contain the magic numbers 0x18,0x2F,0x0E,0x18
;                     and address 0x8465-0x8466 is a pointer to the
;                     module initialization routine.
;
;------------------------------------------------------------------------------
#ifndef NO_BLOCK_WORDS

prepare_forth_program:
            ;prepare to start a forth program
            ;returns with FLAG=0 on success
            PHL
            LD   PTR_L,#<FORTH_START
            LD   PTR_H,#>FORTH_START
            JSR  lap_inc_ptr
            CMP  #FTH_HDR_FLAG
            JNF  return_flag_1
            LAP
            CMP  #ROMVERSION_BINARY^PFFLAG
            JPF  _fexf02
_fexf01     ;wrong forth/rom version
            PRINT text_wrongver
            JMP  return_flag_1
_fexf02     ;initialize the Forth environment
            JSR  pop_r7  ;save return address to R7
            JSR  init_forth_1
            JSR  init_forth_2
            JSR  psh_r7
            LD   R4_L,#<(FORTH_START+4)
            LD   R4_H,#>(FORTH_START+4)
            LD   R5_L,#PRGHEADER
            LD   R5_H,#REGPAGE_HI
            LD   R6_L,#PRGHDR_SIZE
            JSR  memcpy_short
            ;initialize the image, call the init-routine
            ;if a magic number is present
            LD   PTR_L,#<0x8461
            LD   PTR_H,#>0x8461
            JSR  lap_inc_ptr
            CMP  #0x18
            JNF  _fexf03
            JSR  lap_inc_ptr
            CMP  #0x2F
            JNF  _fexf03
            JSR  lap_inc_ptr
            CMP  #0x0E
            JNF  _fexf03
            JSR  lap_inc_ptr
            CMP  #0x18
            JNF  _fexf03
            JSR  0x8464
_fexf03     JMP  return_flag_0


do_run:
            ;find and execute run word
            LD   INPTR_L,#<text_run
            LD   INPTR_H,#>text_run
            LD   R1,#3
            JSR  search_word
            JPF  _doru01
            JSR  exec_word
            LDA  #<prompt
            PSH
            LDA  #>prompt
            PSH
            JMP  _retOk
_doru01     PRINT text_runerr
            JMP  abort_restart


load_line_into_filebuf:
            ;load next line from EEPROM into filebuf
            ;input: BLK = screen number, FILE_LNBR = line number
            PHL
load_line_into_filebuf_rts:
            LD   R4_L,BLK_L
            LD   R4_H,BLK_H
            JSR__push_data_R4
            JSR  prep_eepblk
            ;R4 += 64 * FILE_LNBR
            LD   R7_L,FILE_LNBR
            LD   R7_H,#0
            LD   R0,#6
            CLC
_lsf02      RWL  R7_L
            JLP  _lsf02
            LDA  R4_L
            AD   R7_L
            STA  R4_L
            LDA  R4_H
            ADD  R7_H
            STA  R4_H
            ;R3 = I2C device address
            ;R4 = EEPROM memory address
            JSR  feep_start_read
            ; Load a text line (64 bytes) from EEPROM and interpret it
            LD   R5_L,#64
            LD   R5_H,#0
            LD   EEP_STATE,#1
            JSR  ld_ptr_filebuf
            LD   R2,#0xE0
            LD   R1,#0
            JMP  _lsf09
            ;receive a byte
_lsf07      DEC  R5_H
_lsf01      DEC  R5_L
            TST  R5_L
            JNF  _lsf11
            TST  R5_H
            JNF  _lsf11
            LD   R1,#1
_lsf11      JSR  i2c_recv
            STA  R0
            AND  R2
            TST
            LDA  R0
            JNF  _lsf08
            LDA  #0x20
_lsf08      JSR__sap_inc_ptr
_lsf09      TST  R5_L
            JNF  _lsf01
            TST  R5_H
            JNF  _lsf07
            ;end of line / end of file
            LDA  #0
            SAP
            JSR  eeprom_stop
            LD   EEP_STATE,#0
            RTS


interpret_filebuf_line:
            ;parse the line
            PHL
            JSR  ld_ptr_filebuf
            JSR  skip_space
            LAP
            TST
            JPF  _lsf10
            LDA  #INDEV_FILE
            JSR  input_parser
_lsf10      RTS


stop_i2cread:   ;stop I2C read transfer
            PHL
            LD   R1,#1
            JSR  i2c_recv
            JMP  eeprom_stop_ret


stop_eeprom_transfer:
            TST  EEP_STATE
            JPF  ret_opc
recoverI2Cbus:
            ;recover EEPROM
            PHL
            ;send a pattern that recovers the I2C-bus
            JSR  stop_i2cread
            JSR  i2c_start
            LD   EEP_STATE,#0
            JMP  eeprom_stop_ret
#endif


ld_ptr_tibuf:
            LD   PTR_L,#<TIBUF
            LD   PTR_H,#>TIBUF
            RET

#ifndef NO_BLOCK_WORDS
ld_ptr_filebuf:
            LD   PTR_L,#<FILE_LINEBUF
            LD   PTR_H,#>FILE_LINEBUF
            RET
#endif


;------------------------------------------------------------------------------
;  Input Parser
;------------------------------------------------------------------------------

evaluate:   ;Evaluate a zero-terminated string pointed by INPTR.
            LD  PTR_L,INPTR_L
            LD  PTR_H,INPTR_H

input_parser:
            ;Input:  PTR is the ptr to the input stream buffer.
            ;        The buffer must be terminated with zero (CR not allowed in the buffer).
            ;        ACCU must be set to the current input device
            PHL
            STA  INPUTDEV
_itp11      JSR  ld_inptr_ptr
            LD   REFILL,#0
itp_loop:
_itp01      TST  REFILL  ; quit early when REFILL was flagged for a screen/load
            JNF  _itp04
            ;loop over all words in the input buffer and interpret them
            ;skip leading space characters
            JSR  skipSpaces
            JSR  get_word_len
            TST  R1
            JPF  _itp04

            ;test for special "words" with a length of one character
            LDA  #1
            CMP  R1
            JNF  _itp06
            JSR  ld_ptr_inptr
            LAP
            CMP  #'('
            JPF  _itp10  ;begin of comment
            CMP  #'\\'
            JPF  _itp04  ;drop the end of the line

_itp06      TST  STATE
            JNF  _itp09  ;compile
            ;interpret
            JSR  search_word
            JNF  _itp16
            ;read a number
            JSR  get_number
            JPF  Error_Word
            JSR__push_data_R4
            TST  R3
            JPF  _itp01  ;continue with next word
            JSR__push_data_R5
            JMP  _itp01  ;continue with next word
            ;execute the word and continue with next word
_itp16      LD   IPRSPACE+1,PTR_L
            LD   IPRSPACE+2,PTR_H
            JSR  skipSpace
            ;execute the word and jump to itp_loop
            JMP  IPRSPACE_JMP

_itp09      JSR  memcheck
            ;try to find the word in the dictionary
            JSR  search_word
            JPF  _itp05 ;word not found
            ;word found, is it marked "immediate"?
            LDA  #FL_IMMEDIATE
            AND  R2
            TST
            JNF  _itp14
            ;word found, store a reference to it in memory
            JSR  add_word_call
            JMP  _itp02  ;continue with next word
_itp05      ;try to interpret the input as number
            JSR  get_number
            JPF  Error_Word
            ;number read, store it as code in memory
            TST  R3
            JNF  _itp15
            JSR  add_16bit_literal
            JMP  _itp02  ;continue with next word
_itp15      JSR__push_data_R4
            JSR__push_data_R5
            JSR  popR4R5_add_32bit_literal
            JMP  _itp02

_itp14      ;execute the "immediate" word
            JSR  exec_word
_itp02      JSR  memcheck
            JMP  _itp01  ;continue with next word

_itp10      ;skip comment until next ')' in the input buffer
            JSR__inc_ptr_lap
            TST
            JPF  Error_Word
            CMP  #')'
            JNF  _itp10
            JSR__inc_ptr
            JMP  _itp11

_itp04      ;end of buffer reached
            LDA  INPTR_H
            CMP  #>EVALUATE_BUF
            JPF  return_flag_1 ; the evaluate-word does not print OK
            TST  STATE
            JPF  _retOk
            ;return with message "compiled"
            PRINT text_compiled
            JMP  return_flag_0
printOkStatus:
            PHL
_retOk      ;return with status "ok"
#if defined(PLATFORM_XS) && !defined(NO_I2C)
            JSR  uart_unlock  ; enable printing although I2C may still be busy
#endif
            JSR  OK_VECTOR
            JMP  return_flag_1


skipSpace:  ;Skip one space character in the input buffer
            PHL
            JSR  ld_ptr_inptr
            LAP
            CMP  #0x20
            JNF  return
            JSR__inc_ptr
            JSR  ld_inptr_ptr
            RTS

skipSpaces:
            ;Skip space characters in the input buffer
            ;returns: the character that follows the spaces in ACCU,
            ;         PTR pointing to position after the spaces
            LD   R5_L,#0x20
            
            ; Skip delimiters in R5_L
skipDelimiter:	
            LD   PTR_L,INPTR_L
            LD   PTR_H,INPTR_H
_sks01      LAP
            ;CMP  #0x20
            CMP  R5_L
            JNF  ld_inptr_ptr
            INC  PTR_L
            TST  PTR_L
            JNF  _sks01
            INC  PTR_H
            JMP  _sks01


ld_r4_ptr:  ;load R4 with PTR
            LD   R4_L,PTR_L
            LD   R4_H,PTR_H
            RET


ld_inptr_ptr:
            LD   INPTR_L,PTR_L
            LD   INPTR_H,PTR_H
            RET


ld_ptr_inptr:  ;load PTR with INPTR
            LD   PTR_L,INPTR_L
            LD   PTR_H,INPTR_H
            RET


push_data_inptr:  ;push INPTR to data stack
#ifdef FAST_DSTACK
            PHD  INPTR_L
            RET
#else
            LD   R4_L,INPTR_L
            LD   R4_H,INPTR_H
            JMP__push_data_R4
#endif


push_inptr: ;push the input ptr to the system data stack
            LD   PTR_L,#INPTR_L

push_zpvar: ;push zero-page variable to the system data stack
            ;In: PTR_L pointer to variable
            PHL
            LD   PTR_H,#REGPAGE_HI
            LAP
            STA  PAR1
            INC  PTR_L
            LAP
            JSR  syss_push
            LDA  PAR1
            JSR  syss_push
            RTS


pop_inptr:  ;push the input ptr to the system data stack
            LD   PTR_L,#INPTR_L

pop_zpvar:  ;pop zero-page variable from the system data stack
            ;In: PTR_L pointer to variable
            PHL
pop_zpvar_rts:
            PSH  PTR_L
            JSR  syss_pop
            STA  PAR1
            JSR  syss_pop
            POP  PTR_L
            LD   PTR_H,#REGPAGE_HI
            INC  PTR_L
            SAP
            DEC  PTR_L
            LDA  PAR1
            SAP
            RTS


exec_word:
            ;execute the word that is referenced by PTR
            PHL
            LD   JSRSPACE+1,PTR_L
            LD   JSRSPACE+2,PTR_H
            ;skip one space character
            JSR  skipSpace
            ;execute the word and return
            JMP  JSRSPACE_JMP


Error_Word:      ;the word pointed by INPTR is unknown
#ifdef WITH_I2C_LCD_KYBD   ;Reset I2C bus before print error messages
            JSR  i2c_init
#endif   
#if defined(PLATFORM_XS) && !defined(NO_I2C)
            JSR  uart_unlock  ; enable printing although I2C may still be busy
#endif
            PRINT text_errWord
            JSR  printNextWord
            JMP  _err_prnl

memcheck:   ;check if free memory is still available, quit with an error message if not
            PSH
            LDA  #>(HEAP_END+1)
            CMP  CP_H
            POP
            JNF  ret_opc
Error_Mem:       ;out of memory error
            LDA  #<text_errMem
            DB   OPCODE_LDI  ; trick: jumps over next two bytes, but overwrites register TEMP5
Error_Number:
            LDA  #<text_errNumber
            DB   OPCODE_LDI  ; trick: jumps over next two bytes, but overwrites register TEMP5
Error_Syntax:    ;wrong syntax
            LDA  #<text_errSyntax
            DB   OPCODE_LDI  ; trick: jumps over next two bytes, but overwrites register TEMP5
Error_Value:     ;value error
            LDA  #<text_errValue
            DB   OPCODE_LDI  ; trick: jumps over next two bytes, but overwrites register TEMP5
Error_Stack:     ;stack over- or underflow
            LDA  #<text_errStack
            DB   OPCODE_LDI  ; trick: jumps over next two bytes, but overwrites register TEMP5
Error_ToLong:    ;word name too long
            LDA  #<text_errTooLong
            DB   OPCODE_LDI  ; trick: jumps over next two bytes, but overwrites register TEMP5
#ifdef WITH_I2C_LCD_KYBD            
Error_UnavailforLcd: ;unavailable for LCD
            LDA  #<text_errUnavailforLcd
            DB   OPCODE_LDI  ; trick: jumps over next two bytes, but overwrites register TEMP5
#endif            
Error_Quote:     ;return with error, the " is missing
            LDA  #<text_errQuote
_pr_err     STA  PTR_L
            LD   PTR_H,#>forth_strings_start
#ifdef WITH_I2C_LCD_KYBD   ;Reset I2C bus before print error messages
            JSR  i2c_init
#endif            
#if defined(PLATFORM_XS) && !defined(NO_I2C)
            JSR  uart_unlock  ; enable printing although I2C may still be busy
#endif
            JSR  print_str
_err_prnl   JSR  print_nl
            JMP  abort_restart


#ifndef NO_BLOCK_WORDS
Error_InOut:     ;I/O Error
#ifndef NO_I2C
            JSR  recoverI2Cbus
#endif
            LDA  #<text_errInOut
            JMP  _pr_err
#endif


printNextWord:
            ;print the next word stored in the input buffer
            PHL
            JSR  get_word_len
            JSR  ld_ptr_inptr
_pnw01      TST  R1
            JPF  return
            JSR  lap_inc_ptr
            JSR  print_char
            DEC  R1
            JMP  _pnw01

output_ok:  ;print the "OK" status
            LD   PTR_L,#<text_forth_ok
            LD   PTR_H,#>text_forth_ok
            JMP  print_str



;------------------------------------------------------------------------------
;  Read Numbers
;------------------------------------------------------------------------------

get_number:
            ;Tries to read a number from the input buffer.
            ;Input:   INPTR points to the possible number in the input buffer.
            ;         R1 must contain the length of the word.
            ;         The number must be terminated by SPACE, ';' or NULL
            ;Output:  If a number could be read, FLAG is set to 0
            ;         R3 is 0 for 16-bit and FF for 32-bit,
            ;         R4/R5 contains the read 16/32-bit number, and
            ;         INPTR points behind the word
            ;Changes: ACCU,R0,R1,R2,R4,R5,PTR,FLAG
            PHL
            JSR  ld_ptr_inptr
            JSR  GETNBR_VECTOR
            JPF  return
_rdnb06     JSR  ld_inptr_ptr
            RTS

#define NBR_UNDERSCORE  ; allow underscores within numbers (new in v1.4)

get_number_fn:
            PHL
            ;check for base prefix, set desired base
            LAP
            CMP  #'#'
            JPF  _rdn_b10
            CMP  #'$'
            JPF  _rdn_b16
            CMP  #'%'
            JNF  _rdnb07
_rdn_b2     LD   R2,#2
            JMP  _rdnb08
_rdn_b10    LD   R2,#10
            JMP  _rdnb08
_rdn_b16    LD   R2,#16
_rdnb08     PSH  BASE     ; save old base
            LD   BASE,R2  ; set new base
            JSR__inc_ptr  ; skip number prefix character
            DEC  R1
            TST  R1
            JPF  _rdnb10  ; error, no characters left
            JSR  _rdnb09  ; call read-number routine
_rdnb10     POP  BASE     ; restore old base
            RTS
_rdnb09     PHL
_rdnb07     ;check if the character after the possible number is a dot (its a mark for a 32-bit number)
            JSR  psh_ptr
            LDA  R1
            DEC
            AD   PTR_L
            STA  PTR_L
            JNF  _rdnb05
            INC  PTR_H
_rdnb05     LAP
            JSR  pop_ptr
            CMP  #'.'
            JNF  _rdnb01
            ;read a 32-bit number with respect to BASE from the input buffer
            DEC  R1  ;remove the dot at the end
            LD   R3,#0xFF
            JMP  _gn32b04
_rdnb01
#ifdef INCLUDE_FLOAT_WORDS
            ;check the BASE
            LDA  #10
            CMP  BASE
            JNF  _rdnb02
            JSR  string_to_float
            JNF  rts_opc
_rdnb02
#endif
            ;read a 16-bit number with respect to BASE from the input buffer
            LD   R3,#0
_gn32b04    ;get first digit
            JSR  lap_inc_ptr
            LD   R2,#0   ; sign
            CMP  #'-'
            JNF  _gn32b01
            INC  R2      ; mark number as negative
            DEC  R1
;            TST  R1
;            JPF  return  ; error, sign but no number  (can't happen because there exists the "-" word)
            JSR  lap_inc_ptr
_gn32b01    JSR  char2binnum
            JNF  return_flag_1  ; error, no valid digit for current BASE
            STA  R4_L
            LDA  #0
            STA  R4_H
            STA  R5_L
            STA  R5_H
_gn32b03    DEC  R1
            TST  R1
            JPF  _gn32b02
            ;multiply number in R4/R5 with BASE
            PSH  R6_L
            PSH  R6_H
            LD   R6_L,BASE+0
            LD   R6_H,BASE+1
            JSR  mul32u
            POP  R6_H
            POP  R6_L
            ;add next digit
_gn32b08    LAP
            JSR  char2binnum
            JPF  _gn32b07
            ;no valid number character, now check for an underscore
            LAP
            CMP  #'_'
            JNF  return_flag_1  ; error, no underscore
            JSR__inc_ptr
            DEC  R1
            TST  R1
            JPF  return_flag_1  ; error, no digit following the underscore            
            JMP  _gn32b08
_gn32b07    ;add this digit to the number
            JSR__inc_ptr
            AD   R4_L
            STA  R4_L
            JNF  _gn32b03
            INC  R4_H
            TST  R4_H
            JNF  _gn32b03
            INC  R5_L
            TST  R5_L
            JNF  _gn32b03
            INC  R5_H
            JMP  _gn32b03
_gn32b02    ;return a 16 or 32-bit number in R4 and R5
            TST  R3
            JPF  _gn32b06
            JSR__inc_ptr ;skip the dot
            JMP  _gn32b05
_gn32b06    TST  R5_L
            JNF  Error_Number
            TST  R5_H
            JNF  Error_Number
_gn32b05    TST  R2
            JPF  return_flag_0
            JSR  negate_R45
            JMP  return_flag_0


char2binnum:
            ;Convert a character (0-9,A-Z) to a binary number (0-35)
            ;Input : Accu = character
            ;Output: Accu = binary number, FLAG=1 on success
            PHL
            JSR  is_digit
            JNF  _ch2binn01
            JSR  is_alpha
            JPF  return_flag_0  ;FLAG=0, error
            AND  #0x1F
            ADD  #9
            JMP  _ch2binn02
_ch2binn01  AND  #0x0F
_ch2binn02  STA  R0
            LDA  BASE
            ;CLC
            SUB  R0
            LDA  R0
            RTS        ; OK when FLAG = 1
 
SEG_PRGCODE
