;------------------------------------------------------------------------------
; Math functions for Forth
;------------------------------------------------------------------------------

SEG_LOWCODE

negate_r4:  ; Negate the 16-bit number in R4 (two's complement).
            ; In : R4 = 16-bit signed number
            ; Out: R4 = negated 16-bit signed number
            ; Changes: ACCU, R4
            LDA  #0
            SU   R4_L
            STA  R4_L
            LDA  #0
            SUB  R4_H
            STA  R4_H
            RET


negate_r5:  ; Negate the 16-bit number R5 (two's complement).
            ; In : R5 = 16-bit signed number
            ; Out: R5 = negated 16-bit signed number
            ; Changes: ACCU, R4
            LDA  #0
            SU   R5_L
            STA  R5_L
            LDA  #0
            SUB  R5_H
            STA  R5_H
            RET


negate_R45:
            ;negate the 32-bit number in R4 and R5
            PHL
_negR4R5    JSR  inv_r4
            JSR  inv_r5
            INC  R4_L
            TST  R4_L
            JNF  return_flag_1
            INC  R4_H
            TST  R4_H
            JNF  return_flag_1
            INC  R5_L
            TST  R5_L
            JNF  return_flag_1
            INC  R5_H
            RTS


inv_r4:     ;bit-invert R4
            LD   PAR1,#0xFF
            LDA  R4_L
            XOR  PAR1
            STA  R4_L
            LDA  R4_H
            XOR  PAR1
            STA  R4_H
            RET


inv_r5:     ;bit-invert R5
            LD   PAR1,#0xFF
            LDA  R5_L
            XOR  PAR1
            STA  R5_L
            LDA  R5_H
            XOR  PAR1
            STA  R5_H
            RET


inv_r6:     ;bit-invert R6
            LD   PAR1,#0xFF
            LDA  R6_L
            XOR  PAR1
            STA  R6_L
            LDA  R6_H
            XOR  PAR1
            STA  R6_H
            RET


dec_r4:     ; Decrement R4 register by one
            TST  R4_L
            JNF  _decr41
            DEC  R4_H
_decr41     DEC  R4_L
            RET


mul10:      ; Multiply an unsigned 16-bit number with 10.
            ; In : R4 = 16-bit unsigned number
            ; Out: R4 = R4 * 10, FLAG = 0 (FLAG=1:ERROR)
            ; Changes: ACCU, R4
            CLC
            RWL  R4
            LD   PAR1,R4_L
            LD   PAR2,R4_H
            CLC
            RWL  R4
            CLC
            RWL  R4
            JPF  ret_opc
            LDA  R4_L
            AD   PAR1
            STA  R4_L
            LDA  R4_H
            ADD  PAR2
            STA  R4_H
            RET


signprepare:
            ; Tool function for signed multiplication and division:
            ; Make parameters unsigned and store sign of result in R1 bit 0
            PHL
            LD   R1,#0
            LDA  R4_H
            ROL
            JNF  _sgprep1
            JSR  negate_r4
            INC  R1
_sgprep1    LDA  R5_H
            ROL
            JNF  return
            JSR  negate_r5
            INC  R1
            RTS


div_u32u16:
            ;Divide a 32-bit unsigned number by an 16-bit unsigned number.
            ;In : R4/R5 = divident, R6 = divisor
            ;Out: R4/R5 = 32-bit result, R6 = 16-bit reminder
            TST  R6_H
            JPF  _div32_7  ; divisor has only 8 bit
            TST  R5_L
            JNF  _div32_3
            TST  R5_H
            JPF  _div32_4
_div32_3    LD   LIB_BUF+0,#0
            LD   LIB_BUF+1,#0
            LD   LIB_BUF+2,#0
            LD   LIB_BUF+3,#0
            LD   R0,#32
            ;CLC
_div32_1    RWL  R4
            RWL  R5
            RWL  LIB_BUF+0
            RWL  LIB_BUF+2
            LDA  LIB_BUF+0
            SU   R6_L
            STA  R2
            LDA  LIB_BUF+1
            SUB  R6_H
            STA  R3
            JPF  _div32_5
            LD   PAR1,LIB_BUF+2
            TST  PAR1
            INC  FLAG
            DEC  PAR1
            JPF  _div32_6
            LD   PAR2,LIB_BUF+3
            TST  PAR2
            INC  FLAG
            DEC  PAR2
            JNF  _div32_2
            LD   LIB_BUF+3,PAR2
_div32_6    LD   LIB_BUF+2,PAR1
_div32_5    LD   LIB_BUF+0,R2
            LD   LIB_BUF+1,R3
_div32_2    RWL  LIB_BUF+4
            RWL  MATH_BUF+0
            JLP  _div32_1
            LD   R4_L,LIB_BUF+4
            LD   R4_H,LIB_BUF+5
            LD   R5_L,MATH_BUF+0
            LD   R5_H,MATH_BUF+1
            LD   R6_H,LIB_BUF+1
            LD   R6_L,LIB_BUF+0
            RET
            
_div32_7    ;check if upper 8 bits of 32-bit word are 0
            TST  R5_H
            JNF  _div32_3  ;no, do usual 32/16 division
            ;check if upper 16 bits of 32-bit word are 0
            TST  R5_L
            JPF  _div32_4  ;yes, do 16-bit division

            ;do 24-bit division with an 8-bit divisor
            LD   LIB_BUF+0,#0
            LD   LIB_BUF+1,#0
            LD   LIB_BUF+2,#0
            LD   R0,#24
            ;CLC
_div24_1    RWL  R4
            ROL  R5_L
            RWL  LIB_BUF+0
            ROL  LIB_BUF+2
            LDA  LIB_BUF+0
            SU   R6_L
            STA  R2
            JPF  _div24_3
            LD   R3,LIB_BUF+1
            TST  R3
            INC  FLAG
            DEC  R3
            JPF  _div24_4
            TST  LIB_BUF+2
            INC  FLAG
            JNF  _div24_2
            DEC  LIB_BUF+2
_div24_4    LD   LIB_BUF+1,R3
_div24_3    LD   LIB_BUF+0,R2
_div24_2    RWL  LIB_BUF+4
            ROL  MATH_BUF+0
            JLP  _div24_1
            LD   R4_L,LIB_BUF+4
            LD   R4_H,LIB_BUF+5
            LD   R5_L,MATH_BUF+0
            LD   R5_H,#0
            LD   R6_H,LIB_BUF+1
            LD   R6_L,LIB_BUF+0
            RET

_div32_4    ;do 16-bit division
            PHL
            LD   R5_L,R6_L
            LD   R5_H,R6_H
            JSR  divide_u
            LD   R6_L,R5_L
            LD   R6_H,R5_H
            LD   R5_L,#0
            LD   R5_H,#0
            RTS


div_i32i16:
            ;Divide a 32-bit signed number by an 16-bit signed number.
            ;In : R4/R5 = divident, R6 = divisor
            ;Out: R4/R5 = 32-bit result, R6 = 16-bit reminder (call div_i32_crem to correct the sign)
            PHL
            LD   R1,#0  ;the final sign of the result
            LDA  R5_H
            ROL
            PSH  FLAG
            JNF  _divi32_1
            INC  R1
            JSR  negate_R45
_divi32_1   LDA  R6_H
            ROL
            JNF  _divi32_2
            INC  R1
            JSR  inv_r6
            JSR  inc_r6
_divi32_2   JSR  div_u32u16
            ROR  R1
            JNF  _divi32_3
            JSR  negate_R45
_divi32_3   POP  FLAG
            RTS
            ;remember: call div_i32_crem to correct the sign of the reminder


div_i32_crem:
            ;correct the sign of the reminder in R6
            JNF  ret_opc
            PHL
            JSR  inv_r6
            JSR  inc_r6
            RTS


mul32u: 
            ; Multiply a 32-bit unsigned number with a 16-bit unsigned number.
            ; The result is 32-bit wide.
            ; In : R4,R5 = 32-bit number, R6 = 16-bit number
            ; Out: R4,R5 = 32-bit result
            ; Changes: ACCU, R4, R5
            TST  R5_L
            JNF  _mul32_4
            TST  R5_H
            JPF  _mul32_5
_mul32_4    LDA  #0
            STA  LIB_BUF+0
            STA  LIB_BUF+1
            STA  LIB_BUF+2
            STA  LIB_BUF+3
            LD   PAR1,R0
            LD   R0,#32
            JMP  _mul32_1
_mul32_2    RWL  LIB_BUF+0
            RWL  LIB_BUF+2
_mul32_1    RWL  R4
            RWL  R5
            JNF  _mul32_3
            LDA  LIB_BUF+0
            AD   R6_L
            STA  LIB_BUF+0
            LDA  LIB_BUF+1
            ADD  R6_H
            STA  LIB_BUF+1
            JNF  _mul32_3
            INC  LIB_BUF+2
            TST  LIB_BUF+2
            JNF  _mul32_3
            INC  LIB_BUF+3
_mul32_3    JLP  _mul32_2
            LD   R0,PAR1
            LD   R4_L,LIB_BUF+0
            LD   R4_H,LIB_BUF+1
            LD   R5_L,LIB_BUF+2
            LD   R5_H,LIB_BUF+3
            RET
_mul32_5    ;use 16x16 bit multiplication
            LD   R5_L,R6_L
            LD   R5_H,R6_H

multiply_u: 
            ; Multiply two 16-bit unsigned numbers. The result is 32-bit wide.
            ; In : R4, R5 = two 16-bit numbers
            ; Out: R4, R5 = 32-bit result (R4=bit0-15, R5=bit16-31)
            ; Changes: ACCU, R4, R5
            LD   PAR1,R0
            LDA  #0
            STA  LIB_BUF+2
            STA  LIB_BUF+3
            TST  R4_H
            JNF  _multp04
            TST  R5_H
            JPF  _multp05   ;do a 8b x 8b = 16b multiplication
_multp04    STA  LIB_BUF+0
            STA  LIB_BUF+1
            LD   R0,#16
            JMP  _multp01
_multp02    RWL  LIB_BUF+2
            RWL  LIB_BUF+0
_multp01    RWL  R4
            JNF  _multp03
            LDA  LIB_BUF+2
            AD   R5_L
            STA  LIB_BUF+2
            LDA  LIB_BUF+3
            ADD  R5_H
            STA  LIB_BUF+3
            JNF  _multp03
            INC  LIB_BUF+0
            TST  LIB_BUF+0
            JNF  _multp03
            INC  LIB_BUF+1
_multp03    JLP  _multp02
            LD   R0,PAR1
            JMP  _multp06 
_multp05    ;8x8 multiplication
            LD   R0,#8
            JMP  _multp09
_multp07    RWL  LIB_BUF+2
_multp09    ROL  R4_L
            JNF  _multp08
            LDA  LIB_BUF+2
            AD   R5_L
            STA  LIB_BUF+2
            JNF  _multp08
            INC  LIB_BUF+3
            TST  LIB_BUF+3
_multp08    JLP  _multp07
            LD   R0,PAR1
            LD   R5_L,#0
            JMP  _multp10

SEG_PRGCODE

multiply_s:
            ; Multiply two 16-bit signed numbers. The result is 32-bit wide.
            ; In : R4, R5 = two 16-bit numbers
            ; Out: R4, R5 = 32-bit result (R4=bit0-15, R5=bit16-31)
            ; Changes: ACCU, R4, R5
            PHL
            PSH  R1
            JSR  signprepare
            JSR  multiply_u
            ROR  R1
            JNF  _divsgn1
            JSR  negate_R45
            JMP  _divsgn1

mul32s:
            ; Multiply a 32-bit signed number with a 16-bit signed number.
            ; The result is 32-bit wide.
            ; In : R4,R5 = 32-bit number, R6 = 16-bit number
            ; Out: R4,R5 = 32-bit result
            ; Changes: ACCU, R4, R5, R0
            PHL
            LD   R1,#0
            LDA  R5_H
            ROL
            JNF  _mul32s_1
            INC  R1
            JSR  negate_R45
_mul32s_1   LDA  R6_H
            ROL
            JNF  _mul32s_2
            INC  R1
            JSR  inv_r6
            JSR  inc_r6
_mul32s_2   JSR  mul32u
            ROR  R1
            JNF  return
            JSR  negate_R45
            RTS


divide_u:
            ; Divide a 16-bit unsigned number by an other 16-bit unsigned number.
            ; In : R4 = divident, R5 = divisor
            ; Out: R4 = 16-bit result, R5 = 16-bit reminder
            ; Changes: ACCU, R4, R5
            LD   LIB_BUF+0,#0
            LD   LIB_BUF+1,#0
            TST  R4_H
            JPF  _divi03
            PSH  R0
            TST  R5_H
            JPF  _divide_16_8_u  ; R5 is only 8 bit, so use a faster division routine (see lib1.3_math.asm)
            ;16-bit division (FLAG=0 here)
            LD   R0,#16
_divi01     RWL  R4
            RWL  LIB_BUF+0
            LDA  LIB_BUF+0
            SU   R5_L
            STA  PAR1
            LDA  LIB_BUF+1
            SUB  R5_H
            JNF  _divi02
            LD   LIB_BUF+0,PAR1
            STA  LIB_BUF+1
_divi02     RWL  LIB_BUF+2
            JLP  _divi01
            POP  R0
            LD   R4_L,LIB_BUF+2
            LD   R4_H,LIB_BUF+3
            LD   R5_L,LIB_BUF+0
            LD   R5_H,LIB_BUF+1
            RET
_divi03     TST  R5_H
            JNF  _divi08
            ;8-bit division
            LD   LIB_BUF+0,R4_L
            LD   PAR2,R0
            LD   R0,#8
_divi05     RWL  LIB_BUF+0
            LDA  LIB_BUF+1
            SU   R5_L
            JNF  _divi06
            STA  LIB_BUF+1
_divi06     ROL  R4_L
            JLP  _divi05
            LD   R0,PAR2
            LD   R5_L,LIB_BUF+1
            RET
_divi08     ;return only the reminder
            LD   R5_H,#0
            LD   R5_L,R4_L
            LD   R4_L,#0
            RET


_divide_16_8_u:
            ; Divide a 16-bit unsigned number by an 8-bit unsigned number.
            ; In : R4 = divident, R5_L = divisor, R0 on stack
            ; Out: R4 = 16-bit result, R5 = 16-bit reminder
            ; Changes: ACCU, R4, R5
            LD   R0,#16
_divi11     RWL  R4
            RWL  LIB_BUF+0
            LDA  LIB_BUF+0
            SU   R5_L
            STA  PAR1
            JPF  _divi13
            LD   PAR2,LIB_BUF+1
            TST  PAR2
            DEC  PAR2
            INC  FLAG
            JNF  _divi12
            LD   LIB_BUF+1,PAR2
_divi13     LD   LIB_BUF+0,PAR1
_divi12     RWL  LIB_BUF+2
            JLP  _divi11
            POP  R0
_multp06    LD   R5_L,LIB_BUF+0
            LD   R5_H,LIB_BUF+1
_multp10    LD   R4_L,LIB_BUF+2
            LD   R4_H,LIB_BUF+3
            RET


divide_s:
            ; Divide a 16-bit signed number by an other 16-bit signed number.
            ; In : R4 = divident, R5 = divisor
            ; Out: R4 = 16-bit result
            ; Changes: ACCU, R4, R5
            PHL
            PSH  R1
            JSR  signprepare
            JSR  divide_u
            ROR  R1
            JNF  _divsgn1
            JSR  negate_r4
_divsgn1    POP  R1
            RTS


sub32bit:
            ;32-bit subtraction
            LDA  R4_L
            SU   LIB_BUF+0
            STA  R4_L
            LDA  R4_H
            SUB  LIB_BUF+1
            STA  R4_H
            LDA  R5_L
            SUB  LIB_BUF+2
            STA  R5_L
            LDA  R5_H
            SUB  LIB_BUF+3
            STA  R5_H
            RET

