;------------------------------------------------------------------------------
; Implementation of the various Forth stacks
;------------------------------------------------------------------------------

SEG_LOWCODE

#ifdef PLATFORM_XS
SSV     MACRO  r1,r2
            DB 0x29,r1,r2  ; SSV r1,r2 -> does "LD PAR1,SP / LD r1,r2"
        ENDMACRO
SRT     MACRO
            LD   fPSP,SP
            DB 0x2A        ; SRT -> does "LD SP,PAR1 / RET"
        ENDMACRO
TJZ     MACRO
            DB 0x2B        ; JPF  Error_Stack
        ENDMACRO
#else
SSV     MACRO  r1,r2
            LD   PAR1,SP
            LD   r1,r2
        ENDMACRO
SRT     MACRO
            LD   fPSP,SP
            LD   SP,PAR1
            RET
        ENDMACRO
TJZ     MACRO
            JPF  Error_Stack
        ENDMACRO
#endif



;------------------------------------------------------------------------------
;  System Data Stack (34 bytes)
;------------------------------------------------------------------------------

syss_push:
            ;Push a byte to the system stack
            ;Input:   ACCU
            ;Changes: PTR, ACCU
            LD   PTR_L,SYSSP
            LD   PTR_H,#>SYSDATASTACK
            SAP
            INC  SYSSP
            RET

syss_pop:
            ;Pop a byte from the system stack
            ;Output:  R4
            ;Changes: PTR, ACCU
            DEC  SYSSP
            LD   PTR_L,SYSSP
            LD   PTR_H,#>SYSDATASTACK
            LAP
            RET


;------------------------------------------------------------------------------
;  Return Stack
;------------------------------------------------------------------------------

retstk_push:
            ;Push R4 to the return stack
            ;The stack grow from bottom to top
            ;Input:   R4
            ;Changes: PTR, ACCU
            LD   PTR_L,RETSP
            LD   PTR_H,#>RETURNSTACK
            LDA  R4_L
            SAP
            INC__PTR_L
            LDA  R4_H
            SAP
            INC__PTR_L
            LD   RETSP,PTR_L
            RET

retstk_pop:
            ;Pop R4 from the return stack
            ;The stack grow from bottom to top
            ;Input:   R4
            ;Changes: PTR, ACCU
            LD   PTR_L,RETSP
            LD   PTR_H,#>RETURNSTACK
            DEC  PTR_L
            LAP
            STA  R4_H
            DEC  PTR_L
            LAP
            STA  R4_L
            LD   RETSP,PTR_L
            RET



;------------------------------------------------------------------------------
;  Control Flow Stacks
;------------------------------------------------------------------------------

ccfs_push_cp:
            ;Push current CP value to the control flow stack
            LD  R4_L,CP_L
            LD  R4_H,CP_H
ccfs_push:
            ;Push a word to the control flow stack for compiler constructs
            ;Input:   R4
            ;Changes: PTR, ACCU
            ; LDA  #<CCTLFLOWSTACK
            ; CMP  CCFSP
            ; JPF  Error_Stack   ; stack overflow  (don't check the lower bound, CMP is too slow)
            LD   PTR_L,CCFSP
            LD   PTR_H,#>CCTLFLOWSTACK
            DEC  PTR_L
            LDA  R4_H
            SAP
            DEC  PTR_L
            LDA  R4_L
            SAP
            LD   CCFSP,PTR_L
            RET

ccfs_pop:
            ;Pop a word from the control flow stack for compiler constructs
            ;Output:  R4
            ;Changes: PTR, ACCU
            LDA  #<(CCTLFLOWSTACK+CCTLF_STK_SZ)
            CMP  CCFSP
            TJZ  ;JPF  Error_Stack   ; stack underflow
            LD   PTR_L,CCFSP
            LD   PTR_H,#>CCTLFLOWSTACK
            LAP
            INC__PTR_L
            STA  R4_L
            LAP
            INC__PTR_L
            STA  R4_H
            LD   CCFSP,PTR_L
            RET

lcfs_push:
            ;Push a word to the control flow stack for loops
            ;Input:   R4
            ;Changes: PTR, ACCU
            ; LDA  #<LCTLFLOWSTACK
            ; CMP  LCFSP
            ; JPF  Error_Stack   ; stack overflow  (don't check the lower bound, CMP is too slow)
            LD   PTR_L,LCFSP
            LD   PTR_H,#>LCTLFLOWSTACK
            DEC  PTR_L
            LDA  R4_H
            SAP
            DEC  PTR_L
            LDA  R4_L
            SAP
            LD   LCFSP,PTR_L
            RET

lcfs_pop:
            ;Pop a word from the control flow stack for loops
            ;Output:  R4
            ;Changes: PTR, ACCU
            LDA  #<(LCTLFLOWSTACK+LCTLF_STK_SZ)
            CMP  LCFSP
            TJZ  ;JPF  Error_Stack   ; stack underflow
            LD   PTR_L,LCFSP
            LD   PTR_H,#>LCTLFLOWSTACK
            LAP
            INC__PTR_L
            STA  R4_L
            LAP
            INC__PTR_L
            STA  R4_H
            LD   LCFSP,PTR_L
            RET



;------------------------------------------------------------------------------
;  Data Stack
;------------------------------------------------------------------------------


push_flag:
            JNF  push_data_zero

push_data_ffff:
            LD   R4_L,#0xFF
            LD   R4_H,R4_L
#ifdef FAST_DSTACK
            PHD  R4
            RET
#else
            JMP  push_data_R4
#endif

push_data_zero:
            LD   R4_L,#0
_pushR4LR4L LD   R4_H,R4_L
#ifdef FAST_DSTACK
            PHD  R4
            RET
#else
            JMP  push_data_R4
#endif

push_data_accu_ret:
            STA  R4_L
push_data_R4L_ret:
            LD   R4_H,#0
push_data_R4_ret:
#ifdef FAST_DSTACK
            PHD  R4
            RTS
#else
            POP  LR_H
            POP  LR_L
#endif
push_data_R4:
            ;Push some data to the stack
            ;Input:   R4 contains the 16-bit data
            ;Note: An out-of-bounds-check is not done to speed up things a bit!
#ifdef FAST_DSTACK
            PHD  R4
            RET
#else
            SSV  SP,fPSP
            PSH  R4_L
            PSH  R4_H
            SRT
#endif

push_data_1:
            LDA  #1
push_data_accu:
            ;Push 8-bit data to the stack
            ;Input:   ACCU contains the byte
            ;Note: An out-of-bounds-check is not done to speed up things a bit!
            STA  R4_L
            LD   R4_H,#0
#ifdef FAST_DSTACK
            PHD  R4
            RET
#else
            JMP  push_data_R4
#endif


push_data_R4_R5_ret:
#ifdef FAST_DSTACK
            PHD  R4
            PHD  R5
            RTS
#else
            JSR  push_data_R4
#endif
push_data_R5_ret:
#ifdef FAST_DSTACK
            PHD  R5
            RTS
#else
            POP  LR_H
            POP  LR_L
#endif
push_data_R5:
            ;Push some data to the stack
            ;Input:   R5 contains the 16-bit data
            ;Note: An out-of-bounds-check is not done to speed up things a bit!
#ifdef FAST_DSTACK
            PHD  R5
            RET
#else
            SSV  SP,fPSP
#endif
_pshr5_1    PSH  R5_L
            PSH  R5_H
            SRT

pop_data_R4:
            ;Pop some data from the stack
            ;Output:  R4 contains the 16-bit data
#ifdef FAST_DSTACK
            PLD  R4
            RET
#else
            TST  fPSP
            TJZ  ;JPF  Error_Stack  ; stack underflow (check it because otherwise the callstack would be corrupted)
            SSV  SP,fPSP
            POP  R4_H
            POP  R4_L
            SRT
#endif

pop_data_R5:
            ;Pop some data from the stack
            ;Output:  R5 contains the 16-bit data
#ifdef FAST_DSTACK
            PLD  R5
            RET
#else
            TST  fPSP
            TJZ  ;JPF  Error_Stack  ; stack underflow (check it because otherwise the callstack would be corrupted)
            SSV  SP,fPSP
            POP  R5_H
            POP  R5_L
            SRT
#endif


pop_data_R4_R5:
            ;Pop R4 and R5 from the data stack (R4 first, then R5)
            ;Output:  R4 and R5
#ifdef FAST_DSTACK
            PLD  R4
            PLD  R5
            RET
#else
            TST  fPSP
            TJZ  ;JPF  Error_Stack  ; stack underflow (check it because otherwise the callstack would be corrupted)
            SSV  SP,fPSP
            POP  R4_H
            POP  R4_L
            TST  SP
            TJZ  ;JPF  Error_Stack  ; stack underflow (check it because otherwise the callstack would be corrupted)
            POP  R5_H
            POP  R5_L
            SRT
#endif


pop_data_R5_R4:
            ;Pop R5 and R4 from the data stack (R5 first, then R4)
            ;Output:  R4 and R5
#ifdef FAST_DSTACK
            PLD  R5
            PLD  R4
            RET
#else
            TST  fPSP
            TJZ  ;JPF  Error_Stack  ; stack underflow (check it because otherwise the callstack would be corrupted)
            SSV  SP,fPSP
            POP  R5_H
            POP  R5_L
            TST  SP
            TJZ  ;JPF  Error_Stack  ; stack underflow (check it because otherwise the callstack would be corrupted)
            POP  R4_H
            POP  R4_L
            SRT
#endif


dstack_2rot:
            ;rotate data stack (Forth 2ROT word)
            ;( x1 x2 x3 x4 x5 x6 -- x3 x4 x5 x6 x1 x2 )
#ifdef FAST_DSTACK
            PLD  LIB_BUF+0
            PLD  LIB_BUF+2
            PLD  LIB_BUF+4
            PLD  MATH_BUF+0
            PLD  R5
            PLD  R4
            PHD  MATH_BUF+0
            PHD  LIB_BUF+4
            PHD  LIB_BUF+2
            PHD  LIB_BUF+0
            PHD  R4
            PHD  R5
            RET
#else
            SSV  SP,fPSP
            POP  LIB_BUF+0
            POP  LIB_BUF+1
            POP  LIB_BUF+2
            POP  LIB_BUF+3
            POP  LIB_BUF+4
            POP  LIB_BUF+5
            POP  MATH_BUF+0
            POP  MATH_BUF+1
            POP  R5_H
            POP  R5_L
            POP  R4_H
            POP  R4_L
            PSH  MATH_BUF+1
            PSH  MATH_BUF+0
            PSH  LIB_BUF+5
            PSH  LIB_BUF+4
_dstkpsh8   PSH  LIB_BUF+3   ;x3
            PSH  LIB_BUF+2
            PSH  LIB_BUF+1   ;x4
            PSH  LIB_BUF+0
            PSH  R4_L        ;x1
            PSH  R4_H
            JMP  _pshr5_1    ;x2
#endif


dstack_2swap:
            ;2swap ( x1 x2 x3 x4 -- x3 x4 x1 x2 )
#ifdef FAST_DSTACK
            PLD  LIB_BUF+0
            PLD  LIB_BUF+2
            PLD  R5
            PLD  R4
            PHD  LIB_BUF+2
            PHD  LIB_BUF+0
            PHD  R4
            PHD  R5
            RET
#else
            SSV  SP,fPSP
            POP  LIB_BUF+0
            POP  LIB_BUF+1
            POP  LIB_BUF+2
            POP  LIB_BUF+3
            POP  R5_H
            POP  R5_L
            POP  R4_H
            POP  R4_L
            JMP  _dstkpsh8
#endif


#ifndef FAST_DSTACK
push_constant_inline:
            ;Push a constant value that is inlined to the code to the data stack.
            ;The caller must do "JSR push_constant_inline" followed by two data bytes (lo/hi)
            SSV  SP,fPSP
            LD   PTR_L,LR_L
            LD   PTR_H,LR_H
#endif
_pci03      LAP
            PSH
#ifdef FAST_PTR
            IPT
            LAP
            PSH
            IPT
#else
            INC  PTR_L
            TST  PTR_L
            JNF  _pci01
            INC  PTR_H
_pci01      LAP
            PSH
            INC  PTR_L
            TST  PTR_L
            JNF  _pci02
            INC  PTR_H
#endif
_pci02      LD   LR_L,PTR_L
            LD   LR_H,PTR_H
            SRT


push_2constant_inline:
            ;Push a constant value that is inlined to the code to the data stack.
            ;The caller must do "JSR push_constant_inline" followed by two data bytes (lo/hi)
            SSV  SP,fPSP
            LD   PTR_L,LR_L
            LD   PTR_H,LR_H
            LAP
            PSH
#ifdef FAST_PTR
            IPT
            LAP
            PSH
            IPT
            JMP  _pci03
#else
            INC  PTR_L
            TST  PTR_L
            JNF  _pci04
            INC  PTR_H
_pci04      LAP
            PSH
            INC  PTR_L
            TST  PTR_L
            JNF  _pci03
            INC  PTR_H
            JMP  _pci03
#endif

SEG_PRGCODE
