;------------------------------------------------------------------------------
;  Forth Dictionary
;------------------------------------------------------------------------------
;
;  A Forth word is formatted like this in memory:
;
;  +-----------+-----------+------+--------------+
;  | next_ptr  | len_flags | name | machine code |
;  +-----------+-----------+------+--------------+
;
;  - next_ptr is 2 bytes and points to the next word in the dictionary.
;     It is NULL if this is the last word in the dictionary.
;  - len_flags is 1 byte and has the length of the word's name in bits 0-4
;    and optional flags in bits 5-7 (e.g. the IMMEDIATE flag)
;  - name is n bytes long and is *not* terminated by a zero
;  - machine code has a variable length.
;     The machine code staRTS always with "PSH LR_L, PSH LR_H" and ends with
;     "JMP return". In user defined words it contains mainly JSR calls.
;
;------------------------------------------------------------------------------


get_word_len:
            ;Gets the length of a word into R1.
            ;Input:   INPTR points to the name of the word.
            ;         The word must be terminated by SPACE or NULL
            ;Output:  R1 is the length of the word
            ;Changes: PTR, R1, FLAG
            PHL
            JSR  ld_ptr_inptr
            LD   R1,#0xFF
_gwl1       INC  R1
            JSR__lap_inc_ptr
            CMP  #0x20
            JPF  return
            TST
            JNF  _gwl1
            RTS


chooseDictionary:
            ;Choose the correct dictionary with the help of the word length.
            ;Input:   R1 = word length
            ;Output:  PTR = ptr to the dictionary pointer
            LDA  #7
            AND  R1
            STA  PTR_L
            LD   PTR_H,#>tab_dict
            LAP
            STA  PTR_L
            LD   PTR_H,#REGPAGE_HI
            RET


search_forth_word:
            ;Searches for a word in the dictionary, useful to call Forth words from an assembly program.
            ;Input:   PTR points to the name of the word to search for.
            ;Output:  If the word was found, FLAG is set to 0
            ;         PTR points to the code section of the word,
            ;         INPTR points behind the word
            ;         and R4 points to the header of the word.
            ;         R2 contains optional flags for the word (like the IMMEDIATE flag)
            ;         R1 is the ASCII character length of the word
            PHL
            PSH  INPTR_L
            PSH  INPTR_H
            JSR  ld_inptr_ptr
            JSR  search_word_ex
            POP  INPTR_H
            POP  INPTR_L
            RTS


search_word_ex:
            ;See search_word, but skips spaces and gets word length into R1 first.
            PHL
            JSR  skipSpaces
            JSR  get_word_len
            TST  R1
            JNF  _sw01
            JMP  Error_Syntax

search_word:
            ;Searches for a word in the dictionary
            ;Input:   INPTR points to the name of the word to search for.
            ;         R1 must contain the length of the word.
            ;Output:  If the word was found, FLAG is set to 0
            ;         PTR points to the code section of the word,
            ;         INPTR points behind the word
            ;         and R4 points to the header of the word.
            ;         R2 contains optional flags for the word (like the IMMEDIATE flag)
            ;Changes: all registers except R1 which still contains the word length

            ;Walk through the dictionary.
            PHL
_sw01       JSR  chooseDictionary
            LAP
            STA  R6_L
            INC__PTR_L
            LAP
            STA  R6_H
            ;start loop
            ;R6 points to the header of the current word.
            LD   R3,#0x1F
_sw02       LD   R4_L,R6_L
            LD   R4_H,R6_H
            LD   PTR_L,R4_L
            LD   PTR_H,R4_H
            TST  PTR_H
            JPF  return     ;word not found in dictionary, return with FLAG=1
            JSR__lap_inc_ptr
            STA  R6_L
            JSR__lap_inc_ptr
            STA  R6_H       ;R6: ptr to next entry in the dictionary
            JSR__lap_inc_ptr
            STA  R2
            AND  R3         ;mask out bits 5-7, they are FLAGs for other purposes
            CMP  R1         ;test the length of the word
            JNF  _sw02      ;not equal
            ;compare character by character
            LD   R5_L,INPTR_L
            LD   R5_H,INPTR_H
            LD   PAR2,R1
_sw03       JSR__lap_inc_ptr
            STA  R0
            LD   R7_L,PTR_L
            LD   R7_H,PTR_H
            LD   PTR_L,R5_L
            LD   PTR_H,R5_H
            JSR__lap_inc_ptr
            LD   R5_L,PTR_L
            LD   R5_H,PTR_H
          ;<toLowerCase>
            STA  PTR_L
            LD   PTR_H,#>tab_lowerCase
            LAP
          ;</toLowerCase>
            CMP  R0
            JNF  _sw02  ; not equal
            LD   PTR_L,R7_L
            LD   PTR_H,R7_H
            DEC  PAR2
            TST  PAR2
            JNF  _sw03
            ;found!
            LD   INPTR_L,R5_L
            LD   INPTR_H,R5_H
            CLC
            RTS



;------------------------------------------------------------------------------
;  Functions for adding new words to the dictionary
;------------------------------------------------------------------------------


start_new_word:
            ;Start to add a new word to the dictionary (start compilation)
            ;Input:   INPTR points to the name of the word
            ;Output:  INPTR points behind the word name
            ;Changes: R1,R2,R3,R4,R5,PTR,FLAG
            ;Note:    The word gets not yet added to the words-list.
            ;         Call the finish_new_word_ret to perform this.
            PHL
            JSR  memcheck
            JSR  get_word_len
            TST  R1
            JPF  Error_Syntax
            ;check the length of the word name, it must not be longer than 31 bytes
            LDA  #0xE0
            AND  R1
            TST
            JNF  Error_ToLong
            ;remember CP
            LD   DICTL_L,CP_L
            LD   DICTL_H,CP_H
            ;find correct dictionary and store the link to it in memory (begin new word header)
            JSR  chooseDictionary
            LD   DICTLH,PTR_L
            LAP
            JSR  emit_code
            INC__PTR_L
            LAP
            JSR  emit_code
            ;load CP into PTR
            JSR  ld_ptr_cp
            ;store length of the word name
            LDA  R1
            JSR__sap_inc_ptr
            ;copy the name of the word
_saw1       LD   R2,PTR_L
            LD   R3,PTR_H
            JSR  ld_ptr_inptr
            JSR__lap_inc_ptr
            JSR  ld_inptr_ptr
          ;<toLowerCase>
            STA  PTR_L
            LD   PTR_H,#>tab_lowerCase
            LAP
          ;</toLowerCase>
            LD   PTR_L,R2
            LD   PTR_H,R3
            JSR__sap_inc_ptr
            DEC  R1
            TST  R1
            JNF  _saw1
            ;finished
            LD   CP_L,PTR_L
            LD   CP_H,PTR_H
            LD   CW_L,CP_L
            LD   CW_H,CP_H
            RTS

begin_add_new_word:
            ;Start to add a new word to the dictionary (start compilation)
            ;Input:   INPTR points to the name of the word
            ;Output:  INPTR points behind the word name
            ;Changes: R1,R2,R3,R4,R5,PTR,FLAG
            PHL
            JSR  start_new_word
            DB   OPCODE_LDAI  ;skip next byte (PHL instruction)
finish_new_word:
            ;finish the compilation of the new word, add word to words-list
            PHL
            LD   PTR_L,DICTLH
            LD   PTR_H,#REGPAGE_HI
            LDA  DICTL_L
            JSR__sap_inc_ptr
            LDA  DICTL_H
            SAP
            RTS


add_word_entry_code:
            ;add function entry code (PSH LR_L, PSH LR_H -> PHL)
            PHL
add_word_entry_code_ret:
            LDA  #OPCODE_PHL
emit_code_ret:
            JSR  emit_code
            RTS


add_word_call:
            ;Add a reference to another word to the currently compiled word
            ;Input:   PTR points to the code section of the referenced word
            LDA  #OPCODE_JSR
add_JmpOrJsr:
            PHL
add_JmpOrJsr_ret:
            JSR  emit_code
emit_PTR_ret:  ;emit PTR to position of CP and return
            LDA  PTR_L
            JSR  emit_code
            LDA  PTR_H
            JMP  emit_code_ret

emit_jsr:   ;emit a JSR instruction
            ;Input:  R7 = destination address
            LDA  #OPCODE_JSR
emit_opaadr:
            ;emit OP code and address
            ;Input:  Accu = OP-Code, R7 = address
            PHL
emit_opaadr_ret:
            JSR  emit_code
            LDA  R7_L
            JSR  emit_code
            LDA  R7_H
            JMP  emit_code_ret


write_create_code:
            ;write the body code for CREATE to the position CP
            LD   R0,#0  ;dummy flag
            LDA  #12
            JMP  _wvabdy

write_variable_body:
            ;write the body code for a variable to the position CP
            LD   R0,#OPCODE_JMP
            LDA  #9
_wvabdy     AD   CP_L
            STA  R4_L
            LD   R4_H,CP_H
            JNF  _ldR4jmp
            INC  R4_H
            JMP  _ldR4jmp

write_constant_body:
            LD   R0,#OPCODE_JMP
_ldR4jmp    PHL
            JSR  emit_code_ldi
            LDA  #R4_L
            JSR  emit_code
            JSR  emit_code_R4L
            JSR  emit_code_ldi
            LDA  #R4_H
            JSR  emit_code
            LDA  R4_H
            JSR  emit_code
            LDA  R0
            TST  R0
            JNF  _addlit01
            LDA  #OPCODE_JSR
_addlit01   JSR  emit_code
            LDA  #<push_data_R4
            JSR  emit_code
            LDA  #>push_data_R4
            JSR  emit_code
            TST  R0
            JNF  return
            ;add "JMP return" for special CREATE word

add_JmpReturn_ret:
            LD   DOESH_L,CP_L
            LD   DOESH_H,CP_H
            LDA  #OPCODE_RTS
            JSR  emit_code
            JSR  emit_code     ; insert two "NOPs": required by DOES> to insert a JMP
            JMP  emit_code_ret ; insert two "NOPs"


add_16bit_literal:
            ;Push a constant 16-bit value to the data stack
            PHL
add_16bit_literal_ret:
#ifdef FAST_DSTACK
            LDA  #OPCODE_CLC    ; use CLC to load FLAG with zero
            TST  R4_H
            JPF  _add16blit1
            LDA  #OPCODE_LDI    ; emit: LD FLAG,#(R4_H)
            JSR  emit_code
            LDA  #FLAG
            JSR  emit_code
            LDA  R4_H
_add16blit1 JSR  emit_code
            LDA  #OPCODE_LDAI   ; emit: LDA #(R4_L)
            JSR  emit_code
            JSR  emit_code_R4L
            LDA  #OPCODE_PHD    ; push ACCU and FLAG (low-byte/high-byte) to data stack
            JSR  emit_code
            LDA  #ACCU
            JSR  emit_code
            RTS
#else
            TST  R4_H
            JPF  _add16blit1
            LD   R7_L,#<push_constant_inline
            LD   R7_H,#>push_constant_inline
            JSR  emit_jsr
            JMP  emit_R4_ret
_add16blit1 TST  R4_L
            JPF  _add16blit2
            LDA  #1
            CMP  R4_L
            JPF  _add16blit3
            LDA  #OPCODE_LDAI
            JSR  emit_code
            JSR  emit_code_R4L
            LD   PTR_L,#<push_data_accu
            LD   PTR_H,#>push_data_accu
_add16blit4 LDA  #OPCODE_JSR
            JMP  add_JmpOrJsr_ret
_add16blit2 LD   PTR_L,#<push_data_zero
            LD   PTR_H,#>push_data_zero
            JMP  _add16blit4
_add16blit3 LD   PTR_L,#<push_data_1
            LD   PTR_H,#>push_data_1
            JMP  _add16blit4
#endif


popR4R5_add_32bit_literal:
            ;Push a constant 32-bit value to the data stack (size optimized code)
            PHL
            LD   R7_L,#<push_2constant_inline
            LD   R7_H,#>push_2constant_inline
            JSR  emit_jsr
            JSR__pop_data_R4_R5
            LDA  R5_L
            JSR  emit_code
            LDA  R5_H
            JSR  emit_code
            JMP  emit_R4_ret


emit_code_R4L:
            LDA  R4_L
            JMP  emit_code
emit_code_ldi:
            LDA  #OPCODE_LDI
emit_code:
            ;write a code byte to current compile address
            ;and increment the destination pointer
            ;Input: accu = byte to write
            PSH  PTR_L
            PSH  PTR_H
            LD   PTR_L,CP_L
            LD   PTR_H,CP_H
            SAP
            POP  PTR_H
            POP  PTR_L
inc_cp:     ;increment the CP pointer
            INC  CP_L
            TST  CP_L
            JNF  ret_opc
            INC  CP_H
            PSH
            LDA  #>(HEAP_END+1)
            CMP  CP_H
            POP
            JPF  Error_Mem
            RET

ld_ptr_cp:  ;load CP into PTR
            LD   PTR_L,CP_L
            LD   PTR_H,CP_H
            RET

ld_r4_cp:   ;load CP into R4
            LD   R4_L,CP_L
            LD   R4_H,CP_H
            RET


;------------------------------------------------------------------------------
;  Library functions used by the implemented words
;------------------------------------------------------------------------------

print_unsigned:
            ;print unsigned integer number on the data stack with the currently set base
            PHL
            JSR__pop_data_R4
            JMP  _pint01

print_integer:
            ;print integer number on the data stack with the currently set base
            PHL
print_integer_rts:
            JSR__pop_data_R4
            LDA  R4_H
            ROL
            JNF  _pint01
            LDA  #'-'
            JSR  print_char
            JSR  inv_r4
            JSR  inc_r4
_pint01     
#ifndef ROM_16KB
            TST  BASE+1
            JNF  _pint02
            LDA  BASE
            CMP  #10
            JPF  print_decword_rts
#endif
_pint02     JSR__push_data_R4
            JSR  push_data_zero
            JSR  c_lns
            JSR  c_nss
            JSR  c_nsg
            JMP  _c_type_ret


memcpy_short:
            ;must copy from lower to upper address,
            ;otherwise "evaluate" will fail
            PHL
            LD   R6_H,#0
            JMP  _mcpy01

memcpy:     ; Copy memory
            ; In: R4 = source address, R5 = destination address, R6 = length
            ; Changes: ACCU, PTR, R4, R5, R6
            PHL
            LDA  R5_L
            SU   R4_L
            LDA  R5_H
            SUB  R4_H
            JPF  _mcpy04
            ;copy from lower to upper address
            JMP  _mcpy01
_mcpy02     JSR  ld_ptr_r4
            JSR  inc_r4
            LAP
            JSR  ld_ptr_r5
            JSR  inc_r5
            SAP
            TST  R6_L
            JNF  _mcpy03
            DEC  R6_H
_mcpy03     DEC  R6_L
_mcpy01     TST  R6_L
            JNF  _mcpy02
            TST  R6_H
            JNF  _mcpy02
            RTS
            ;copy from upper to lower address
_mcpy04     LDA  R4_L
            AD   R6_L
            STA  R4_L
            LDA  R4_H
            ADD  R6_H
            STA  R4_H
            LDA  R5_L
            AD   R6_L
            STA  R5_L
            LDA  R5_H
            ADD  R6_H
            STA  R5_H
            JMP  _mcpy05
_mcpy06     TST  R4_L
            JNF  _mcpy07
            DEC  R4_H
_mcpy07     DEC  R4_L
            TST  R5_L
            JNF  _mcpy08
            DEC  R5_H
_mcpy08     DEC  R5_L
            JSR  ld_ptr_r4
            LAP
            JSR  ld_ptr_r5
            SAP
            TST  R6_L
            JNF  _mcpy09
            DEC  R6_H
_mcpy09     DEC  R6_L
_mcpy05     TST  R6_L
            JNF  _mcpy06
            TST  R6_H
            JNF  _mcpy06
            RTS


if_check:   ;called by IF to perform boolean check
            ;returns with FLAG=1 if the value on top of the stack was zero
            PHL
            JSR__pop_data_R4
            TST  R4_L
            JNF  return
            TST  R4_H
            RTS


of_check:   ;called by OF to perform the check
            ;returns with FLAG=1 if the value on top of the stack was zero
            PHL
            JSR__pop_data_R4_R5
            LDA  R4_L
            CMP  R5_L
            JNF  _ofck01
            LDA  R4_H
            CMP  R5_H
            JPF  return_flag_0
_ofck01     JSR__push_data_R5
            SEC
            RTS


do_code:    ;this code fragment is called when the word DO is executed
            PHL
            ;get the index start value
            JSR__pop_data_R4
            ;push the index value to the control flow stack
            JSR  lcfs_push
            ;get the end value
            JSR__pop_data_R4
            ;push the end value to the control flow stack
            JSR  lcfs_push
            CLC
            RTS  ;return with FLAG=0 : start the loop


qmdo_code:  ;this code fragment is called when the word ?DO is executed
            PHL
            ;get the index start value
            JSR__pop_data_R4
            ;get the end value
            JSR__pop_data_R5
            ;compare index and end value
            LDA  R4_L
            CMP  R5_L
            JNF  _qmdocd01
            LDA  R4_H
            CMP  R5_H
            JPF  return  ;return with FLAG=1 : do not enter the loop
_qmdocd01   ;push the index value to the control flow stack
            JSR  lcfs_push
            LD   R4_L,R5_L
            LD   R4_H,R5_H
            ;push the end value to the control flow stack
            JSR  lcfs_push
            CLC
            RTS  ;return with FLAG=0 : start the loop


#ifdef FAST_PTR

loop_code:  ;this code fragment is called when the word LOOP is executed
            LD   PTR_L,LCFSP
            LD   PTR_H,#>LCTLFLOWSTACK ; this is a constant
            LAP
            INC__PTR_L
            STA  R6_L
            LAP
            INC__PTR_L
            STA  R6_H
            ;get and increment the index on the control flow stack
            LAP
            INC
            SAP
            TST
            JPF  _flpc01 ;need to increment hi-byte
            ;compare with dest.L
            CMP  R6_L
            JPF  _flpc02
            RET
_flpc02     INC__PTR_L
            ;compare with dest.h
_flpc03     LAP
            CMP  R6_H
            JPF  _flpc04
            RET
_flpc01     ;increment also the hi-byte
            INC__PTR_L
            STA  PAR1
            LAP
            INC
            SAP
            ;compare with dest.L
            LDA  PAR1
            CMP  R6_L
            JPF  _flpc03
            RET
_flpc04     ;limit reached, drop control data from LCFS and quit the loop
            INC__PTR_L
            LD   LCFSP,PTR_L
            RET  ;flag = 1

ploop_code: ;this code fragment is called when the word +LOOP is executed
            PHL
            LD   PTR_L,LCFSP
            LD   PTR_H,#>LCTLFLOWSTACK ; this is a constant
            LAP
            INC__PTR_L
            STA  R6_L
            LAP
            INC__PTR_L
            STA  R6_H
            ;get the index from the control flow stack
            LAP
            INC__PTR_L
            STA  R4_L
            LAP
            STA  R4_H
            ;add the value on top of the stack to the index counter
            JSR__pop_data_R5
            ;R4 = index, R5 = delta, R6 = limit
            ;Check if the limit is crossed (idea copied from pforth, which in turn has it from Gforth):
            ;if ( ((OldDiff ^ (OldDiff + Delta)) & (OldDiff ^ Delta)) < 0 )  { quit loop }
            LDA  R4_L
            SU   R6_L
            STA  R7_L
            LDA  R4_H
            SUB  R6_H
            STA  R7_H
            LDA  R7_H
            XOR  R5_H
            STA  R3
            LDA  R7_L
            AD   R5_L
            LDA  R7_H
            ADD  R5_H
            XOR  R7_H
            AND  R3
            ROL
            JPF  _loopcd03
            ;calculate and save new index value
            LDA  R4_L
            ADD  R5_L
            STA  R4_L
            LDA  R4_H
            ADD  R5_H
            STA  R4_H
            SAP
            DEC  PTR_L
            LDA  R4_L
            SAP
            CLC
            RTS
_loopcd03   ;limit reached, drop control data from LCFS and quit the loop
            INC__PTR_L
            LD   LCFSP,PTR_L
            RTS  ;flag = 1

#else

loop_common:
            ;common code for loop and +loop
            ;get the end value from the loop control flow stack
            LD   PTR_L,LCFSP
            LD   PTR_H,#>LCTLFLOWSTACK ; this is a constant
            LAP
            INC__PTR_L
            STA  R6_L
            LAP
            INC__PTR_L
            STA  R6_H
            ;get the index from the control flow stack
            LAP
            INC__PTR_L
            STA  R4_L
            LAP
            STA  R4_H
            RET


loop_code:  ;this code fragment is called when the word LOOP is executed
            PHL
            JSR  loop_common
            ;increment the index
            INC  R4_L
            TST  R4_L
            JNF  _loopcd01
            INC  R4_H
_loopcd01   ;compare with the limit
            LDA  R4_L
            CMP  R6_L
            JNF  _loopcd02
            LDA  R4_H
            CMP  R6_H
            JPF  _loopcd03
_loopcd02   ;store the updated index value
            LDA  R4_H
            SAP
            DEC  PTR_L
            LDA  R4_L
            SAP
            RTS  ;flag = 0


ploop_code: ;this code fragment is called when the word +LOOP is executed
            PHL
            JSR  loop_common
            ;add the value on top of the stack to the index counter
            JSR__pop_data_R5
            ;R4 = index, R5 = delta, R6 = limit
            ;Check if the limit is crossed (idea copied from pforth, which in turn has it from Gforth):
            ;if ( ((OldDiff ^ (OldDiff + Delta)) & (OldDiff ^ Delta)) < 0 )  { quit loop }
            LDA  R4_L
            SU   R6_L
            STA  R7_L
            LDA  R4_H
            SUB  R6_H
            STA  R7_H
            LDA  R7_H
            XOR  R5_H
            STA  R3
            LDA  R7_L
            AD   R5_L
            LDA  R7_H
            ADD  R5_H
            XOR  R7_H
            AND  R3
            ROL
            JPF  _loopcd03
            ;calculate and save new index value
            LDA  R4_L
            ADD  R5_L
            STA  R4_L
            LDA  R4_H
            ADD  R5_H
            STA  R4_H
            SAP
            DEC  PTR_L
            LDA  R4_L
            SAP
            CLC
            RTS
_loopcd03   ;limit reached, drop control data from LCFS and quit the loop
            INC__PTR_L
            LD   LCFSP,PTR_L
            RTS  ;flag = 1
#endif


until_code: ;this code fragment is called when the word UNTIL (or WHILE) is executed
#ifdef FAST_DSTACK
            PLD  R4
            TST  R4_L
            JNF  ret_opc
            TST  R4_H
            RET
#else
            PHL
            JSR__pop_data_R4
            TST  R4_L
            JNF  return
            TST  R4_H
            RTS
#endif


abort_code: ;this code fragment is called when the word ABORT" is executed
            PHL
            JSR  until_code
            JPF  return
            JSR  print_str
            JMP  c_abort
            
#ifndef NO_BLOCK_WORDS
toEEPindex: ;convert R4 to valid EEPROM index number in R0
            TST  R4_H
            JNF  Error_Value
            LD   R0,R4_L
            DEC  R0
            LDA  #0xF8
            AND  R0
            TST
            JNF  Error_Value
            RET

prep_eepblk ;prepare loading/saving a block
            PHL
            ;get screen/block number from stack
            JSR__pop_data_R4
            JSR__push_data_R4
            ;translate block number in R4 to EEPROM I2C device address (ACCU) and memory address (R4)
            JSR  fblockToEepromAddr
            JPF  Error_Value
            STA  R3  ;EEPROM device address
            ;R4 = memory address
            JSR__pop_data_R5  ; get block number into R5
            RTS  ; return with I2C dev in R3, EEPROM addr in R4
#endif

ld_R5_1024  LD   R5_L,#<1024
            LD   R5_H,#>1024
            RET

#ifndef NO_BLOCK_WORDS
feep_start_read:
            ; Start a read-transfer.
            ; In : R3 : 8-bit I2C-address of the EEPROM
            ;      R4 : 16-bit EEPROM read address
            ; Out: FLAG = 0 on success
            PHL
            JSR  feep_start_write_ackpoll
            LD   R0,R3
            INC  R0
            JSR  i2c_start_addr
            JNF  return          
            JMP  Error_InOut


feep_start_write_ackpoll:
            ; Start a write-transfer, but wait until
            ; previous write has been completed in the EEPROM.
            ; In : R3 : 8-bit I2C-address of the EEPROM
            ;      R4 : 16-bit EEPROM write address
            ; Out: FLAG = 0 on success
            PHL
            LD   R1,#0
_feepstw0   JSR  i2c_start
            JNF  Error_InOut
            PSH  R1
            LD   R1,R3
            JSR  i2c_send
            POP  R1
            JNF  _eepstw3  ; ok, device is ready again, send address in R4
            JSR  i2c_stop
            DEC  R1
            TST  R1
            JPF  Error_InOut
            JMP  _feepstw0
#endif

test_blk:   ;test if BLK is zero
            LDA  BLK_L
            OR   BLK_H
            TST
            RET



;==============================================================================
;  Here is a basic set of primitives for the dictionary
;  To improve performance, the dictionary is split into 8 parts.
;  The correct dictionary is selected by the help of the word length.
;==============================================================================

dictionary_1:
#define WORDS_LEN_1
#define WORDS_LEN_9
#include "forth_allwords.asm"
  DB 0,0,0 ;end-marker
#undef WORDS_LEN_1
#undef WORDS_LEN_9

dictionary_2:
#define WORDS_LEN_2
#define WORDS_LEN_10
#include "forth_allwords.asm"
  DB 0,0,0 ;end-marker
#undef WORDS_LEN_2
#undef WORDS_LEN_10

dictionary_3:
#define WORDS_LEN_3
#define WORDS_LEN_11
#include "forth_allwords.asm"
  DB 0,0,0 ;end-marker
#undef WORDS_LEN_3
#undef WORDS_LEN_11

dictionary_4:
#define WORDS_LEN_4
#define WORDS_LEN_12
#include "forth_allwords.asm"
  DB 0,0,0 ;end-marker
#undef WORDS_LEN_4
#undef WORDS_LEN_12

dictionary_5:
#define WORDS_LEN_5
#define WORDS_LEN_13
#include "forth_allwords.asm"
  DB 0,0,0 ;end-marker
#undef WORDS_LEN_5
#undef WORDS_LEN_13

dictionary_6:
#define WORDS_LEN_6
#define WORDS_LEN_14
#include "forth_allwords.asm"
  DB 0,0,0 ;end-marker
#undef WORDS_LEN_6
#undef WORDS_LEN_14

dictionary_7:
#define WORDS_LEN_7
#define WORDS_LEN_15
#include "forth_allwords.asm"
  DB 0,0,0 ;end-marker
#undef WORDS_LEN_7
#undef WORDS_LEN_15

dictionary_8:
#define WORDS_LEN_8
#define WORDS_LEN_16
#include "forth_allwords.asm"
  DB 0,0,0 ;end-marker
#undef WORDS_LEN_8
#undef WORDS_LEN_16


