;------------------------------------------------------------------------------
;  Binary Driver Loader
;------------------------------------------------------------------------------
;
;  This module can load and relocate a binary driver image into RAM memory.
;
;  The binary driver image must begin with this header:
;
;    0. byte :  ID 0x1B
;    1. byte :  reserved, must be 0
;    2. byte :  file size, bits 0-7
;    3. byte :  file size, bits 8-15
;    4. byte :  module compile address, bits 0-7
;    5. byte :  module compile address, bits 8-15
;    6. byte :  size of the following data section, bits 0-7
;    7. byte :  size of the following data section, bits 8-15
;    8. byte :  pointer to the list of exports, bits 0-7
;    9. byte :  pointer to the list of exports, bits 8-15
;   10. byte :  pointer to the list of imports, bits 0-7
;   11. byte :  pointer to the list of imports, bits 8-15
;   12. byte :  offset to the startup function, bits 0-7   (can be 0 if not used)
;   13. byte :  offset to the startup function, bits 8-15  (can be 0 if not used)
;
;  This module relocates all address it finds in the code section.
;  Data pointers must be written as a pair of " LD PTR_L,# / LD PTR_H,# ",
;  otherwise they won't be recognized.
;
;------------------------------------------------------------------------------

#ifdef WORDS_LEN_5

BDL_MEMSTART    SET LIB_BUF+0  ; binary blob start address in SRAM
BDL_FILESIZE    SET LIB_BUF+2  ; binary blob size
BDL_SRCADDR     SET LIB_BUF+4  ; original compile address of the binary
BDL_DATASIZE    SET LIB_BUF+6  ; size of the data section
BDL_EXPORTS     SET LIB_BUF+8  ; pointer to list of exported words
BDL_IMPORTS     SET LIB_BUF+10 ; pointer to list of imported words
BDL_FSTARTOFS   SET LIB_BUF+12 ; offset to the start function


            ; BLOAD ( n -- )   Load a binary driver from block n and following blocks
            DW   e_bload
            DB   5,"bload"
c_bload     PHL
            ;----------------------------------------------
            ; Load the header and then the whole file
            ;----------------------------------------------
            
            ;get block number from stack, but keep the original on stack
            JSR  c_dup
            JSR__pop_data_R4
            JSR  fblockToEepromAddr
            JPF  Error_Value
            ;read first 14 bytes from EEPROM (the beginning of the program header)
            JSR  eeprom_start_read
            JPF  Error_InOut
            LD   R5_L,#14
            LD   R5_H,#0
            LD   PTR_L,#LIB_BUF
            LD   PTR_H,#>REGPAGE
            JSR  eeprom_read_block
            JSR  eeprom_stop
            LDA  LIB_BUF+0
            CMP  #BIN_HDR_FLAG
            JNF  _c_loadferr
            ;store the start address of the binrary blob in memory in LIB_BUF+0/1
            LD   LIB_BUF+0,CP_L
            LD   LIB_BUF+1,CP_H
            ;allocate the required memory, LIB_BUF+2/3 contains the size of the binary image in the EEPROM
            LD   R4_L,LIB_BUF+2
            LD   R4_H,LIB_BUF+3
            JSR__push_data_R4
            JSR  c_allot
            ;load R6 with the size of the binary image and PTR with the memory destination address
            LD   PTR_L,LIB_BUF+0
            LD   PTR_H,LIB_BUF+1
            LD   R6_L,LIB_BUF+2
            LD   R6_H,LIB_BUF+3
            ;now load the whole binary blob into memory (the block number is on the data stack)
            JSR  loadForthBlocksToMemory
            ;NOTE: the program is now in memory,
            ;and the whole program header is also stored in LIB_BUF - LIB_BUF+13

            ;----------------------------------------------
            ; Correct memory addresses in the code
            ;----------------------------------------------

            ;calculate code offset, put offset into R7
            LDA  BDL_MEMSTART
            SU   BDL_SRCADDR
            STA  R7_L
            LDA  BDL_MEMSTART+1
            SUB  BDL_SRCADDR+1
            STA  R7_H
            ;re-locate the code
            ;get ptr to code section into R4
            LDA  BDL_MEMSTART
            AD   BDL_DATASIZE
            STA  R4_L
            LDA  BDL_MEMSTART+1
            ADD  BDL_DATASIZE+1
            STA  R4_H
            ;put size of code section -1 into R5
            ;the end of the code is reached when R5_H is 0xFF
            CLC
            LDA  BDL_FILESIZE
            SUB  BDL_DATASIZE
            STA  R5_L
            LDA  BDL_FILESIZE+1
            SUB  BDL_DATASIZE+1
            STA  R5_H
            ;loop over all the code
_binld01    JSR  ld_ptr_r4
            LAP
            STA  R0							; R0 is opcode
            STA  PTR_L
            LD   PTR_H,#>tab_opcode
            LAP
            STA  R1							; ACCU and R1 is instruction length
            ROL
            JNF  _binld02
            ;the op-code must be modified
            LDA  R0
            ;test for LD PTR_L,#
            CMP  #OPCODE_LDI
            JNF  _binld07
            JSR  ld_ptr_r4
            JSR__inc_ptr_lap
            CMP  #PTR_L
            JNF  _binld10
            JSR__inc_ptr_lap
            STA  R6_L
            JSR__inc_ptr_lap
            CMP  #OPCODE_LDI
            JNF  _binld10
            JSR__inc_ptr_lap
            CMP  #PTR_H
            JNF  _binld10
            ;it's "load pointer"
            JSR__inc_ptr_lap
            STA  R6_H
            ROL
            JNF  _binld10 ;don't touch this pointer, it points into ROM code
            ;R6 = original address
            JSR  _bl_translate
            JSR  ld_ptr_r4
            JSR__inc_ptr
            JSR__inc_ptr
            LDA  R6_L
            JSR__sap_inc_ptr
            JSR__inc_ptr
            JSR__inc_ptr
            JMP  _binld08
_binld07    ;it's a jump instruction
            JSR  ld_ptr_r4
            JSR__inc_ptr_lap
            STA  R6_L
            JSR__inc_ptr_lap
            STA  R6_H
            ROL
            JNF  _binld10 ;don't touch this pointer, it points into ROM code
            JSR  _bl_translate
            JSR  dec_ptr
            LDA  R6_L
            JSR__sap_inc_ptr
_binld08    LDA  R6_H
            SAP
            ;finished with this op-code
_binld10    LDA  R1
            ROL
_binld02    ;skip bytes and continue with next op-code
            CLC
            ROR
            STA  R0
_binld03    INC  R4_L
            TST  R4_L
            JNF  _binld04
            INC  R4_H
_binld04    TST  R5_L
            JNF  _binld05
            TST  R5_H
            JPF  _binld06
            DEC  R5_H
_binld05    DEC  R5_L
            JLP  _binld03
            JMP  _binld01
_binld06    ;Code translated.

            ;----------------------------------------------
            ; Walk through the list of imports
            ;----------------------------------------------

            LDA  BDL_IMPORTS
            AD   R7_L
            STA  PTR_L
            LDA  BDL_IMPORTS+1
            ADD  R7_H
            STA  PTR_H
            JSR  psh_r7
_binld20    JSR__lap_inc_ptr
            TST
            JPF  _binld21
            JSR  psh_ptr
            JSR__inc_ptr
            JSR__inc_ptr
            JSR  search_forth_word
            JNF  _binld22
            PRINT text_errImport
            JSR  pop_ptr
            JSR__inc_ptr
            JSR__inc_ptr
            JSR  print_str
            JMP  _err_prnl
_binld22    LD   R5_L,PTR_L
            LD   R5_H,PTR_H
            JSR  pop_ptr
            LDA  R5_L
            JSR__sap_inc_ptr
            LDA  R5_H
            JSR__sap_inc_ptr
            JSR  strlen
            JMP  _binld20
_binld21    JSR  pop_r7

            ;----------------------------------------------
            ; Walk through the list of exports
            ;----------------------------------------------

            JSR  push_inptr
            LDA  BDL_EXPORTS
            AD   R7_L
            STA  PTR_L
            LDA  BDL_EXPORTS+1
            ADD  R7_H
            STA  PTR_H
_binld23    JSR__lap_inc_ptr
            STA  R6_L
            JSR__lap_inc_ptr
            TST
            JPF  _binld24
            STA  R6_H
            JSR  _bl_translate
            JSR  ld_inptr_ptr
            JSR  begin_add_new_word
            LDA  #OPCODE_JMP
            JSR  emit_code
            LDA  R6_L
            JSR  emit_code
            LDA  R6_H
            JSR  emit_code
            JSR  ld_ptr_inptr
            JSR__lap_inc_ptr   ; name-end-marker can be 0x00 or 0x20
            TST
            JPF  _binld23
            JSR  psh_ptr
            JSR  c_immediate   ; if marker is 0x20, this is an immediate word
            JSR  pop_ptr
            JMP  _binld23
_binld24    JSR  pop_inptr

            ;----------------------------------------------
            ; Now execute the init function
            ;----------------------------------------------

            TST  BDL_FSTARTOFS+1
            JPF  _binld09
            LD   R6_L,BDL_FSTARTOFS
            LD   R6_H,BDL_FSTARTOFS+1
            JSR  _bl_translate
            LD   R5_H,#OPCODE_JMP
            JSR  REGPAGE+R5_H
_binld09    ;back from init function
            ;all done
            RTS

_bl_translate:
            ;Translate memory address in R6 to new, correct address.
            PHL
            LDA  R7_L
            AD   R6_L
            STA  R6_L
            LDA  R7_H
            ADD  R6_H
            STA  R6_H
            RTS

e_bload     ;------------------------------------------------------------------
#endif
