;------------------------------------------------------------------------------
; I2C Interface (bit-banging)
;------------------------------------------------------------------------------

SEG_LOWCODE

i2c_init:
          ;@API I2C
          ; Initialize the I2C-Bus (bring the bus into a defined state).
          PHL
          LD   R1,#1
          JSR  i2c_start
          JSR  i2c_recv
          JSR  i2c_stop_xs
          JSR  i2c_start
          JSR  i2c_recv

i2c_stop_ret:
          JSR  i2c_stop
          RTS


i2c_start: 
          ;@API I2C
          ; Send a start-condition and prepare registers R6 and R7
          ; for the following I2C-transfer. While the I2C-transfer
          ; is in progress, the user is not allowed to write to
          ; OUT_PORT3 and OUTP.
          ; Out: FLAG = 1 on success
          ; Changes: ACCU
#ifdef PLATFORM_XS
          LDA  #0x06
          STA  I2C_BUSY
          ;set TXD low (=bit0) to block the RXD line
#else
          LDA  #0x02
          OR   OUTP
#endif
          OUT                     ; SDA = 1
          AND  #0xFB
          STA  LIB_I2C_C0D1       ; SDA = 1, SCL = 0
          OR   #0x04
          OUT                     ; SCL = 1
          ;test if SDA is 1
          PSH
          IN
#ifndef PLATFORM_XS
          ROR
          STA  FLAG
#endif
          POP
          STA  LIB_I2C_C1D1       ; SDA = 1, SCL = 1
          AND  #0xFD
          OUT                     ; SDA = 0
          STA  LIB_I2C_C1D0       ; SDA = 0, SCL = 1
          AND  #0xFB
          STA  LIB_I2C_C0D0       ; SDA = 0, SCL = 0
          OUT                     ; SCL = 0
          STA  OUTP
          RET


i2c_stop_xs:
#ifdef PLATFORM_XS
          LDA  OUTP
          AND  #0xFB
          OUT             ; SDA = X, SCL = 0
          LDA  LIB_I2C_C0D0
          OUT             ; SDA = 0, SCL = 0
          LDA  LIB_I2C_C1D0
          OUT             ; SDA = 0, SCL = 1
          LDA  LIB_I2C_C1D1
          OUT             ; SDA = 1, SCL = 1, TXD = 0
          STA  OUTP
          RET
#endif
i2c_stop:
          ;@API I2C
          ; Send a stop-condition
          ; Changes: ACCU
          LDA  OUTP
          AND  #0xFB
          OUT             ; SDA = X, SCL = 0
          LDA  LIB_I2C_C0D0
          OUT             ; SDA = 0, SCL = 0
          LDA  LIB_I2C_C1D0
          OUT             ; SDA = 0, SCL = 1
#ifdef PLATFORM_XS
i2c_finish:
          LD   I2C_BUSY,#0
          LDA  #7  ; SDA, SCL and TXD = 1
#else
          LDA  LIB_I2C_C1D1
#endif
          OUT             ; SDA = 1, SCL = 1
          STA  OUTP
          RET


i2c_start_addr:
          ;@API I2C
          ; Send a start-condition followed by the I2C-address-byte.
          ; In : R0 = I2C-address and R/W-bit
          ; Out: FLAG = acknowledge-bit  (FLAG = 0 when OK)
          ; Changes: ACCU, R0, R1
          PHL
          JSR  i2c_start
          JNF  return_flag_1
          POP  LR_H
          POP  LR_L
          LD   R1,R0


i2c_send:
          ;@API I2C
          ; Send a byte over I2C
          ; In : R1 = byte to send
          ; Out: FLAG = acknowledge-bit  (FLAG = 0 when OK)
          ; Changes: ACCU, R0, R1
#ifdef PLATFORM_XS
          ;The XS-platform has a special built-in instruction
          ;to send data over I2C:
          ICO
          ICH
          ;receive ACK/NAK
          IN
          OUT  LIB_I2C_C0D1       ; SDA = 1, SCL = 0
          RET
#else
#ifdef MY4TH_NFD
          ;The My4TH nfd ROM speeds up I2C transmission with the
          ;help of some additional microcode.
          LD   R0,#8
          LD   OUTP,LIB_I2C_C0D1
          I2T
          OUT  OUTP
#else
          LD   R0,#8
_i2csn4   ROL  R1
          JPF  _i2csn5
          ;SDA = 0
          OUT  LIB_I2C_C0D0       ; SDA = 0, SCL = 0
          OUT  LIB_I2C_C1D0       ; SDA = 0, SCL = 1
          OUT  LIB_I2C_C0D0       ; SDA = 0, SCL = 0
          JLP  _i2csn4
          JMP  _i2csn6
_i2csn5   ;SDA = 1
          OUT  LIB_I2C_C0D1       ; SDA = 1, SCL = 0
          OUT  LIB_I2C_C1D1       ; SDA = 1, SCL = 1
          OUT  LIB_I2C_C0D1       ; SDA = 1, SCL = 0
          JLP  _i2csn4
#endif
_i2csn6   ;last bit send, prepare to receive ACK
          OUT  LIB_I2C_C0D1       ; SDA = 1, SCL = 0
          OUT  LIB_I2C_C1D1       ; SDA = 1, SCL = 1
          CLC                     ; delay
          ;receive ACK/NAK
          IN
          ROR
          STA  FLAG
          OUT  LIB_I2C_C0D1       ; SDA = 1, SCL = 0
          RET
#endif


i2c_recv:
          ;@API I2C
          ; Receive a byte over I2C
          ; In : R1 = acknowledge-bit
          ; Out: R0 and ACCU = received byte
          ; Changes: ACCU, R0
#ifdef PLATFORM_XS
          ;The XS-platform has special built-in instructions
          ;to receive data over I2C faster:
          OUT  LIB_I2C_C0D1       ; SDA = 1
          ICH
          IN
          ICL
          ICH
          IN
          ICL
          ICH
          IN
          ICL
          ICH
          IN
          ICL
          ICH
          IN
          ICL
          ICH
          IN
          ICL
          ICH
          IN
          ICL
          ICH
          IN
          OUT  LIB_I2C_C0D1       ; SCL = 0
          ROL  R0
          ;send ACK/NAK
          LDA  R0
          TST  R1
          JNF  _i2crv2
          ;send ACK (SDA = 0)
          OUT  LIB_I2C_C0D0       ; SDA = 0, SCL = 0
          OUT  LIB_I2C_C1D0       ; SDA = 0, SCL = 1
          OUT  LIB_I2C_C0D0       ; SDA = 0, SCL = 0
          OUT  LIB_I2C_C0D1       ; SDA = 1, SCL = 0
          RET
_i2crv2   ;send NAK (SDA = 1)
          OUT  LIB_I2C_C0D1       ; SDA = 1, SCL = 0
          ICH                     ; SDA = 1, SCL = 1
          OUT  LIB_I2C_C0D1       ; SDA = 1, SCL = 0
          RET
#else
          OUT  LIB_I2C_C0D1       ; SDA = 1
#ifdef MY4TH_NFD
          ;The My4TH nfd ROM speeds up I2C with the help of some
          ;additional microcode. The I2C instruction does:
          ;IN, ROR, OUT LIB_I2C_C0D1, STA FLAG, ROL R0
          OUT  LIB_I2C_C1D1
          I2R
          OUT  LIB_I2C_C1D1
          I2R
          OUT  LIB_I2C_C1D1
          I2R
          OUT  LIB_I2C_C1D1
          I2R
          OUT  LIB_I2C_C1D1
          I2R
          OUT  LIB_I2C_C1D1
          I2R
          OUT  LIB_I2C_C1D1
          I2R
          OUT  LIB_I2C_C1D1
          I2R
#else
          OUT  LIB_I2C_C1D1       ; SCL = 1
          IN
          ROR
          OUT  LIB_I2C_C0D1       ; SCL = 0
          STA  FLAG
          ROL  R0
          OUT  LIB_I2C_C1D1       ; SCL = 1
          IN
          ROR
          OUT  LIB_I2C_C0D1       ; SCL = 0
          STA  FLAG
          ROL  R0
          OUT  LIB_I2C_C1D1       ; SCL = 1
          IN
          ROR
          OUT  LIB_I2C_C0D1       ; SCL = 0
          STA  FLAG
          ROL  R0
          OUT  LIB_I2C_C1D1       ; SCL = 1
          IN
          ROR
          OUT  LIB_I2C_C0D1       ; SCL = 0
          STA  FLAG
          ROL  R0
          OUT  LIB_I2C_C1D1       ; SCL = 1
          IN
          ROR
          OUT  LIB_I2C_C0D1       ; SCL = 0
          STA  FLAG
          ROL  R0
          OUT  LIB_I2C_C1D1       ; SCL = 1
          IN
          ROR
          OUT  LIB_I2C_C0D1       ; SCL = 0
          STA  FLAG
          ROL  R0
          OUT  LIB_I2C_C1D1       ; SCL = 1
          IN
          ROR
          OUT  LIB_I2C_C0D1       ; SCL = 0
          STA  FLAG
          ROL  R0
          OUT  LIB_I2C_C1D1       ; SCL = 1
          IN
          ROR
          OUT  LIB_I2C_C0D1       ; SCL = 0
          STA  FLAG
          ROL  R0
#endif
          ;send ACK/NAK
          TST  R1
          JNF  _i2crv2
          ;send ACK (SDA = 0)
          OUT  LIB_I2C_C0D0       ; SDA = 0, SCL = 0
          OUT  LIB_I2C_C1D0       ; SDA = 0, SCL = 1
          CLC                     ; delay
          OUT  LIB_I2C_C0D0       ; SDA = 0, SCL = 0
          JMP  _i2crv3
_i2crv2   ;send NAK (SDA = 1)
          OUT  LIB_I2C_C0D1       ; SDA = 1, SCL = 0
          OUT  LIB_I2C_C1D1       ; SDA = 1, SCL = 1
          CLC                     ; delay
_i2crv3   OUT  LIB_I2C_C0D1       ; SDA = 1, SCL = 0
          LDA  R0
          RET
#endif


#ifdef PLATFORM_XS
i2c_ping:
          ; Check if a I2C device is addressable.
          ; In: R2 = address
          ; Out: FLAG = acknowledge-bit  (FLAG = 0 when OK, device is present)
          ; Note: The caller must call i2c_finish to free the TXD line again
          PHL
          LD   R0,R2
          JSR  i2c_start_addr
          PSH  FLAG
          JSR  i2c_stop_xs
          POP  FLAG
          RTS
#endif


