;==============================================================================
; This module adds support for the Forth Deck to the My4TH-nfd ROM.
;
; Assemble this module with the myca command
;   $ ../myca m4-i2c-lcd-kybd.asm -o m4-i2c-lcd-kybd.bin
;
; Upload the binary module to My4TH with the my4th tool
; (for example. See "my4th --help" for details):
;   $ ../my4th write /dev/ttyUSB0 binary 60 m4-i2c-lcd-kybd.bin
;
; Configure your Forth Deck to auto-start this driver. In screen 0,
; write the number '1' at the beginning of the first line.
; Then edit screen 1, clear it completely and write this into line 1:
;   60 bload
;
;==============================================================================

#include <my4th/binmod.hsm>

#include "fast-ptr-instrs.hsm"
#include "lcd-kybd-defs.hsm"

;------------------------------------------------------------------------------
; Jumplist for incomming jumps
;------------------------------------------------------------------------------

CODEOFS   SET  0x30
          DB   0,0  ; reserved bytes, used for alignment

fdvect:   JMP  CODEOFS + lcd_init
          JMP  CODEOFS + lcd_clear
          JMP  CODEOFS + lcd_gotoxy
          JMP  CODEOFS + lcd_cursor_on
          JMP  CODEOFS + lcd_cursor_off
          JMP  CODEOFS + lcd_print
          JMP  CODEOFS + lcd_out_ch
          JMP  CODEOFS + lcd_out_fast
          JMP  CODEOFS + lcd_load_charset
          JMP  CODEOFS + keyb_readkey
          JMP  CODEOFS + keyb_inputstring
          JMP  0x0E2F                ; magic number
          JMP  CODEOFS + init_module ; initialization vector for load-image



;------------------------------------------------------------------------------
; Exported Words
;------------------------------------------------------------------------------

EXPORTS:  ;List of exported words.
          EXPORT  w_lcd,      "lcd"
          EXPORT  w_terminal, "terminal"
          EXPORT  w_keyq,     "key?"
          EXPORT  w_edit,     "edit"
          EXPORT  w_list,     "list"
          ; utils
          EXPORT  w_cr_w,     "cr-w"   
          EXPORT  w_cold,     "cold"                    
          EXPORT  w_i2c_wr,   "i2c-wr"
          EXPORT  w_i2c_rr,   "i2c-rr"          
          EXPORT  w_ch453_wr, "ch453-wr"
          EXPORT  w_ctrloutp, "ctrloutp"
          EXPORT  w_firstline, "firstline"          
          EEND



;------------------------------------------------------------------------------
; Imported Words
;------------------------------------------------------------------------------

IMPORTS:  ;List of imported words.
c_list    IMPORT  "list"
c_edit    IMPORT  "edit"
c_key     IMPORT  "key"
c_dup     IMPORT  "dup"
c_block   IMPORT  "block"
c_update  IMPORT  "update"
c_savebuf IMPORT  "save-buffers"
c_emptybu IMPORT  "empty-buffers"
c_fill    IMPORT  "fill"
c_move    IMPORT  "move"
          IEND


#include "lcd-kybd-data.asm"
#include "edit-data.asm"
#include "lcd-kybd-table-keynorm.asm"
#include "lcd-kybd-table-keyctrl.asm"
#include "lcd-kybd-table-keyshift.asm"
#include "lcd-kybd-table-lcd-chr.asm"

;------------------------------------------------------------------------------
; Code Section. This section must be placed last, after all other sections.
;------------------------------------------------------------------------------

CODE:

init_module:
          ;check if our environment is ok
          PHL
          LD   PTR_L,#<(rom_lcd_init+2)
          LD   PTR_H,#>(rom_lcd_init+2)
          LAP
          ROL
_inim01   LD   PTR_L,#<text_errrom
          LD   PTR_H,#>text_errrom
          JNF  _inim04
          LD   PTR_L,#<fdvect
          LD   PTR_H,#>fdvect
          LDA  #0x40
          CMP  PTR_L
          JNF  _inim02
          LDA  #0x84
          CMP  PTR_H
          JPF  _inim03
_inim02   LD   PTR_L,#<text_errload
          LD   PTR_H,#>text_errload
_inim04   JSR  rom_print_str
          RTS
_inim03   ;initialize the LCD and keyboard i2c devices
          JSR  lcd_kybd_i2c_init
          LD   PTR_L,#<text_errfd
          LD   PTR_H,#>text_errfd
          TST  LIB_BIGKEYB
          JPF  _inim04
          ;initialize the LCD
          LD   R7_L,#0
          JSR  lcd_init
          LDA  #0x9C
          CMP  R7_L
          JNF  _inim01
_inim05   ;get ROM version and CPU speed
          LD   PTR_L,#0xFE
          LD   PTR_H,#0x7D
          JSR__lap_inc_ptr
          STA  R7_L  ; CPU speed
          LAP
          AD   #0x11
          STA  R7_H  ; ROM version
          
          ;calculate delay values from CPU speed (delay_value=cpu_MHz*5)
          LDA  R7_L
          ROL
          ROL
          AD   R7_L
          ; change immediate value in delay_ms
          LD   PTR_L,#<(delay_ms+2)
          LD   PTR_H,#>(delay_ms+2)
          SAP
          
          ;print hello message
          LD   PTR_L,#<text_hello1
          LD   PTR_H,#>text_hello1
          JSR  rom_print_str
          ;print version
          LDA  R7_H
          CLC
          ROR
          CLC
          ROR
          CLC
          ROR
          CLC
          ROR
          STA  R4_L
          LD   R4_H,#0
          JSR  rom_print_decword
          LDA  #'.'
          JSR  rom_print_char
          LDA  #0x0F
          AND  R7_H
          STA  R4_L
          LD   R4_H,#0
          JSR  rom_print_decword
          LD   PTR_L,#<text_hello2
          LD   PTR_H,#>text_hello2
          JSR  rom_print_str
          ;print speed
          LD   R4_L,R7_L
          LD   R4_H,#0
          JSR  rom_print_decword
          LD   PTR_L,#<text_hello3
          LD   PTR_H,#>text_hello3
          JSR  rom_print_str
          ;print blocks
          LD   R4_L,LIB_NUMPAGES+0
          LD   R4_H,LIB_NUMPAGES+1
          JSR  rom_print_decword
          LD   PTR_L,#<text_hello4
          LD   PTR_H,#>text_hello4
          JSR  rom_print_str
          ;ready
rts_opc   RTS


;------------------------------------------------------------------------------
; Library Functions
;------------------------------------------------------------------------------

lcdWait2ms:
          LDA  #2
delay_ms: ;wait some milliseconds
          LD   R0,#200  ; slow at init
_dlms01   JLP  _dlms01
          CLC
          DEC
          TST
          JNF  delay_ms
          RET
          
ret_regadr  LD   R4_H,#REGPAGE_HI
            JMP__push_data_R4
                      
#include "lcd-kybd-edit-libs.asm"

#include "lcd-kybd-utils1.asm"
#include "lcd-kybd-utils2.asm"
#include "lcd-kybd-utils3.asm"
#include "lcd-kybd-utils4.asm"
#include "lcd-kybd-utils5.asm"

#include "lcd-kybd.asm"
#include "lcd-kybd-inputstring.asm"
#include "lcd-kybd-words.asm"

#include "edit.asm"


;------------------------------------------------------------------------------
END_MODULE  ; This line terminates the code section of this binary module
