;=============================================================================;
;  MY4TH Operating System: A very small FORTH system with discrete logic CPU  ;
;  (c) in 2023 by Dennis Kuschel, dennis_k@freenet.de                         ;
;  This work is puplished under the Creative Commons "CC BY-SA 4.0" license.  ;
;=============================================================================;


#include <my4th/my4th.hsm>

#ifndef CPUCLOCK
#define CPUCLOCK  8 ;MHz
#endif
#ifndef PFFLAG
#define PFFLAG  0
#endif

#ifdef ROM_16KB
; My4TH XS / 16kB ROM
ROM_UPPER     SET  0x3D00
ROM_HI        SET  ROM_UPPER+0150h
ROM_STRINGS   SET  ROM_UPPER+0080h
ROM_TABLES1   SET  ROM_UPPER+0100h
ROM_TABLES2   SET  ROM_UPPER+0200h
#else
ROM_TABLES1   SET  7A00h
ROM_TABLES2   SET  7C00h
#ifdef PLATFORM_XS
; My4TH light / 32kB ROM
ROM_HI        SET  0x7770
ROM_STRINGS   SET  ROM_HI+0100h-70h
#else
; My4TH
#ifndef WITH_I2C_LCD_KYBD
ROM_HI        SET  0x7670
ROM_STRINGS   SET  ROM_HI+0200h-70h
#else ; make use of 96 bytes
ROM_HI        SET  0x7604
ROM_STRINGS   SET  0x7800
#endif ;WITH_I2C_LCD_KYBD
#endif
#endif


;------------------------------------------------------------------------------
; Includes and Macros
;------------------------------------------------------------------------------

#include "version.hsm"
#include "tables.asm"
#include "strings.asm"

strings_end ;--------------

#ifdef MY4TH_NFD
  #include "my4th-nfd-defs.hsm"
#endif

#include "ptr-ops.hsm"

; make use of 2xx bytes of free ROM space
#ifdef WITH_I2C_LCD_KYBD
#include "m4-i2c-lcd-kybd-fast/lcd-kybd-utils1.asm"
#else
#include "keydefs.hsm"	;lcd-kybd-defs.hsm is included in my4th-nfd.asm
#endif

#include "variables.asm"

PRINT   MACRO x
          JSR  printstrm
          DB   <x
          DB   >x
        ENDMACRO 


;------------------------------------------------------------------------------
; The program code starts here
;------------------------------------------------------------------------------

#ifdef PLATFORM_XS
  #ifdef ROM_16KB
    #define SEG_LOWCODE SEG_PRGCODE
    SEG_PRGCODE
    ORG 0x1600
  #else
    #define ROM_XFER 0x7A00
    SEG_PRGCODE
    ORG 0x2D00
    #include "jumplist.asm"  ; must be placed at address 0x2D00
    #define SEG_LOWCODE _lowprog Segment Code
    SEG_LOWCODE
    ORG 0x1600
  #endif
  JMP reset_entry
  JMP Error_Stack  ; vector for TJZ instruction
  #include "platform_xs.asm"
#else
  #define ROM_XFER 0x7700
  #define SEG_LOWCODE SEG_PRGCODE
  SEG_PRGCODE
  ORG 0x2D00
  #include "jumplist.asm"  ; must come first, starts at address 0x2D00
  
#ifdef PLATFORM_MYNOR  
  #include "mynor-out.asm" ; must come after jumplist, implements code at address 0x2E00
#endif

#endif

#ifdef MY4TH_NFD
  #include "my4th-nfd.asm"
#endif

;delay, uart and lcd-kybd-i2c-init in one 256-byte segment

#include "delay.asm"
#include "uart.asm"

;uart_functions_end ;--------------

#ifdef WITH_I2C_LCD_KYBD
#include "m4-i2c-lcd-kybd-fast/lcd-kybd-utils3.asm"
#endif

SEG_PRGCODE

#ifndef NO_I2C
  #include "i2c.asm"
#endif
#ifndef NO_LCD
  #include "lcd.asm"
#endif
#include "print.asm"
#include "input.asm"
#ifndef NO_I2C
  #include "eeprom.asm"
#endif
#ifndef NO_LCD
  #include "keyboard.asm"
#endif
#include "misc.asm"
#ifndef NO_BLOCK_WORDS
  #include "transfer.asm"
#endif
#ifdef INCLUDE_FLOAT_WORDS
  #define FLOAT_CODE
  #include "forth_words_float.asm"
  #undef FLOAT_CODE
#endif
#ifdef INCLUDE_IOBOARD_WORDS
  #define IOBOARDS_CODE
  #include "forth_words_ioboards.asm"
  #undef IOBOARDS_CODE
#endif

#include "forth_math.asm"
#include "forth_stacks.asm"
#include "forth_main.asm"
#include "forth_words.asm"



reset_entry:
          ;initialize stack and output port first
          LD   SP,#0
          LD   OUTP,#0xFF  ; TXD = 1, SDA = 1, SCL = 1
          OUT  OUTP
          ; delay 100ms
          LDA  #100
          JSR  delay_ms
#ifdef PLATFORM_MYNOR
          ;if compiled for MyNOR, initialize the two addition output ports
          JSR  mynor_init_ports
#endif
#ifndef PLATFORM_XS
          ;test mode: blink a LED connected to TXD if configured by user
          JSR  test_and_blink_led
#endif

          ;initialize variables
          JSR  init_vectors

          ;initialize the local console (LCD+keyboard or terminal via RS232)
;          LD   LIB_LCDOUT,#0	; in init_console
          JSR  init_console

#ifdef MY4TH_NFD
#ifndef WITH_I2C_LCD_KYBD    ; in lcd_kybd_i2c_init      
          LD   LIB_BIGKEYB,#0
#endif          
#ifndef NO_DETECT_FORTHDECK
          ;Check if this system is a Forth Deck. When yes, print a warning on the LCD.
          JSR  detect_fourthdeck
#endif

#endif

          ;initialize the system
#ifndef NO_I2C
          JSR  i2c_init
#ifdef WITH_I2C_LCD_KYBD
          JSR  detect_i2c_lcd_kybd
#endif                    
          JSR  eeprom_init
#endif
          PRINT text_gitcommit

          ;start forth
          JMP  forth_start


init_console:
#ifndef NO_LCD
          PHL
          ;Initialize the keyboard (if any is present). Must be initialized before the LCD gets initialized.
          JSR  terminal_init
          JSR  keyboard_init
          ;initialize LCD (if any is present) 
          TST  LIB_BIGKEYB
          JPF  _aenokeyb
          JSR  lcd_init
_aenokeyb RTS
#endif

terminal_init:
          LD   LIB_LCDOUT,#0
          LD   LIB_DISPMAXX,#80
          LD   LIB_DISPMAXY,#24
          RET


#ifndef PLATFORM_XS
test_and_blink_led:
          ;check if jumper wire is installed between SCL and RXD
          IN
          ROR
          JNF  ret_opc
          LDA  #0xFB
          OUT
          IN
          ROR
          JPF  ret_opc
          OUT  OUTP
          IN
          ROR
          JNF  ret_opc
          ;blink the LED

_tabl1    LDA  OUTP
          XOR  #1
          OUT
          STA  OUTP
          LDA  #200
          JSR  delay_ms          
          JMP  _tabl1

#endif

#ifdef WITH_I2C_LCD_KYBD
#include "m4-i2c-lcd-kybd-fast/lcd-kybd-utils5.asm"
#endif
