#include <stdio.h>
#include <string.h>
#include "simulation.h"
#include "vcd.h"
#include "ioports.h"
#include "cpu.h"


typedef struct dff8gate {
  iowritecb_t  callback;
  ioreadcb_t   incallback;
  signal_t     cp;
  dbus_t       flipflops;
} dff8gate_t;


/* Gates and Memory */
static dbus_t     IC5_HC574_g;      // current OP-Code latch
static dbus_t     IC6_HC574_g;      // direct address latch, signals A0-A7
static dbus_t     IC7_HC574_g;      // direct address latch, signals A8-A15
static dbus_t     IC10_HC574_g;     // data storage register (and ALU output)
static dbus_t     IC12_HC273_g;     // control word latch 
static dff8gate_t IC17_HC574_g;     // Input Port
static dff8gate_t IC18_HC574_g;     // Output Port
static dbus_t     EPROM_g[0x8000];  // EPROM 32kB
static dbus_t     SRAM_g[0x8000];   // SRAM 32kB
static signal_t   SRAM_oe_g = 1;

/* Clock and Reset */
static signal_t   sig_rcreset      = LOW;   // RC-delay for reset
static signal_t   sig_reset        = HIGH;
static signal_t   sig_reset_n      = LOW;
static signal_t   sig_clk          = LOW;
static signal_t   sig_clk_d        = LOW;

/* Buses */
static abus_t     sig_address      = 0xA5A5;
static dbus_t     sig_addr_l       = 0xA5;
static dbus_t     sig_addr_h       = 0xA5;
static dbus_t     sig_data         = 0xA5;
static dbus_t     sig_eprom_out    = 0xEE;  // output of EPROM
static dbus_t     sig_control      = 0xEE;  // control word from EPROM  

/* Microcode Step Counter */
static dbus_t     sig_counter1_out = 0;     // Output of 4-bit counter 1
static dbus_t     sig_counter2_out = 0;     // Output of 4-bit counter 2
static dbus_t     sig_ctrbus       = 0;     // combined 8-bit output of both counters
static signal_t   sig_phase        = LOW;   // Output bit 0 of counter 1 (= address line 0)
static signal_t   sig_ctrcarry     = LOW;   // Carry of sig_counter1_out
static signal_t   sig_ctrload      = LOW;   // load-enable for counters
static signal_t   sig_ctrload_n    = HIGH;  // load-enable for counters
static signal_t   sig_oectr        = HIGH;  // select output of counters to drive the address bus
static signal_t   sig_fetch        = HIGH;  // fetch op-code, gated version of sig_c_op
static signal_t   sig_load_ctr     = HIGH;  // enable loading the counter, <= sig_ctl_ldctr and sig_phase

/* Direct Address Latch and OE-Driver */
static signal_t   sig_oeadr        = HIGH;  // select output of 16-bit address latch to drive the address bus

/* "ALU" and data storage */
static signal_t   sig_norin        = LOW;   // input to the NOR-gate
static signal_t   sig_norin_n      = LOW;   // input to the NOR-gate, negated
static signal_t   sig_norout       = LOW;   // output of the NOR-gate
static signal_t   sig_oedatar      = HIGH;  // read output of the ALU

/* Data-Target Decoder Outputs */
static signal_t   sig_c_op         = HIGH;  // clock signal for fetching the OP-Code
static signal_t   sig_c_norin      = HIGH;  // clock signal for latching the ALU-input-bit
static signal_t   sig_c_datar      = HIGH;  // clock signal for latching the ALU-output in the data storage register
static signal_t   sig_c_adr_l      = HIGH;  // clock signal for loading direct address low
static signal_t   sig_c_adr_h      = HIGH;  // clock signal for loading direct address high
static signal_t   sig_c_ram        = HIGH;  // clock signal for writing data into the RAM
static signal_t   sig_c_out        = HIGH;  // clock signal for writing data into the I/O output register

/* Control Outputs of Micro-Code EPROM */
static signal_t   sig_ctl_dtsel    = LOW;   // data register select
static signal_t   sig_ctl_tg0      = HIGH;  // data target select, bit 0
static signal_t   sig_ctl_tg1      = HIGH;  // data target select, bit 1
static signal_t   sig_ctl_tg2      = HIGH;  // data target select, bit 2
static signal_t   sig_ctl_asel     = LOW;   // if low, select micro-code-address to drive the EEPROM during 2nd clock cycle
static signal_t   sig_ctl_ldctr    = LOW;   // if high, enable loading the program counter if also D0 is high
static signal_t   sig_ctl_resnor_n = HIGH;  // if low, reset the flip flop in front of the NOR gate (input 1 of NOR)
static signal_t   sig_ctl_setnor_n = HIGH;  // if low, set the flip flop in front of the NOR gate (input 1 of NOR)
static signal_t   sig_ctl_insel    = LOW;   // if high, select the input port instead the RAM as data source

/* Memory mapped I/O */
static signal_t   sig_ckctl        = HIGH;  // enable-input to 2nd decoder stage (clock controlled enable input)
static signal_t   sig_rdram        = HIGH;  // read from RAM if low
static signal_t   sig_rdrom        = HIGH;  // read from ROM if low
static signal_t   sig_rdin         = HIGH;  // read from input register if low

/* IO-Ports */
static dbus_t     sig_input_port   = 0x6E;
static dbus_t     sig_output_port  = 0x00;

/* misc */
static int        enable_vcd_g  = 0;
static unsigned   resettimer_g  = 0;
extern int        clock_period_g;
       uint64_t   exectime_g = 0;  // execution time in [ns]
       uint64_t   uart_lastexec_g = 0;

/* Function pointer for CPU simulation */
typedef void (*fn_dologic_t) (void);
static fn_dologic_t  pfn_dologic_g = NULL;


/* Function Prototypes */
static inline void DFF8_gate(dff8gate_t *gate, signal_t nOE, signal_t CP, dbus_t in, dbus_t *out);



/*===========================================================================*/
/*  My4TH Board Simulation                                                   */
/*===========================================================================*/


/** This function implements a fast simulation of the MyTH CPU.
 */
static void My4TH_all_logic(void)
{
  static signal_t sig_rdin_d = HIGH;
  uint64_t dif;

  static signal_t IC1a_HC74 = 0;
  static signal_t IC5_HC574_ck = 0;
  static signal_t IC6_HC574_ck = 0;
  static signal_t IC7_HC574_ck = 0;
  static signal_t IC10_HC574_ck = 0;
  static signal_t IC12_HC273_ck = 0;
  static signal_t IC1b_HC74_ff = 0;
  static signal_t IC1b_HC74_ck = 0;
  static uint8_t  ucCtr = 0;

  int clock_event = (sig_clk_d == LOW) && (sig_clk == HIGH) ? 1 : 0;
  sig_clk_d = sig_clk;

  if (clock_event)
  {
    IC1a_HC74 = sig_rcreset;
    sig_reset_n = IC1a_HC74;
    sig_reset = (IC1a_HC74 == LOW) ? HIGH : LOW;
  }
    
  /* Microcode Step Counter */  
  sig_fetch     = sig_c_op & sig_reset_n;
  sig_oectr     = sig_ctl_asel & sig_phase;
  sig_load_ctr  = sig_ctl_ldctr & sig_phase;
  sig_ctrload   = sig_load_ctr & sig_norin_n;
  sig_ctrload_n = sig_ctrload ^ 1;

  if (clock_event)
  {
    if (sig_ctrload_n == LOW) {
      ucCtr = (uint8_t) (sig_address & ~1);
    } else {
      ucCtr++;
    }
  }
  if (sig_reset_n == LOW)
    ucCtr = 0;

  sig_counter1_out = ucCtr & 0x0F;
  sig_counter2_out = (ucCtr >> 4) & 0x0F;
  sig_ctrcarry = (sig_counter1_out == 0x0F) ? HIGH : LOW;
  sig_ctrbus = ucCtr;
  sig_phase = ucCtr & 1; // address bit 0 from output of counter 1
  
  if (sig_oectr == LOW)
    sig_addr_l = sig_ctrbus;

  /* OP-Code Latch */
  if (!IC5_HC574_ck && sig_fetch)
    IC5_HC574_g = sig_data & 0x7F;
  if (sig_oectr == LOW)
    sig_addr_h = IC5_HC574_g & 0xFF;
  IC5_HC574_ck = sig_fetch;
  
  /* Direct Address Latch and OE-Driver */
  sig_oeadr = (sig_oectr ^ 1) & 1;

  if (!IC6_HC574_ck && sig_c_adr_l)
    IC6_HC574_g = sig_data;
  if (sig_oeadr == LOW)
    sig_addr_l = IC6_HC574_g;
  IC6_HC574_ck = sig_c_adr_l;
  
  if (!IC7_HC574_ck && sig_c_adr_h)
    IC7_HC574_g = sig_data;
  if (sig_oeadr == LOW)
    sig_addr_h = IC7_HC574_g;
  IC7_HC574_ck = sig_c_adr_h;

  sig_address = (((abus_t)sig_addr_h) << 8) + sig_addr_l;
  
  /* "ALU" and data storage register */
  if (!IC1b_HC74_ck && sig_c_norin)
    IC1b_HC74_ff = sig_data & 1;
  if (sig_ctl_resnor_n == LOW)
    IC1b_HC74_ff = LOW;
  sig_norin    = IC1b_HC74_ff;
  sig_norin_n  = IC1b_HC74_ff ^ 1;
  IC1b_HC74_ck = sig_c_norin;
  sig_norout   = ((sig_norin | sig_data) ^ 1) & 1;

  /* EPROM Output */
  if (!sig_rdrom)
    sig_data = sig_eprom_out;
  
  if (!IC12_HC273_ck && sig_phase)
    IC12_HC273_g = sig_data;
  if (sig_reset)
    IC12_HC273_g = 0;
  sig_control = IC12_HC273_g;
  IC12_HC273_ck = sig_phase;
  
  sig_ctl_dtsel     = (signal_t) ((sig_control >> 0) & 1);
  sig_ctl_tg0       = (signal_t) ((sig_control >> 1) & 1);
  sig_ctl_tg1       = (signal_t) ((sig_control >> 2) & 1);
  sig_ctl_tg2       = (signal_t) ((sig_control >> 3) & 1);
  sig_ctl_asel      = (signal_t) ((sig_control >> 4) & 1);
  sig_ctl_ldctr     = (signal_t) ((sig_control >> 5) & 1);
  sig_ctl_resnor_n  = (signal_t) ((sig_control >> 6) & 1);
  sig_ctl_insel     = (signal_t) ((sig_control >> 7) & 1);

  /* Control Signals Decoder */
  sig_c_op    = HIGH;
  sig_c_norin = HIGH;
  sig_c_datar = HIGH;
  sig_c_adr_l = HIGH;
  sig_c_adr_h = HIGH;
  sig_c_ram   = HIGH;
  sig_c_out   = HIGH;
  if ((sig_phase == HIGH) && (sig_clk == LOW))
  {
    switch ((sig_control >> 1) & 7)
    {
      case 0 : sig_c_op    = LOW; break;
      case 1 : sig_c_out   = LOW; break;
      case 2 : sig_c_norin = LOW; break;
      case 3 : sig_c_datar = LOW; break;
      case 4 : sig_c_adr_l = LOW; break;
      case 5 : sig_c_adr_h = LOW; break;
      case 6 : sig_c_ram   = LOW; break;
      default: break;
    }
  }

  /* Address Decoders */
  /* U14A */
  sig_ckctl = HIGH;
  sig_oedatar = HIGH;
  if (!sig_clk && sig_phase)
  {
    if (!sig_ctl_dtsel) {
      sig_ckctl = LOW;
    } else {
      sig_oedatar = LOW;
    }
  }
  /* U14B */
  sig_rdram = HIGH;
  sig_rdin = HIGH;
  if (!sig_ckctl && (sig_address & (1<<15)))
  {
    if (sig_ctl_insel) {
      sig_rdin = LOW;
    } else {
      sig_rdram = LOW;
    }
  }

  sig_rdrom = ((!sig_phase) | !((sig_address>>15) | sig_ckctl)) ? LOW : HIGH;

  /* EPROM */
  sig_eprom_out = 0x00;  // Pull-Down-Resistors at EPROM output
  if (sig_reset == LOW)
      sig_eprom_out = EPROM_g[sig_address & 0x7FFF];

  /* RAM */
  if (sig_rdram == LOW)
    sig_data = SRAM_g[sig_address & 0x7FFF];
  if (sig_c_ram && !SRAM_oe_g)
  {
    SRAM_g[sig_address & 0x7FFF] = sig_data;
//if ((sig_address & 0x7FFF) >= 0x200)
// printf("RAM 0x%04X := 0x%02X\n",sig_address,sig_data);
  }
  SRAM_oe_g = sig_c_ram;

  /* DATA Register */
  if (!IC10_HC574_ck && sig_c_datar)
    IC10_HC574_g = (sig_data & ~1) | sig_norout;
  if (sig_oedatar == LOW)
    sig_data = IC10_HC574_g;
  IC10_HC574_ck = sig_c_datar;

  /* I/O */
  DFF8_gate(&IC17_HC574_g, sig_rdin, sig_fetch, sig_input_port, &sig_data);
  DFF8_gate(&IC18_HC574_g, LOW, sig_c_out, sig_data, &sig_output_port);

  if (iodisplay_g == iodisp_uarttiming)
  {
    /* check for falling edge of sig_rdin */
    if ((sig_rdin_d == HIGH) && (sig_rdin == LOW))
    {
      dif = exectime_g - uart_lastexec_g;
      uart_lastexec_g = exectime_g;
      printf("INPORT:  0x%02X  (%d Hz, %d us)\n", sig_data, (unsigned)(1000000000UL/dif), (unsigned)(dif/1000));
  //    printf("Read IO-INPUT: %ld Hz\n", 1000000000UL/dif);
    }
    sig_rdin_d = sig_rdin;
  }
}



/*===========================================================================*/
/*  My4TH-XS Board Simulation                                                */
/*===========================================================================*/


/** This function implements a fast simulation of the My4TH XS CPU.
 */
static void My4TH_XS_all_logic(void)
{
  static signal_t sig_rdin_d = HIGH;
  uint64_t dif;

  static signal_t IC1a_HC74 = 0;
  static signal_t IC5_HC574_ck = 0;
  static signal_t IC6_HC574_ck = 0;
  static signal_t IC7_HC574_ck = 0;
  static signal_t IC10_HC574_ck = 0;
  static signal_t IC12_HC273_ck = 0;
  static signal_t IC1b_HC74_ff = 0;
  static signal_t IC1b_HC74_ck = 0;
  static uint8_t  ucCtr = 0;

  int clock_event = (sig_clk_d == LOW) && (sig_clk == HIGH) ? 1 : 0;
  sig_clk_d = sig_clk;

  if (clock_event)
  {
    IC1a_HC74 = sig_rcreset;
    sig_reset_n = IC1a_HC74;
    sig_reset = (IC1a_HC74 == LOW) ? HIGH : LOW;
  }
    
  /* Microcode Step Counter */  
  sig_fetch     = sig_c_op & sig_reset_n;
  sig_oectr     = sig_ctl_asel & sig_phase;
  sig_load_ctr  = sig_ctl_ldctr & sig_phase;
  sig_ctrload   = sig_load_ctr & sig_norin_n;
  sig_ctrload_n = sig_ctrload ^ 1;

  if (clock_event)
  {
    if (sig_ctrload_n == LOW) {
      ucCtr = (uint8_t) (sig_address & 6) | 8;
    } else {
      ucCtr = (ucCtr + 1) & 0x0F;
    }
  }
  if (sig_reset_n == LOW)
    ucCtr = 0;

  sig_counter1_out = ucCtr & 0x0F;
  sig_ctrbus = ucCtr;
  sig_phase = ucCtr & 1; // address bit 0 from output of counter 1
  

  /* OP-Code Latch */
  if (!IC5_HC574_ck && sig_fetch)
    IC5_HC574_g = sig_data;
  if (sig_oectr == LOW)
  {
    sig_addr_l = ((IC5_HC574_g << 4) & 0xF0) | (sig_ctrbus & 0x0F);
    sig_addr_h = (IC5_HC574_g >> 4) & 0x0F;
  }
  IC5_HC574_ck = sig_fetch;
  
  /* Direct Address Latch and OE-Driver */
  sig_oeadr = (sig_oectr ^ 1) & 1;

  if (!IC6_HC574_ck && sig_c_adr_l)
    IC6_HC574_g = sig_data;
  if (sig_oeadr == LOW)
    sig_addr_l = IC6_HC574_g;
  IC6_HC574_ck = sig_c_adr_l;
  
  if (!IC7_HC574_ck && sig_c_adr_h)
    IC7_HC574_g = sig_data;
  if (sig_oeadr == LOW)
    sig_addr_h = IC7_HC574_g;
  IC7_HC574_ck = sig_c_adr_h;

  sig_address = (((abus_t)sig_addr_h) << 8) + sig_addr_l;
  
  /* "ALU" and data storage register */
  if (!IC1b_HC74_ck && sig_c_norin)
    IC1b_HC74_ff = sig_data & 1;
  if (sig_ctl_resnor_n == LOW)
    IC1b_HC74_ff = LOW;
  if (sig_ctl_setnor_n == LOW)
    IC1b_HC74_ff = HIGH;
  sig_norin    = IC1b_HC74_ff;
  sig_norin_n  = IC1b_HC74_ff ^ 1;
  IC1b_HC74_ck = sig_c_norin;
  sig_norout   = ((sig_norin | sig_data) ^ 1) & 1;

  /* EPROM Output */
  if (sig_rdrom == LOW)
  {
    sig_data = sig_reset_n ? sig_eprom_out : 0;
  }

  if (!IC12_HC273_ck && sig_phase)
    IC12_HC273_g = sig_data;
  if (sig_reset)
    IC12_HC273_g = 0;
  sig_control = IC12_HC273_g;
  IC12_HC273_ck = sig_phase;
  
  sig_ctl_dtsel     = (signal_t) ((sig_control >> 0) & 1);
  sig_ctl_tg0       = (signal_t) ((sig_control >> 1) & 1);
  sig_ctl_tg1       = (signal_t) ((sig_control >> 2) & 1);
  sig_ctl_tg2       = (signal_t) ((sig_control >> 3) & 1);
  sig_ctl_asel      = (signal_t) ((sig_control >> 4) & 1);
  sig_ctl_ldctr     = (signal_t) ((sig_control >> 5) & 1);
  sig_ctl_resnor_n  = (signal_t) ((sig_control >> 6) & 1);
  sig_ctl_insel     = (signal_t) ((sig_control >> 7) & 1);

  /* Control Signals Decoder */
  sig_c_op    = HIGH;
  sig_c_norin = HIGH;
  sig_c_datar = HIGH;
  sig_c_adr_l = HIGH;
  sig_c_adr_h = HIGH;
  sig_c_ram   = HIGH;
  sig_c_out   = HIGH;
  if ((sig_phase == HIGH) && (sig_clk == LOW))
  {
    switch ((sig_control >> 1) & 7)
    {
      case 0 : sig_c_op    = LOW; break;
      case 1 : sig_c_out   = LOW; break;
      case 2 : sig_c_norin = LOW; break;
      case 3 : sig_c_datar = LOW; break;
      case 4 : sig_c_adr_l = LOW; break;
      case 5 : sig_c_adr_h = LOW; break;
      case 6 : sig_c_ram   = LOW; break;
      default: break;
    }
  }

  /* Address Decoders */
  /* U14A */
  sig_ckctl = HIGH;
  sig_oedatar = HIGH;
  if (!sig_clk && sig_phase)
  {
    if (!sig_ctl_dtsel) {
      sig_ckctl = LOW;
    } else {
      sig_oedatar = LOW;
    }
  }

  /* U14B : used to read the input port (sets Flip-Flop U1A when input is high) */
  sig_rdin = sig_ctl_insel;
  if (sig_ctl_insel == LOW)
  {
    uint8_t indata;
    DFF8_gate(&IC17_HC574_g, HIGH, HIGH, sig_input_port, &indata);
    DFF8_gate(&IC17_HC574_g, LOW,  LOW,  sig_input_port, &indata);

    if ((IC18_HC574_g.flipflops & 1) == 0)
      indata |= 1;  // mask the RXD-signal if TXD is low  (this is extra logic on the My4TH light board)

    sig_ctl_setnor_n = ((indata & 3) == 3) ? HIGH : LOW;  // combine RxD and SDA to one single signal
  }
  else
  {
    sig_ctl_setnor_n = HIGH;
  }


  sig_rdram = sig_ckctl & ((sig_address >> 15) & 1);
  sig_rdrom = ((!sig_phase) | !((sig_address>>15) | sig_ckctl)) ? LOW : HIGH;

  /* EPROM */
  sig_eprom_out = EPROM_g[sig_address & 0x7FFF];  // allow big EPROM with 32kB for emulating "My4TH light"
  if ((sig_rdrom == LOW) && (sig_reset_n == LOW))
    sig_eprom_out = 0x00;  // fix a simulation problem

  /* RAM */
  if (sig_address & 0x8000)  // A15 at CS2-pin
  {
    if (sig_rdram == LOW)
      sig_data = SRAM_g[sig_address & 0x1FFF];
    if (sig_c_ram && !SRAM_oe_g)
    {
      SRAM_g[sig_address & 0x7FFF] = sig_data;
  //if (((sig_address & 0x7FFF) >= 0x010)) // || (sig_address == 0x800A) || (sig_address == 0x800B))
  // printf("RAM 0x%04X := 0x%02X\n",sig_address,sig_data);
    }
  }
  SRAM_oe_g = sig_c_ram;

  /* DATA Register */
  if (!IC10_HC574_ck && sig_c_datar)
    IC10_HC574_g = (sig_data & ~1) | sig_norout;
  if (sig_oedatar == LOW)
    sig_data = IC10_HC574_g;
  IC10_HC574_ck = sig_c_datar;

  /* I/O */
  DFF8_gate(&IC18_HC574_g, LOW, sig_c_out, sig_data, &sig_output_port);

  if (iodisplay_g == iodisp_uarttiming)
  {
    /* check for falling edge of sig_rdin */
    if ((sig_rdin_d == HIGH) && (sig_rdin == LOW))
    {
      dif = exectime_g - uart_lastexec_g;
      uart_lastexec_g = exectime_g;
      printf("INPORT:  0x%02X  (%d Hz, %d us)\n", sig_data, (unsigned)(1000000000UL/dif), (unsigned)(dif/1000));
  //    printf("Read IO-INPUT: %ld Hz\n", 1000000000UL/dif);
    }
    sig_rdin_d = sig_rdin;
  }
}


/*===========================================================================*/


static inline void DFF8_gate(dff8gate_t *gate, signal_t nOE, signal_t CP, dbus_t in, dbus_t *out)
{
  if (!nOE)
    *out = gate->flipflops;

  if (CP && !gate->cp)
  {
    if (gate->incallback != NULL)
      in = (gate->incallback)();
    gate->flipflops = in;
    if (gate->callback != NULL)
      gate->callback((uint8_t)in);
  }
  gate->cp = CP;
}


/*---------------------------------------------------------------------------*/

extern void generate_my4th_microcode_eprom(uint8_t *buffer);
extern void generate_my4th_xs_microcode_eprom(uint8_t *buffer);
extern void generate_mynor_microcode_eprom(uint8_t *buffer);

void init_eprom(int ucode)
{
  memset(EPROM_g, 0xFF, 0x8000);
  memset(SRAM_g, 0x00, 0x8000);
  
  if (ucode == UC_MY4TH_XS) {
    generate_my4th_xs_microcode_eprom(EPROM_g);
  } else
  if (ucode == UC_MYNOR) {
    generate_mynor_microcode_eprom(EPROM_g);
  } else {
    generate_my4th_microcode_eprom(EPROM_g);
  }
}


/*---------------------------------------------------------------------------*/


/** Write data into the CPU memory space.
 *  @param addr  memory address: 0x0000-0x7FFF = EPROM, 0x8000-0xFFFF = RAM
 *  @param data  data byte to write
 */
void cpu_memory_write(uint16_t addr, uint8_t data)
{
  if (addr < 0x8000)
  {
    EPROM_g[addr] = data;
  }
  else
  {
    SRAM_g[addr-0x8000] = data;
  }
}


/** Read data out of the CPU space.
 *  @param addr  memory address: 0x0000-0x7FFF = EPROM, 0x8000-0xFFFF = RAM
 *  @return  data byte
 */
uint8_t cpu_memory_read(uint16_t addr)
{
  uint8_t data = 0;

  if (addr < 0x8000)
  {
    data = EPROM_g[addr];
  }
  else
  {
    data = SRAM_g[addr-0x8000];
  }
  return data;
}


/** Clock the CPU. Each call to this function simulates one clock cycle.
 */
void cpu_clock(void)
{
  exectime_g += clock_period_g;

  sig_clk = 0;
  pfn_dologic_g();
  pfn_dologic_g();
  if (enable_vcd_g)
    vcd_timestep();

  sig_clk = 1;
  pfn_dologic_g();
  pfn_dologic_g();
  if (enable_vcd_g)
    vcd_timestep();

  /* reset generator (R-C-circuit) */
  if (resettimer_g != 0) {
    if (--resettimer_g == 0) 
       sig_rcreset = HIGH;
  }
}



/** Enable the generation of a VCD file.
 *  @param vcdfile  name of the vcd file to generate or NULL
 */
void cpu_enable_vcd(char *vcdfile)
{
  // initialize data logger (signal recording into a .vcd file)
  if (!enable_vcd_g && (vcdfile != NULL))
  {
    enable_vcd_g = 1;

    vcd_add_wire("clk",           &sig_clk);
    vcd_add_wire("clk_d",         &sig_clk_d);
    vcd_add_wire("rcreset",       &sig_rcreset);
    vcd_add_wire("reset",         &sig_reset);
    vcd_add_wire("reset_n",       &sig_reset_n);
    vcd_add_abus("address",       &sig_address);
    vcd_add_dbus("data",          &sig_data);
    vcd_add_dbus("control",       &sig_control);
    vcd_add_dbus("ctrbus",        &sig_ctrbus);
    vcd_add_wire("phase",         &sig_phase);
    vcd_add_wire("ctrload_n",     &sig_ctrload_n);
    vcd_add_wire("load_ctr",      &sig_load_ctr);
    vcd_add_wire("oectr",         &sig_oectr);
    vcd_add_wire("oeadr",         &sig_oeadr);
    vcd_add_wire("norin",         &sig_norin);
    vcd_add_wire("norout",        &sig_norout);
    vcd_add_wire("oedatar",       &sig_oedatar);
    vcd_add_wire("fetch",         &sig_fetch);
    vcd_add_wire("c_op",          &sig_c_op);
    vcd_add_wire("c_out",         &sig_c_out);
    vcd_add_wire("c_norin",       &sig_c_norin);
    vcd_add_wire("c_datar",       &sig_c_datar);
    vcd_add_wire("c_adr_l",       &sig_c_adr_l);
    vcd_add_wire("c_adr_h",       &sig_c_adr_h);
    vcd_add_wire("c_ram",         &sig_c_ram);
    vcd_add_wire("ctl_dtsel",     &sig_ctl_dtsel);
    vcd_add_wire("ctl_tg0",       &sig_ctl_tg0);
    vcd_add_wire("ctl_tg1",       &sig_ctl_tg1);
    vcd_add_wire("ctl_tg2",       &sig_ctl_tg2);
    vcd_add_wire("ctl_asel",      &sig_ctl_asel);
    vcd_add_wire("ctl_ldctr",     &sig_ctl_ldctr);
    vcd_add_wire("ctl_resnor_n",  &sig_ctl_resnor_n);
    vcd_add_wire("ctl_insel",     &sig_ctl_insel);
    vcd_add_wire("ckctl",         &sig_ckctl);
    vcd_add_wire("rdram",         &sig_rdram);
    vcd_add_wire("rdrom",         &sig_rdrom);
    vcd_add_wire("rdin",          &sig_rdin);
    vcd_add_dbus("input_port",    &IC17_HC574_g.flipflops);
    vcd_add_dbus("output_port",   &sig_output_port);
    //vcd_add_dbus("eprom_out",     &sig_eprom_out);
    vcd_add_dbus("U5_Instruction",&IC5_HC574_g);
    vcd_add_dbus("U6_Addr-Low",   &IC6_HC574_g);
    vcd_add_dbus("U7_Addr-High",  &IC7_HC574_g);
    vcd_add_dbus("U10_DATAR",     &IC10_HC574_g);

    vcd_newfile(vcdfile, "My4TH CPU Simulation", clock_period_g/2);
  }
}


/** Initialize the CPU simulation.
 *  @param  ucode  can be UC_MY4TH or UC_MY4TH
 *                 Simulation of ARCH_MYNOR is not yet supported.
 */
void cpu_init(int ucode)
{
  if (ucode == UC_MY4TH_XS)
  {
    pfn_dologic_g = My4TH_XS_all_logic;
  }
  else
  {
    pfn_dologic_g = My4TH_all_logic;
  }

  // initialize gates:
  memset(&IC17_HC574_g, 0, sizeof(IC17_HC574_g));
  memset(&IC18_HC574_g, 0, sizeof(IC18_HC574_g));

  IC5_HC574_g = 0xFF;  // worst-case OP-code
  IC6_HC574_g = 0xFF;  // worst-case address
  IC7_HC574_g = 0xFF;  // worst-case address
  IC10_HC574_g= 0x6B;  // garbage data
  IC12_HC273_g= 0xFF;  // worst-case control word

  IC17_HC574_g.incallback = get_inport;
  IC18_HC574_g.callback = set_outport;

  init_eprom(ucode);
  
  enable_vcd_g = 0;
  sig_reset_n = LOW;
  resettimer_g = 500 / (clock_period_g/2);
}


/** Terminate the CPU simulation
 */
void cpu_term(void)
{
  if (enable_vcd_g)
    vcd_closefile();
}
