#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include "eeprom.h"


/* I2C addresses of possible EEPROM population:

  EEPROM Socket 1:
      24LC512 : 0xA0
      24LC1025: 0xA0 / 0xA8
      24LC1026: 0xA0 / 0xA2

  EEPROM Socket 2:
      24LC512 : 0xA6
      24LC1025: 0xA6 / 0xAE
      24LC1026: 0xA4 / 0xA6
*/

typedef enum {
  i2c_idle = 0,
  i2c_getaddr,
  i2c_addrack,
  i2c_addrnack,
  i2c_getdata,
  i2c_dataack,
  i2c_datanack,
  i2c_senddata,
} i2cstate_t;


typedef struct {
  uint8_t     devaddr;
  uint8_t     *mem;
  i2cstate_t  i2c_state;
  int         last_scl;
  int         last_sda;
  int         bitctr;
  int         mode;
  int         rxctr;
  uint8_t     rxdata;
  uint8_t     txdata;
  unsigned    mem_addr;
  int         pull_sda_low;
  int         changed;
} eeprom_t;


#define EEPROM_SIZE           0x10000
#define EEPROM_COUNT          4
#define OVERALL_EEPROM_SIZE   (EEPROM_COUNT * EEPROM_SIZE)
#define EEPROM_FILE_NAME      "my4th-eeprom.bin"

static eeprom_t     eeprom[EEPROM_COUNT];
static uint8_t      *eeprom_data_g = NULL;
static int          eeprom_inited_g = 0;

uint8_t eeprom_addresses_g[EEPROM_COUNT] = {0, };


static void new_eeprom(eeprom_t *eep, uint8_t devaddr, uint8_t *mem)
{
#if 0
  if (devaddr != 0)
    printf("New EEPROM, addr 0x%02X\n", devaddr);
#endif
  eep->mem = mem;
  eep->devaddr = devaddr;
  eep->i2c_state = i2c_idle;
  eep->last_scl = 1;
  eep->last_sda = 1;
  eep->bitctr = 0;
  eep->mode = 0;
  eep->rxctr = 0;
  eep->rxdata = 0;
  eep->txdata = 0;
  eep->mem_addr = 0;
  eep->pull_sda_low = 0;
  eep->changed = 0;
}


static void term_eeprom(void)
{
  FILE *f;
  int i, ch, rc;
  
  for (ch=0, i=0; i<EEPROM_COUNT; i++)
  {
    ch += eeprom[i].changed;
  }
  
  if (ch)
  {
    f = fopen(EEPROM_FILE_NAME, "w+b");
    if (f != NULL)
    {
      rc = fwrite(eeprom_data_g, 1, OVERALL_EEPROM_SIZE, f);
      fclose(f);
      if (rc != OVERALL_EEPROM_SIZE)
        printf("Error: Failed to write data to file " EEPROM_FILE_NAME "\n");
    }
  }

  free(eeprom_data_g);
}


static void init_eeprom(void)
{
  FILE *f;
  int i, rc;
  
  if (eeprom_inited_g)
    return;
  
  eeprom_data_g = (uint8_t*)malloc(OVERALL_EEPROM_SIZE);
  if (eeprom_data_g == NULL)
  {
    fprintf(stderr, "Error: Failed to allocate memory\n");
    exit(1);
  }

  memset(eeprom_data_g, 0xFF, OVERALL_EEPROM_SIZE);
  atexit(term_eeprom);
  eeprom_inited_g = 1;

  f = fopen(EEPROM_FILE_NAME, "rb");
  if (f != NULL)
  {
    rc = fread(eeprom_data_g, 1, OVERALL_EEPROM_SIZE, f);
    fclose(f);
    if (rc != OVERALL_EEPROM_SIZE)
      printf("Error: Failed to read data from file " EEPROM_FILE_NAME "\n");
  }

  // Setup 24LC512 EEPROMs
  for (i=0; i<EEPROM_COUNT; i++)
  {
    new_eeprom(&eeprom[i], eeprom_addresses_g[i], eeprom_data_g + (i*EEPROM_SIZE));
  }
}


static void simulate_eeprom(eeprom_t *eep, int *scl, int *sda)
{
  int valid_rx = 0;

  if (eep->devaddr == 0)
    return;

  if (eep->last_scl && !*scl && !eep->mode)
    eep->pull_sda_low = 0;
  
  if (eep->last_sda && eep->last_scl && !*sda && *scl)
  {
    /* start condition detected */
    eep->i2c_state = i2c_getaddr;
    eep->bitctr = 0;
  }
  else
  if (!eep->last_sda && eep->last_scl && *sda && *scl)
  {
    /* stop condition detected */
    eep->i2c_state = i2c_idle;
  }
  else
  {
    if (!eep->last_scl && *scl)
    {
      eep->rxdata <<= 1;
      eep->rxdata |= *sda & 1;
    }
    
    if (!eep->last_scl && *scl)
    {
      eep->pull_sda_low = 0;
      
      switch (eep->i2c_state)
      {
        case i2c_getaddr:
        {
          if (++eep->bitctr == 8)
          {
            eep->i2c_state = ((eep->rxdata & 0xFE) == eep->devaddr) && (eep->devaddr != 0) ? i2c_addrack : i2c_addrnack;
            eep->mode = ((int)eep->rxdata) & 1;
          }
          break;
        }
        
        case i2c_addrnack:
        {
          /* do nothing until stop-condition is received */
          break;
        }
        
        case i2c_addrack:
        {
          eep->pull_sda_low = 1;
          eep->bitctr = 0;
          eep->rxctr = 0;
          if (!eep->mode) {
            eep->i2c_state = i2c_getdata;
          } else {
            eep->i2c_state = i2c_senddata;
            eep->bitctr = 0;
            eep->txdata = eep->mem[eep->mem_addr];
          }
          break;
        }
        
        case i2c_getdata:
        {
          if (++eep->bitctr == 8)
          {
            eep->i2c_state = i2c_dataack;
            valid_rx = 1;
          }
          break;
        }
        
        case i2c_dataack:
        {
          eep->pull_sda_low = 1;
          eep->bitctr = 0;
          eep->i2c_state = i2c_getdata;
          break;
        }
        
        case i2c_senddata:
        {
          if (++eep->bitctr <= 8)
          {
            if ((eep->txdata & 0x80) == 0)
              eep->pull_sda_low = 1;
            eep->txdata <<= 1;
          }
          else
          {
            eep->mem_addr = (eep->mem_addr + 1) & 0xFFFF;
            eep->txdata = eep->mem[eep->mem_addr];
            eep->bitctr = 0;
            if (*sda != 0)
              eep->i2c_state = i2c_idle;
          }
        }
        
        default: break;
      }
    }
  }
  
  /* write data into EEPROM address register and memory */
  if (valid_rx)
  {
    if (eep->rxctr < 2)
    {
      eep->mem_addr = (eep->mem_addr << 8) & 0xFF00;
      eep->mem_addr += eep->rxdata;
    }
    else
    {
      eep->mem[eep->mem_addr] = eep->rxdata;
      eep->mem_addr = (eep->mem_addr + 1) & 0xFFFF;
      eep->changed = 1;
    }
    eep->rxctr++;
  }
  
  if (eep->pull_sda_low)
    *sda = 0;

  eep->last_scl = *scl;
  eep->last_sda = *sda;
}


/*---------------------------------------------------------------------------*/


/** Simulate EEPROMs
 *  @param scl  ptr to SCL wire variable
 *  @param sda  ptr to SDA wire variable
 *  @note This function must be called when ever
 *        the SCL or the SDA wire changes its state.
 */
void i2c_eeprom(int *scl, int *sda)
{
  int i;

  init_eeprom();
  
  // Simulate the 24LC512 EEPROMs
  for (i=0; i<EEPROM_COUNT; i++)
  {
    simulate_eeprom(&eeprom[i], scl, sda);
  }
}


/** Configure the two EEPROMs on the Board.
 *  @param socket1 :  String "24LC512" / "24LC1025" / "24LC1026"
 *  @param socket2 :  String "24LC512" / "24LC1025" / "24LC1026"
 *  @return zero on success
 */
int i2c_eeprom_setup(char *socket1, char *socket2)
{
  char *s;
  int i;
  
  for (i=0; i<EEPROM_COUNT; i++)
  {
    eeprom_addresses_g[i] = 0;
  }
  
  for (s = socket1, i=0; i<2; i++, s = socket2)
  {
    if (!strcmp(s, "24LC512") || !strcmp(s, "24lc512") || !strcmp(s, "512"))
    {
      eeprom_addresses_g[2*i+0] = !i ? 0xA0 : 0xA6;
      eeprom_addresses_g[2*i+1] = 0;
    }
    else
    if (!strcmp(s, "24LC1025") || !strcmp(s, "24lc1025") || !strcmp(s, "1025"))
    {
      eeprom_addresses_g[2*i+0] = !i ? 0xA0 : 0xA6;
      eeprom_addresses_g[2*i+1] = !i ? 0xA8 : 0xAE;
    }
    else
    if (!strcmp(s, "24LC1026") || !strcmp(s, "24lc1026") || !strcmp(s, "1026"))
    {
      eeprom_addresses_g[2*i+0] = !i ? 0xA0 : 0xA4;
      eeprom_addresses_g[2*i+1] = !i ? 0xA2 : 0xA6;
    }
    else
    {
      if ((*s != 0) && (strcmp(s, "none") != 0))
        return -1;
    }
    
    if (eeprom_addresses_g[1] == 0)
    {
      eeprom_addresses_g[1] = eeprom_addresses_g[2];
      eeprom_addresses_g[2] = eeprom_addresses_g[3];
      eeprom_addresses_g[3] = 0;
    }
  }
  
#if EEPROM_COUNT > 4
  eeprom_addresses_g[5] = 0xAA;
  eeprom_addresses_g[6] = 0xAC;
#endif
  return 0;
}

