/*
 *   Format FORTH strings and screens
 */

#ifndef _FORTHFMT_H
#define _FORTHFMT_H

#define FFMT_RUN    (1<<0)
#define FFMT_SCRFMT (1<<2)


/** This function removes unnecessary characters from a string:
 *  leading and trailing spaces, non-printable characters.
 *  @param s  ptr to the string
 */
void ffmt_cleanup_string(char *s);


/** This function removes spaces and control characters
 *  from the end of a line.
 *  @param s    ptr to the string
 *  @param len  string length
 *  @return final string length
 */
int ffmt_remove_trailing_whitespaces(char *s, int len);


/** Make all characters in a buffer printable.
 *  @param s    ptr to the string
 *  @param len  string length
 */
void ffmt_make_printable(char *s, int len);


/** Check if a string contains a line that is used
 *  for an mynor-tool internal comment.
 *  @param s    ptr to the string
 *  @return nonzero if the line is a comment.
 */
int ffmt_is_internal_comment_line(char *s);


/** Initialize the screen parser
 *  @param filename   name of the source file, used for error messages
 */
void ffmt_screen_parser_init(char *filename);


/** Parses a FORTH line and writes it to the screen buffer.
 *  @param string   a string read from a file
 *  @param scrbuf   output screen buffer
 *  @param used     this flag is set to 1 when the screen gets used
 *  @return 0 on success, 1 if the screen is finished and
 *          can be further processed -1 if an error occured
 */
int ffmt_screen_parse_line(char *string, uint8_t *scrbuf, int *used);


/** Load and compress a FORTH source code text file.
 *  Make lines no longer than 64 characters. Remove
 *  comments where necessary and possible.
 *  @param filename   name of the text file
 *  @param flags  Optional. If not NULL, this variable
 *                will be set to some flags  (see FFMT_xxx defines)
 *                FFMT_RUN    : the word "run" was found in the text
 *                FFMT_SCRFMT : the source code is preformatted screen code
 *  @return ptr to "compressed" text data.
 *          NULL is returned when an error occured.
 *          This function prints also the corresponding
 *          error message.
 *          The user is responsible for freeing the returned
 *          memory blob by calling the free() - function.
 */
char* ffmt_load_and_compress_file(const char *filename, int *flags);


#endif /* _FORTHFMT_H */
