#include <stdio.h>
#include "input.h"
#include "ioports.h"
#include "eeprom.h"
#include "cpu.h"  // for exectime_g


#define I2C_SDA   0x02   // inport and outport
#define I2C_SCL   0x04   // outport


iodisplay_t  iodisplay_g = iodisp_inline;
static uint8_t inport_g  = 0xFF;
static uint8_t outport_g = 0;

static uint8_t inputkey_g = 0;
static int     clrcounter_g = 0;
static int     simulateLCDBoard_g = 0;   // enable the simulation of the LCD board

static int     i2c_scl_g = 1;
static int     i2c_sda_g = 1;



static char *binstr(uint8_t value)
{
  static char str[9];
  for (int i=0; i<8; i++)
    str[i] = ((value & (0x80>>i)) ? '1' : '0');
  str[8] = 0;
  return str;
}


static void print_ports(void)
{
  printf("OUT: 0x%02X %s    ", outport_g, binstr(outport_g));
  printf("IN : 0x%02X ", inport_g);
  fflush(stdout);
}


extern uint64_t uart_lastexec_g;


/** Set new value for output port 3
 *  @param value  new port data value
 */
void set_outport(uint8_t value)
{
  static int outenabled = 0;
  static uint8_t outport_last = 0x00;
  uint64_t dif;
  
  outport_g = value;
  if (outenabled)
  {
    if (!simulateLCDBoard_g)
    {
      if (iodisplay_g == iodisp_inline) {
        print_ports();
      }
      else 
      if ((iodisplay_g == iodisp_simple) || (iodisplay_g == iodisp_uarttiming)) {
        dif = exectime_g - uart_lastexec_g;
        uart_lastexec_g = exectime_g;
        //printf("OUTPORT:  0x%02X  %s  (%d Hz)\n", value, binstr(value), (unsigned)(1000000000UL/dif));
        printf("OUTPORT:  0x%02X  %s  (%d Hz, %d us)\n", value, binstr(value), (unsigned)(1000000000UL/dif), (unsigned)(dif/1000));
      }
    }
    
    /* I2C-Bus */
    if ((outport_g ^ outport_last) & (I2C_SCL|I2C_SDA))
    {
      outport_last = outport_g;
    
      i2c_scl_g = (outport_g & I2C_SCL) ? 1 : 0;
      i2c_sda_g = (outport_g & I2C_SDA) ? 1 : 0;
      
      i2c_eeprom(&i2c_scl_g, &i2c_sda_g);
    }
    
#ifdef TEST_SPI
    spi_receiver(value);
#endif
  }
  outenabled = 1;
}


/** Get value from input port
 */
uint8_t get_inport(void)
{
  uint8_t mask = 0xFF;
//  uint64_t dif;

  if (iodisplay_g == iodisp_inline)
  {
    inport_g = inputkey_g;
  }
  else
/*
  if (iodisplay_g == iodisp_simple)
  {
    inport_g = inputkey_g;
    dif = exectime_g - uart_lastexec_g;
    uart_lastexec_g = exectime_g;
    printf("INPORT:  0x%02X  %s  (%d Hz)\n", inport_g, binstr(inport_g), (unsigned)(1000000000UL/dif));
  }
  else
*/
  {
    /* simulate read-back of SDA-line */
    if (i2c_sda_g == 0) {
      inport_g &= ~I2C_SDA;
    } else {
      inport_g |= I2C_SDA;
    }
  }

  return inport_g & mask;
}


/** Get the level of the TxD signal at Output Port 1
 *  @return TxD signal level in bit0
 */
uint8_t get_txd(void)
{
  return outport_g & 1;
}


/** Set the level of the RxD signal at Input Port 1
 *  @param level  new RxD input level at bit0
 */
void set_rxd(uint8_t level)
{
  inport_g = (inport_g & ~1) | (level & 1);
}


/** This function must be called periodically.
 *  @param readkeyboard   set to true to read keyboard as input for Input Port 1
 */
void ioports_poll(uint8_t readkeyboard)
{
  if (inputkey_g == 0)
  {
    if (readkeyboard)
    {
      if (keyboard_hit())
      {
        inputkey_g = (uint8_t)keyboard_getch();
        clrcounter_g = 5;
      }
    }
  }
  else
  if (clrcounter_g != 0)
  {
    if (--clrcounter_g == 0)
      inputkey_g = 0;
  }
}


/** Enable or disable the simulation of the 4x20 LCD display
 *  extension board and 3x5 keypad.
 *  @param enable   1:enable, 0:disable LCD simulation
 */
void ioports_enable_LCD(int enable)
{
  simulateLCDBoard_g = enable;
}


/** Initialize IO-Ports
 *  @param display    type of display
 */
void ioports_init(iodisplay_t display)
{
  iodisplay_g = display;

  if (iodisplay_g == iodisp_inline)
    printf("\n");
}


/** Terminate IO-Ports
 */
void ioports_term(void)
{
  if (iodisplay_g == iodisp_inline)
    printf("\n\n");
}
