#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "opsystem.h"
#include "input.h"
#include "cpu.h"
#include "uart.h"
#include "ioports.h"
#include "eeprom.h"
#include "simulation.h"
#include "timing.h"
#include "xfer-uart.h"
#include "transfer.h"
#include "my4th_microcode.h"
#include "forthfmt.h"
#include "version.h"
#ifdef _LINUX
#include <signal.h>
#endif

static char vcd_file_g[500];
static int  option_vcd = 0;
static int  simulation_clocks_g = 0;
static int  opt_terminal = 0;
static int  opt_lcd = 0;
static int  run_app_flag_g = 0;
static volatile int stop_simulation_g = 0;
#define DEFAULT_CLOCK_FREQ 16
int clock_freq_g = DEFAULT_CLOCK_FREQ;
int clock_period_g = 1000L/DEFAULT_CLOCK_FREQ;

#define SIM_BAUDRATE  default_baudrate_g

static int uart_byte_received(uint8_t b)
{
  if ((b != 0x00) && (b != 0xE0))
  {
    if (((b < 0x20) || (b >= 0x80)) && (b != 0x0A) && (b != 0x0D) && (b != 27))
      b = '?';

    printf("%c", (char)b);
    fflush(stdout);
  }
  return 1;
}


void run_cpu(void)
{
  int byte_pending = -1;
  int clocks, ioctr=100;
  int div10 = 0;
  int uarttickctr = 0;
  int millisecctr = 0;
  int twohertzctr = 0;
  int kbctr = -4500;
  int startctr = 0;
  int ch;
  unsigned elapsed_ns = 0;

  for (clocks = 0; (clocks < simulation_clocks_g) || (simulation_clocks_g <= 0); clocks++)
  {
    cpu_clock();
    elapsed_ns += clock_period_g;
    if (elapsed_ns >= TIMING_GRID)
    {
      elapsed_ns -= TIMING_GRID;
      sleep_until_next_timing_grid();
    }

    if (++div10 >= 10)
    {
      div10 = 0;

      if (--ioctr == 0)
      {
        ioctr = 5000;
        ioports_poll( opt_terminal ? 0:1 );
      }

      /* dirty trick: change PC to execute the application at 0x8200 */
      if (run_app_flag_g)
      {
        if (++startctr > 70)
        {
          run_app_flag_g = 0;
          cpu_memory_write(0x8000+SP,   0x00);  // reset stack pointer
          cpu_memory_write(0x8000+PC_L, 0x00);  // set execution address, low
          cpu_memory_write(0x8000+PC_H, 0x82);  // set execution address, high
        }
      }
    }

    #define UARTTICKCMP   ((1000000000UL/clock_period_g+(SIM_BAUDRATE*2))/(SIM_BAUDRATE*4))
    if (++uarttickctr >= UARTTICKCMP)
    {
      /* The UART baud rate shall be 4800 baud.
       * The tick rate here should be 19200 Hz (4x baud rate).
       */
      uarttickctr = 0;
      if ((byte_pending >= 0) && (kbctr >= 0))
      {
        kbctr = 20;
        if (uart_sendbyte((uint8_t)byte_pending)) 
          byte_pending = -1;
      }
      else
      if (++kbctr > 20)
      {
        kbctr = 0;
        if (keyboard_hit())
        {
          ch = keyboard_getch();
          //printf("CH: %02X\n", ch);          
          
          if (ch == 10)
            ch = 13;
          
          if ((uint8_t)ch == 0xF6)  // "รถ"
          {
            kbctr = -9600;  // wait 1 second
          }
          else
          if ((ch > 0) && (ch < 256))
          {
            if (opt_terminal)
            {
              if (!uart_sendbyte((uint8_t)ch)) 
                byte_pending = ch;
            }
          }
        }
      }

      if (opt_terminal)
        uart_timetick();
      
      if (twohertzctr == 0)
      {
        /* 2 Hz here */
        twohertzctr = 9592/2-4;
      }
      twohertzctr--;

      if (++millisecctr >= 9)
      {
        /* approx. 1 ms time tick here */
        millisecctr = 0;
      }
      
      if (stop_simulation_g)
      {
        printf("\n");
        break;
      }
      
    }
  }
}


void help(void)
{
  printf("\nMy4TH swiss-knife tool " VER_VERSION ", (c) by Dennis Kuschel in " VER_YEAR ", " VER_URL "\n");
  printf("\nUsage: my4th [OPTION]\n\n");

  printf("  read|write|exec         Transfer data between the My4TH computer and this PC.\n");
  printf("                          To get help about the read function, for example,\n");
  printf("                          enter:   $ my4th read --help\n");
  printf("  --vcd filename          Simulate real hardware and write the resulting\n");
  printf("                          waveform into a .vcd file that can be displayed\n");
  printf("                          with GTKWave ( http://gtkwave.sourceforge.net )\n");
  printf("  -c, --clocks nmbr       Simulate only nmbr of clock cycles and stop then.\n");
  printf("                          Set to 0 to simulate endlessly (default)\n");
  printf("  -f, --freq mhz          Set CPU clock frequency to mhz MHz (integer value)\n");
  printf("  -b, --baud num          Set default baud rate to num baud (integer value)\n");
  printf("  -t, --terminal          Enable terminal emulation over UART\n");
  printf("  -e socket1[,socket2]    Configure EEPROMs. Default is 24LC1026,24LC1026\n");
  printf("  -r [addr,]filename      Read binary file into memory. addr=memory address.\n");
  printf("  -w from,to,filename     Write memory content (EPROM and RAM) into a file and\n");
  printf("                          quit the program before CPU simulation starts.\n");
  printf("                          from=start address, to=end address (range 0x0-0x9FFF)\n");
  printf("  -wr from,to,filename    Like option -w above, but run CPU simulation before\n");
  printf("                          the memory dump is written to the specified file.\n");
  printf("  -d, --display fmt       Display format of output formats. fmt can be 'simple'\n");
  printf("                          for one line per write to IO-port, 'inline' for one\n");
  printf("                          line containing all ports (without scrolling), or\n");
  printf("                          'timing' to debug UART timing.\n");
  printf("   --uc-mynor             Create microcode for MyNOR instead for My4TH\n");
  printf("   -x, --uc-xs            Create microcode for My4TH XS, and simulate My4TH XS\n");
  printf("   --init value           Initializes the RAM with the given value (default 0)\n");
#ifdef _LINUX
  printf("  -p inpipe,outpipe       Connect My4TH UART simulation to filesystem pipes\n");
#endif
  printf("\n");
}


static void quit_with_error(char *errormsg)
{
  fprintf(stderr, "%s", errormsg);
  exit(1);
}


void check_for_argument(int argc, int i, char *errormsg)
{
  if (i >= argc)
    quit_with_error(errormsg);
}


static int write_memorydump(unsigned start, unsigned end, char *filename)
{
  FILE *f;
  unsigned i;
  
  f = fopen(filename, "w+b");
  if (f == NULL)
    quit_with_error("Failed to write memory dump to file\n");

  for (i=start; i<=end; i++)
  {
    fputc(cpu_memory_read((uint16_t)i), f);
  }
  
  fclose(f);
  printf("Memory dump (range 0x%04X-0x%04X) successfully written to file %s\n", start, end, filename);
  return 0;
}


static int read_memory(unsigned start, const char *filename)
{
  uint8_t *buf;
  unsigned i, cnt, end = start;
  int ret;
  FILE *f;
  
  buf = (uint8_t*)malloc(0x10000);
  if (!buf)
    quit_with_error("Out of Memory Error\n");

  f = fopen(filename, "rb");
  if (f == NULL)
  {
    fprintf(stderr, "File not found: %s\n", filename);
    return 1;
  }
  ret = fread(buf, 1, 0x10000, f);
  fclose(f);
  if (ret < 0)
  {
    quit_with_error("Failed to read file into memory\n");
  }
  cnt = (unsigned)ret;

  for (i=0; i<cnt; i++)
  {
    end = start+i;
    cpu_memory_write((uint16_t)end, buf[i]);
  }
  
  free(buf);
  printf("File %s loaded into memory (range 0x%04X-0x%04X)\n", filename, start, end);
  return 0;
}



static int getNumberFromString(char **str, int *nbr)
{
  char *s = *str;
  if (*s == '$') {
    s++;
    if (sscanf(s, "%X", nbr) != 1)
      return 0;
  } else
  if ((s[0] == '0') && ((s[1] == 'X') || (s[1] == 'x'))) {
    s+=2;
    if (sscanf(s, "%X", nbr) != 1)
      return 0;
  } else {
    if (sscanf(s, "%X", nbr) != 1)
      return 0;
  }
  while (*s && (*s!=',')) s++;
  if (*s == ',') s++;
  *str = s;
  return 1;
}


#ifdef _LINUX
void handle_sigint(int sig)
{
  stop_simulation_g = 1;
}
#endif


int main(int argc, char *argv[])
{
  char rfilename[500];
  char wfilename[500];
  char *param, *romfile;
  int rstart=0, wstart=0, wend=0;
  int code_entry = CODE_START; // default for My4TH and MyNOR
  int opt_write = 0;
  int opt_read = 0;
  int opt_display = 0;
  int opt_clocks = 0;
  iodisplay_t dispfmt = iodisp_simple;
  int i, rc;
  
  if (argc > 1)
  {
    if (!strcmp(argv[1], "read") || !strcmp(argv[1], "write") ||
        !strcmp(argv[1], "exec") || !strcmp(argv[1], "term" ) || 
        !strcmp(argv[1], "--help"))
      return transfer_main(argc, argv);
    
    if (!strcmp(argv[1], "--help-sim"))
    {
      help();
      return 0;
    }
  }
  
  if (argc < 2)
  {
    printf("\nMy4TH swiss-knife tool " VER_VERSION ", (c) by Dennis Kuschel in " VER_YEAR ", " VER_URL "\n");
    printf("\nUsage: my4th [read|write|exec|term] [options]\n\n");
    printf("For help with data transfer options, enter  'my4th --help'\n");
    printf("For help with My4TH CPU simulation options, enter  'my4th --help-sim'\n\n");
    return 0;
  }

  // The default platform is My4TH
  cpu_init(UC_MY4TH);
  romfile = "my4th-rom.bin";
  
  i2c_eeprom_setup("24LC1026","24LC1026");
  
  for (i=1; i<argc; i++)
  {
    param = argv[i];
    if (!strcmp(param, "--help") || !strcmp(param, "-h"))
    {
      help();
      return 0;
    }
    else
    if (!strcmp(param, "--terminal") || !strcmp(param, "-t"))
    {
      opt_terminal = 1;
    }
    else
    if (!strcmp(param, "--display") || !strcmp(param, "-d"))
    {
      check_for_argument(argc, ++i, "ERROR: Parameter --display is missing a format\n");
      if (!strcmp(argv[i], "simple")) {
        dispfmt = iodisp_simple;
      } else
      if (!strcmp(argv[i], "inline")) {
        dispfmt = iodisp_inline;
      } else
      if (!strcmp(argv[i], "timing")) {
        dispfmt = iodisp_uarttiming;
      } else {
        quit_with_error("Unknown display format in option --display\n");
      }
      opt_display = 1;
    }
    else
    if (!strcmp(param, "--init"))
    {
      check_for_argument(argc, ++i, "ERROR: Parameter --init is missing a 8-bit value\n");
      int j, iv;
      if (sscanf(argv[i], "0x%X", &iv) != 1) {
        if (sscanf(argv[i], "0x%x", &iv) != 1) {
           iv = atoi(argv[i]);
        }
      }
      if ((iv < 0) || (iv > 255))
        quit_with_error("RAM initialization value must be in the range 0 - 255\n");
      for (j=0; j<0x8000; j++) {
        cpu_memory_write((uint16_t)(j+0x8000), (uint8_t)iv);
      }
    }
    else
    if (!strcmp(param, "--clocks") || !strcmp(param, "-c"))
    {
      check_for_argument(argc, ++i, "ERROR: Parameter --clocks is missing a number\n");
      simulation_clocks_g = atoi(argv[i]);
      opt_clocks = 1;
    }
    else
    if (!strcmp(param, "--freq") || !strcmp(param, "-f"))
    {
      check_for_argument(argc, ++i, "ERROR: Parameter --freq is missing a number\n");
      clock_freq_g = atoi(argv[i]);
      if ((clock_freq_g < 1) || (clock_freq_g > 40))
        quit_with_error("ERROR: Illegal clock frequency set, valid range 1 - 40 MHz\n");
    }
    else
    if (!strcmp(param, "--baud") || !strcmp(param, "-b"))
    {
      check_for_argument(argc, ++i, "ERROR: Parameter --baud is missing a number\n");
      default_baudrate_g = atoi(argv[i]);
      fast_baudrate_g = default_baudrate_g;
      if ((default_baudrate_g < 1) || (default_baudrate_g > 2000000))
        quit_with_error("ERROR: Illegal baud set, valid range 1 - 2000000\n");
    }    
    else
    if (!strcmp(param, "--vcd"))
    {
      check_for_argument(argc, ++i, "ERROR: Parameter --vcd is missing a filename\n");
      strncpy(vcd_file_g, argv[i], sizeof(vcd_file_g));
      vcd_file_g[sizeof(vcd_file_g)-1] = 0;
      option_vcd = 1;
    }
    else
    if (!strcmp(param, "-w"))
    {
      check_for_argument(argc, ++i, "ERROR: Parameter -w is missing an argument\n");
      param = argv[i];
      rc = getNumberFromString(&param, &wstart);
      if (rc == 1)
        rc = getNumberFromString(&param, &wend);
      if (rc == 1) {
        strcpy(wfilename, param);
        if (!*wfilename) rc = 0;
      }
      if ((rc != 1) || (wstart < CPU_MIN_MEMORY_ADDR) || (wstart > CPU_MAX_MEMORY_ADDR) ||
         (wend < CPU_MIN_MEMORY_ADDR) || (wend > CPU_MAX_MEMORY_ADDR) || (wstart > wend)) {
        quit_with_error("ERROR: -w has invalid parameters\n");
      }
      opt_write = 1;
    }
    else
    if (!strcmp(param, "-wr"))
    {
      check_for_argument(argc, ++i, "ERROR: Parameter -wr is missing an argument\n");
      param = argv[i];
      rc = getNumberFromString(&param, &wstart);
      if (rc == 1)
        rc = getNumberFromString(&param, &wend);
      if (rc == 1) {
        strcpy(wfilename, param);
        if (!*wfilename) rc = 0;
      }
      if ((rc != 1) || (wstart < CPU_MIN_MEMORY_ADDR) || (wstart > CPU_MAX_MEMORY_ADDR) ||
         (wend < CPU_MIN_MEMORY_ADDR) || (wend > CPU_MAX_MEMORY_ADDR) || (wstart > wend)) {
        quit_with_error("ERROR: -wr has invalid parameters\n");
      }
      opt_write = 2;
    }
    else
    if (!strcmp(param, "-r"))
    {
      char *s;
      check_for_argument(argc, ++i, "ERROR: Parameter -r is missing an argument\n");
      s = param = argv[i];
      while ((*s != 0) && (*s != ',')) s++;
      rstart = code_entry;
      rc = 1;
      if (*s == ',')
        rc = getNumberFromString(&param, &rstart);
      *rfilename = 0;
      if (rc == 1) {
        strcpy(rfilename, param);
        if (!*rfilename) rc = 0;
      }
      if ((rc != 1) || (rstart < CPU_MIN_MEMORY_ADDR) || (rstart > CPU_MAX_MEMORY_ADDR))
        quit_with_error("ERROR: -r has invalid parameters\n");
      if (*rfilename != 0)
        read_memory((uint16_t)rstart, rfilename);
      opt_read = 1;
    }
    else
    if (!strcmp(param, "-e"))
    {
      char argbuf[100];
      char *s1, *s2;
      check_for_argument(argc, ++i, "ERROR: Parameter -e is missing an argument\n");
      strncpy(argbuf, argv[i], sizeof(argbuf)-1);
      s1 = s2 = argbuf;
      while ((*s2 != 0) && (*s2 != ',')) s2++;
      if (*s2 == ',')
        *s2++ = 0;
      if (i2c_eeprom_setup(s1, s2))
      {
        fprintf(stderr, "Invalid EEPROM name in parameter -e.\nValid names are 24LC512, 24LC1025 and 24LC1025.\n");
        return 1;
      }
    }
    else
    if (!strcmp(param, "--uc-mynor"))
    {
      romfile = "mynor-rom.bin";
      cpu_init(UC_MYNOR);
    }
    else
    if (!strcmp(param, "--uc-xs") || !strcmp(param, "-x"))
    {
      romfile = "my4th-xs-rom.bin";
      code_entry = 0x1600;
      cpu_init(UC_MY4TH_XS);
    }
#ifdef _LINUX
    else
    if (!strcmp(param, "-p"))
    {
      char buf[500], *inp, *outp;
      check_for_argument(argc, ++i, "ERROR: Parameter -p is missing an argument\n");
      strncpy(buf, argv[i], 499);
      inp = outp = buf;
      while (*outp != 0)
      {
        if (*outp == ',')
        {
          *outp++ = 0;
          break;
        }
        outp++;
      }
      if (!*outp)
        quit_with_error("ERROR: -p is missing the name of the output pipe\n");
     if (uart_open_pipe(inp, outp) != 0)
       quit_with_error("ERROR: -p: failed to open the pipes\n");
    }
#endif
    else
    {
      fprintf(stderr, "Illegal parameter: %s\n", param);
      fprintf(stderr, "Enter 'my4th --help' to get help\n");
      return 1;
    }
  }

  
  clock_period_g = 1000L/clock_freq_g;
  
  if (run_app_flag_g)
  {
    if (!opt_display && !opt_terminal && !opt_lcd)
      opt_terminal = 1;
  }
  
  if (opt_display || opt_terminal || opt_lcd || option_vcd)
  {
    if (!opt_read)
    {
      FILE *f = fopen(romfile, "rb");
      int fs = 0;
      if (f != NULL)
      {
        fseek(f, 0, SEEK_END);
        fs = ftell(f);
        fclose(f);
      }
      if (read_memory((fs == 0x8000) ? 0x0000 : code_entry, romfile) != 0)
      {
        printf("\nNOTE:\n");
        printf("To enable My4TH CPU simulation, you must copy a My4TH EPROM image file\n");
        printf("for 8 MHz operation to this directory and rename it to '%s'.\n\n", romfile);
        return 1;
      }
    }
  }
  else
  {
    if (!opt_read && !opt_write)
    {
      printf("Missing options, please use option --help-sim to display help!\n");
      return 1;
    }
  }

  if (option_vcd && !opt_clocks)
    simulation_clocks_g = 2000;

  if (option_vcd && (*vcd_file_g != 0))
    cpu_enable_vcd(vcd_file_g);


  if (opt_write == 1)
  {
    cpu_term();
    return write_memorydump((uint16_t)wstart, (uint16_t)wend, wfilename);
  }
  
  if (opt_terminal)
  {
    uart_register_callback(uart_byte_received);
    
    if (!opt_display)
      dispfmt = iodisp_none;
  }

  ioports_init(dispfmt);
  
#ifdef _LINUX
  signal(SIGINT, handle_sigint);
#endif

  run_cpu();
  ioports_term();

  if (opt_write == 2)
    write_memorydump((uint16_t)wstart, (uint16_t)wend, wfilename);

  cpu_term();
  return 0;
}
