#include <stdio.h>
#include <string.h>
#include "my4th-xs_microcode.h"


/* MACRO: Load one operand from memory into TEMP3, continue in next ucode */
#define LOAD_OPERAND( nextuc )       /* call subroutine, load 1 operand             */ \
    WRITE_ADDRH( ZEROPAGE_HI ),      /* select zero-page                            */ \
    WRITE_ADDRL( RADR1 ),            /* select RADR1 register for next write        */ \
    WRITE_DATAR_CONST(nextuc),       /* set constant value for next memory write    */ \
    WRITE_RAM( READ_DATAR ),         /* RADR1 := nextuc                             */ \
    WRITE_ADDRL( JUMP(7) ),          /* write jump-target-address in ADDRL          */ \
    JUMP_ALWAYS,                     /* perform the jump to last micro-instruction  */ \
    WRITE_NOP,                       /* this NOP is never executed                  */ \
    WRITE_OPCODE( UC_1OPR )          /* call subroutine to load 2 operands          */ \
    // counter roll-over, continue with next OP

/* MACRO: Load two operands from memory into TEMP3 and TEMP2, continue in next ucode */
#define LOAD_2_OPERANDS( nextuc )    /* call subroutine, load 2 operands            */ \
    WRITE_ADDRH( ZEROPAGE_HI ),      /* select zero-page                            */ \
    WRITE_ADDRL( RADR1 ),            /* select RADR1 register for next write        */ \
    WRITE_DATAR_CONST(nextuc),       /* set constant value for next memory write    */ \
    WRITE_RAM( READ_DATAR ),         /* RADR1 := nextuc                             */ \
    WRITE_ADDRL( JUMP(7) ),          /* write jump-target-address in ADDRL          */ \
    JUMP_ALWAYS,                     /* perform the jump to last micro-instruction  */ \
    WRITE_NOP,                       /* this NOP is never executed                  */ \
    WRITE_OPCODE( UC_2OPR )          /* call subroutine to load 2 operands          */ \
    // counter roll-over, continue with next OP

/* MACRO: Decode SAP write address */
#define SAP_DECODE( uc1, uc2, uc3 )  \
    WRITE_ADDRL( READ_DATAR ),       /* rotate left                                 */ \
    WRITE_DATAR_0( READ_MEMORY ),    /* perform table-lookup, bit0 = PTR_H.x ^ 1    */ \
    WRITE_NORIN( READ_DATAR ),       /* use bit0 for jump decision                  */ \
    WRITE_ADDRL( JUMP(7) ),          /* write jump-target-address in ADDRL          */ \
    JUMP_IF_ZERO,                    /* perform the jump to uc1 if (PTR_H.x == 1)   */ \
    WRITE_NOP,                       /* else jump to uc3                            */ \
    WRITE_OPCODE(uc2),  \
    WRITE_OPCODE(uc1)   \
  }, \
  { uc2,  \
    WRITE_NOP, WRITE_NOP, WRITE_NOP, WRITE_NOP, WRITE_NOP, WRITE_NOP, WRITE_NOP,  \
    WRITE_OPCODE(uc3),

/* MACRO: Write ACCU to memory cell pointed by PTR_L and adr_h */
#define SAP_WRITE( adr_h ) \
    WRITE_ADDRH( ZEROPAGE_HI ),      /* select zero-page                            */ \
    WRITE_ADDRL( ACCU ),             /* select ACCU for next read                   */ \
    WRITE_DATAR_0( READ_MEMORY ),    /* read ACCU into DATAR                        */ \
    WRITE_DATAR( READ_DATAR ),       /* correct bit0                                */ \
    WRITE_ADDRL( PTR_L ),            /* select PTR_L for next read                  */ \
    WRITE_ADDRL( READ_MEMORY ),      /* read PTR_L into ADDR_L                      */ \
    WRITE_ADDRH( adr_h ),            /* set ADDR_H to adr_h                         */ \
    WRITE_OPCODE(UC_SAP_2)           /* execute next OP-Code                        */ \
    // counter roll-over, continue with UC_INCPC_NEXTOP

/* MACRO: Microinstruction roll-over, continue execution in next microinstruction */
#define CONT(uc)   WRITE_OPCODE(uc) },{ uc


/* allocate private microcode numbers */
enum ucnumbers {
  UC_RST_2  = UC_SUB_START,
  UC_RST_3,
  UC_NEXTOP,
  UC_INCPC_NEXTOP,
  UC_IPCNOP_2, UC_IPCNOP_3, UC_IPCNOP_4, UC_IPCNOP_5,
  UC_1OPR,   UC_1OPR_2, UC_1OPR_3, UC_1OPR_4, UC_1OPR_5, UC_1OPR_6, UC_1OPR_7,
  UC_2OPR,   UC_2OPR_2, UC_2OPR_3, UC_2OPR_4, UC_2OPR_5, UC_2OPR_6, UC_2OPR_7,
  UC_LDI_2,
  UC_LD_2,
  UC_LDAI_2,
  UC_LDA_2,
  UC_LAP_2,
  UC_STA_2,
  UC_SAP_1,  UC_SAP_2,  UC_SAP_3,  UC_SAP_4,  UC_SAP_5,  UC_SAP_6,  UC_SAP_7,  UC_SAP_8,  UC_SAP_9,
  UC_SAP_10, UC_SAP_11, UC_SAP_12, UC_SAP_13, UC_SAP_14, UC_SAP_15, UC_SAP_16, UC_SAP_17, UC_SAP_18, UC_SAP_19,
  UC_SAP_20, UC_SAP_21, UC_SAP_22, UC_SAP_23, UC_SAP_24, UC_SAP_25, UC_SAP_26, UC_SAP_27, UC_SAP_28, UC_SAP_29,
  UC_SAP_30, UC_SAP_31, UC_SAP_32, UC_SAP_33, UC_SAP_34, UC_SAP_35, UC_SAP_36, UC_SAP_37, UC_SAP_38, UC_SAP_39,
  UC_SAP_40, UC_SAP_41, UC_SAP_42, UC_SAP_43, UC_SAP_44, UC_SAP_45, UC_SAP_46, UC_SAP_47, UC_SAP_48, UC_SAP_49,
  UC_SAP_50, UC_SAP_51, UC_SAP_52, UC_SAP_53, UC_SAP_54, UC_SAP_55, UC_SAP_56, UC_SAP_57, UC_SAP_58, UC_SAP_59,
  UC_SAP_60, UC_SAP_61, UC_SAP_62, UC_SAP_63, UC_SAP_64, UC_SAP_65, UC_SAP_66, UC_SAP_67, UC_SAP_68, UC_SAP_69,
  UC_SAP_70, UC_SAP_71, UC_SAP_72, UC_SAP_73, UC_SAP_74, UC_SAP_75, UC_SAP_76, UC_SAP_77, UC_SAP_78, UC_SAP_79,
  UC_SAP_80, UC_SAP_81, UC_SAP_82, UC_SAP_83, UC_SAP_84, UC_SAP_85, UC_SAP_86, UC_SAP_87, UC_SAP_88, UC_SAP_89,
  UC_SAP_90, UC_SAP_91, UC_SAP_92, UC_SAP_93, UC_SAP_94, UC_SAP_95,
  UC_INC_2,
  UC_DEC_2,
  UC_SAVE2REG,
  UC_ROL_2,  UC_ROL_3,  UC_ROL_4,  UC_ROL_5,
  UC_ROR_2,  UC_ROR_3,
  UC_SUB_LP, UC_SUB_LP_2, UC_SUB_LP_3,
  UC_AND_2,  UC_AND_3,  UC_AND_4,  UC_AND_5,  UC_AND_6,
  UC_OR_2,   UC_OR_3,   UC_OR_4,   UC_OR_5,   UC_OR_6,
  UC_XOR_2,  UC_XOR_3,  UC_XOR_4,  UC_XOR_5,  UC_XOR_6,  UC_XOR_7,  UC_XOR_8,  UC_XOR_9,  UC_XOR_10, UC_XOR_11,
  UC_CMP_2,  UC_CMP_3,
  UC_CMPI_2, UC_CMPI_3,
  UC_ADD_2,  UC_ADD_3,  UC_ADD_4,  UC_ADD_5,  UC_ADD_6,  UC_ADD_7,  UC_ADD_8,  UC_ADD_9,  UC_ADD_10,
  UC_ADD_11, UC_ADD_12,
  UC_SUB_2,  UC_SUB_3,
  UC_TEST_2,
  UC_JPF_2,  UC_JPF_3,  UC_JPF_4,  UC_JPF_5,
  UC_JNF_2,
  UC_JSR_2,  UC_JSR_3,  UC_JSR_4,  UC_JSR_5, UC_JSR_6,
  UC_RET_2,
  UC_PSH_2,  UC_PSH_3,  UC_PSH_4,
  UC_POP_2,  UC_POP_3,
  UC_PHL_2,  UC_PHL_3,  UC_PHL_4,  UC_PHL_5,
  UC_RTS_2,  UC_RTS_3,  UC_RTS_4,  UC_RTS_5,
  UC_RWL_2,  UC_RWL_3,  UC_RWL_4,  UC_RWL_5,  UC_RWL_6,
  SU_JLP_2,  SU_JLP_3,
  UC_ICO_2,  UC_ICO_3,  UC_ICO_4,  UC_ICO_5,  UC_ICO_6,  UC_ICO_7,  UC_ICO_8,  UC_ICO_9,
  UC_SRT_2,
};


/* This is the big microcode table
 */
static uint32_t ucode_table[NUM_OPCODES][8+1] =
{
  /*-------------------------------------------------------------------------*/
  /* Reset entry and useful subroutines                                      */
  /*-------------------------------------------------------------------------*/

  /* RST / Reset */
  { OPCODE_RST,
    WRITE_NOP,                    // dummy (NOP slide)
    WRITE_NOP,                    // dummy (NOP slide)
    WRITE_NOP,                    // dummy (NOP slide)
    WRITE_NOP,                    // dummy (NOP slide)
    WRITE_NOP,                    // dummy (NOP slide)
    WRITE_NOP,                    // dummy (NOP slide)
    WRITE_NOP,                    // dummy (NOP slide)
    CONT(UC_RST_2),
    WRITE_NOP,                    // dummy (NOP slide)
    WRITE_NOP,                    // dummy (NOP slide)
    WRITE_NOP,                    // dummy (NOP slide)
    WRITE_NOP,                    // dummy (NOP slide)
    WRITE_NOP,                    // dummy (NOP slide)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR_CONST( 0x00 ),    // set constant value for next memory write
    CONT(UC_RST_3),
    WRITE_ADDRL( SP ),            // select SP register for next memory write
    WRITE_RAM( READ_DATAR ),      // SP := 0x00
    WRITE_ADDRL( PC_L ),          // select PC_L register for next memory write
    WRITE_RAM( READ_DATAR ),      // PC_L := 0x00
    WRITE_ADDRL( PC_H ),          // select PC_H register for next memory write
    WRITE_DATAR_CONST( (CODE_START>>8) ),  // set constant value for next memory write
    WRITE_RAM( READ_DATAR ),      // PC_H := 0x16
    WRITE_OPCODE( UC_NEXTOP ),
  },
  /*-------------------------------------------------------------------------*/
  { UC_NEXTOP,                    // Execute next OP-Code
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_L ),          // select PC_L register for next read
    WRITE_DATAR_0( READ_MEMORY ), // read memory and write PCL to DATA-register
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_H ),          // select PC_H register for next read
    WRITE_ADDRH( READ_MEMORY ),   // set ADDRH to PC_H
    WRITE_ADDRL( READ_DATAR ),    // set ADDRL to PC_L
    WRITE_OPCODE( READ_MEMORY ),  // read OP-Code from memory where PC points to
  },
  /*-------------------------------------------------------------------------*/
  { UC_INCPC_NEXTOP,              // increment PC and execute next OP code
    /* INC PC_L */
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_L ),          // select PC_L register for next memory write
    WRITE_ADDRL( READ_MEMORY ),   // set ADDRL to PC_L
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR_0( READ_MEMORY ), // read incremented PC_L from EPROM and write it to DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_L ),          // select PC_L register for next memory write
    CONT(UC_IPCNOP_2),
    WRITE_RAM( READ_DATAR ),      // write back PC_L
    /* prepare for zero-test of PC_L */
    WRITE_ADDRH( TABLE_TST  ),    // select table for zero-test
    WRITE_ADDRL( READ_DATAR ),    // write value to test for zero in ADDRL
    WRITE_NORIN( READ_MEMORY ),   // read test-value into NORIN
    /* prepare jump behind code for incrementing PC_H */
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    WRITE_OPCODE( UC_IPCNOP_3 ),  // jump to code that increments also PC_H
    WRITE_OPCODE( UC_NEXTOP ),    // execute next OP-Code
    },{ UC_IPCNOP_3,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE( UC_IPCNOP_4 ),
    },{ UC_IPCNOP_4,
    /* INC PC_H */
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_H ),          // select PC_H register for next memory write
    WRITE_ADDRL( READ_MEMORY ),   // set ADDRL to PC_L
    WRITE_ADDRH( TABLE_INC   ),   // select table for increment
    WRITE_DATAR_0( READ_MEMORY ), // read incremented PC_L from EEPROM and write it to DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_H ),          // select PC_H register for next memory write
    CONT(UC_IPCNOP_5),
    WRITE_RAM( READ_DATAR ),      // write back PC_H
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump to last micro-instruction
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_OPCODE( UC_NEXTOP ),
  },
  /*-------------------------------------------------------------------------*/
  { UC_2OPR,                      // subroutine: load 2 operands into TEMP2 and TEMP3 and return to caller via RADR1
    /* INC PC_L */
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_L ),          // select PC_L register for next memory write
    WRITE_ADDRL( READ_MEMORY ),   // set ADDRL to PC_L
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR_0( READ_MEMORY ), // read incremented PC_L from EPROM and write it to DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_L ),          // select PC_L register for next memory write
    CONT(UC_2OPR_2),
    WRITE_RAM( READ_DATAR ),      // write back PC_L
    /* prepare for zero-test of PC_L */
    WRITE_ADDRH( TABLE_TST  ),    // select table for zero-test
    WRITE_ADDRL( READ_DATAR ),    // write value to test for zero in ADDRL
    WRITE_NORIN( READ_MEMORY ),   // read test-value into NORIN
    /* prepare jump behind code for incrementing PC_H */
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    WRITE_OPCODE( UC_2OPR_3 ),    // jump to code that increments also PC_H
    WRITE_OPCODE( UC_2OPR_6 ),    // execute next OP-Code
    },{ UC_2OPR_3,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE( UC_2OPR_4 ),
    },{ UC_2OPR_4,
    /* INC PC_H */
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_H ),          // select PC_H register for next memory write
    WRITE_ADDRL( READ_MEMORY ),   // set ADDRL to PC_L
    WRITE_ADDRH( TABLE_INC   ),   // select table for increment
    WRITE_DATAR_0( READ_MEMORY ), // read incremented PC_L from EEPROM and write it to DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_H ),          // select PC_H register for next memory write
    CONT(UC_2OPR_5),
    WRITE_RAM( READ_DATAR ),      // write back PC_H
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump to last micro-instruction
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_OPCODE( UC_2OPR_6 ),
    },{ UC_2OPR_6,                // load operand from memory into TEMP3
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_L ),          // select PC_L register for next read
    WRITE_DATAR_0( READ_MEMORY ), // read memory and write PCL to DATA-register
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_H ),          // select PC_H register for next read
    WRITE_ADDRH( READ_MEMORY ),   // set ADDRH to PC_H
    WRITE_ADDRL( READ_DATAR ),    // set ADDRL to PC_L
    CONT(UC_2OPR_7),
    WRITE_DATAR( READ_MEMORY ),   // read operand value into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_RAM( READ_DATAR ),      // write register content into TEMP2
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE( UC_1OPR ),      // continue in UC_1OPR to load second operand
  },
  /*-------------------------------------------------------------------------*/
  { UC_1OPR,                      // subroutine: load 1 operand into TEMP3 and return to caller via RADR1
    /* INC PC_L */
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_L ),          // select PC_L register for next memory write
    WRITE_ADDRL( READ_MEMORY ),   // set ADDRL to PC_L
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR_0( READ_MEMORY ), // read incremented PC_L from EPROM and write it to DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_L ),          // select PC_L register for next memory write
    CONT(UC_1OPR_2),
    WRITE_RAM( READ_DATAR ),      // write back PC_L
    /* prepare for zero-test of PC_L */
    WRITE_ADDRH( TABLE_TST  ),    // select table for zero-test
    WRITE_ADDRL( READ_DATAR ),    // write value to test for zero in ADDRL
    WRITE_NORIN( READ_MEMORY ),   // read test-value into NORIN
    /* prepare jump behind code for incrementing PC_H */
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    WRITE_OPCODE( UC_1OPR_3 ),    // jump to code that increments also PC_H
    WRITE_OPCODE( UC_1OPR_6 ),    // load operand into temp3
  },
  { UC_1OPR_3,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP,
    CONT(UC_1OPR_4),
    /* INC PC_H */
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_H ),          // select PC_H register for next memory write
    WRITE_ADDRL( READ_MEMORY ),   // set ADDRL to PC_L
    WRITE_ADDRH( TABLE_INC   ),   // select table for increment
    WRITE_DATAR_0( READ_MEMORY ), // read incremented PC_L from EEPROM and write it to DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_H ),          // select PC_H register for next memory write
    CONT(UC_1OPR_5),
    WRITE_RAM( READ_DATAR ),      // write back PC_H
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOPs
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    CONT(UC_1OPR_6),
    /* load operand into TEMP3 */
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_L ),          // select PC_L register for next read
    WRITE_DATAR_0( READ_MEMORY ), // read memory and write PC_L to DATA-register
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_H ),          // select PC_H register for next read
    WRITE_ADDRH( READ_MEMORY ),   // set ADDRH to PC_H
    WRITE_ADDRL( READ_DATAR ),    // set ADDRL to PC_L
    CONT(UC_1OPR_7),
    WRITE_DATAR( READ_MEMORY ),   // read operand value into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next write
    WRITE_RAM( READ_DATAR ),      // write register content into TEMP3
    // return to caller
    WRITE_NOP,
    WRITE_ADDRL( RADR1 ),         // select RADR1 for next read
    WRITE_OPCODE( READ_MEMORY ),  // read return address, jump to next ucode block
  },

  /*-------------------------------------------------------------------------*/
  /* Implementation of OP-Codes                                              */
  /*-------------------------------------------------------------------------*/

  /* LD reg,# / load register with immediate value */
  { OPCODE_LDI,
    LOAD_2_OPERANDS( UC_LDI_2 ),  // load two operands into TEMP3 and TEMP2
  },
  { UC_LDI_2,                     
    // load register in TEMP2 with immediate value in TEMP3
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_DATAR( READ_MEMORY ),   // read content of TEMP3 into DATAR ("immediate value")
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read dest.addr into ADDRL
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* LD reg,reg / load register from other register */  
  { OPCODE_LD,
    LOAD_2_OPERANDS( UC_LD_2 ),  // load two operands into TEMP3 and TEMP2
  },
  { UC_LD_2,                     
    // load register in TEMP2 with immediate value in TEMP3
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_DATAR( READ_MEMORY ),   // read source register value into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read dest.addr into ADDRL
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* LDAI # / Load ACCU with immediate value */
  { OPCODE_LDAI,
    LOAD_OPERAND( UC_LDAI_2 ),    // load operand into TEMP3
  },
  { UC_LDAI_2,                     
    // load ACCU with immediate value in TEMP3
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_DATAR( READ_MEMORY ),   // read content of TEMP3 into DATAR ("immediate value")
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( ACCU ),          // select ACCU for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* LDA reg / Load ACCU from register */
  { OPCODE_LDA,
    LOAD_OPERAND( UC_LDA_2 ),     // load operand into TEMP3
  },
  { UC_LDA_2,                     
    // load ACCU from register in TEMP3
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_DATAR( READ_MEMORY ),   // read content of reg into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( ACCU ),          // select ACCU for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* LAP / Load Accu indirect through PTR register */  
  { OPCODE_LAP,
    // load PTR_L and PTR_H into ADDRL and ADDRH
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_DATAR_0( READ_MEMORY ), // read content of first ZP-register into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_H ),         // select PTR_H for next read
    WRITE_ADDRH( READ_MEMORY ),   // store content of second ZP-register in ADDRH
    WRITE_ADDRL( READ_DATAR ),    // move DATAR into ADDRL (content of first ZP-register)
    CONT(UC_LAP_2),
    // read memory pointed by PTR into DATAR
    WRITE_DATAR( READ_MEMORY ),   // finally read the value from address pointed by ZP and ZP+1
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // store DATAR to ACCU
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( ACCU ),          // select ACCU for next write
    WRITE_RAM( READ_DATAR ),      // DATAR into the ACCU
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* STA reg / Store ACCU to register */
  { OPCODE_STA,
    LOAD_OPERAND( UC_STA_2 ),     // load operand into TEMP3
  },
  { UC_STA_2,                     
    // load ACCU into DATAR and save DATAR in target register
    WRITE_ADDRL( ACCU ),          // select ACCU for next write
    WRITE_DATAR( READ_MEMORY ),   // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read address of register into ADDRL
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* SAP / Store Accu to memory pointed by PTR register */  
  { OPCODE_SAP,
    // load PTR_H 
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PTR_H ),         // select PTR_H for next read
    WRITE_ADDRL( READ_MEMORY ),   // write PTR_H into ADDRL for table lookup
    WRITE_ADDRH( TABLE_ROL ),     // select ROL table
    WRITE_ADDRL( READ_MEMORY ),   // rotate left
    WRITE_ADDRL( READ_MEMORY ),   // rotate left
    WRITE_ADDRL( READ_MEMORY ),   // rotate left
    CONT(UC_SAP_1),
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 4) ^ 0x01  (bit0 = PTR_H.4 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.4 == 1), handle address range 0x9000-0x9FFF
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_SAP_4),
    WRITE_OPCODE(UC_SAP_3)
  },
  { UC_SAP_2,                     // Note: This label is also referenced by OPCODE_POP, so do not change!
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L and PTR_H
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOPs
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP)
  },
  { UC_SAP_4,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_SAP_5),
  },
  { UC_SAP_5, // handle 0x8000-0x8FFF
    SAP_DECODE( UC_SAP_6,  UC_SAP_7,  UC_SAP_8  )  // UC_SAP_8 : 0x8000-0x87FF, UC_SAP_6 : 0x8800-0x8FFF
  },
  { UC_SAP_8, // handle 0x8000-0x87FF
    SAP_DECODE( UC_SAP_9,  UC_SAP_10, UC_SAP_11 )  // UC_SAP_11: 0x8000-0x83FF, UC_SAP_9 : 0x8400-0x87FF
  },
  { UC_SAP_11, // handle 0x8000-0x83FF
    SAP_DECODE( UC_SAP_12, UC_SAP_13, UC_SAP_14 )  // UC_SAP_14: 0x8000-0x81FF, UC_SAP_12: 0x8200-0x83FF
  },
  { UC_SAP_14, // handle 0x8000-0x81FF
    SAP_DECODE( UC_SAP_15, UC_SAP_16, UC_SAP_17 )  // UC_SAP_17: 0x8000-0x80FF, UC_SAP_15: 0x8100-0x81FF
  },
  { UC_SAP_12, // handle 0x8200-0x83FF
    SAP_DECODE( UC_SAP_18, UC_SAP_19, UC_SAP_20 )  // UC_SAP_20: 0x8300-0x83FF, UC_SAP_18: 0x8200-0x82FF
  },
  { UC_SAP_9,  // handle 0x8400-0x87FF
    SAP_DECODE( UC_SAP_21, UC_SAP_22, UC_SAP_23 )  // UC_SAP_23: 0x8400-0x85FF, UC_SAP_21: 0x8600-0x87FF
  },
  { UC_SAP_23, // handle 0x8400-0x85FF
    SAP_DECODE( UC_SAP_24, UC_SAP_25, UC_SAP_26 )  // UC_SAP_26: 0x8400-0x84FF, UC_SAP_24: 0x8500-0x85FF
  },
  { UC_SAP_21, // handle 0x8600-0x87FF
    SAP_DECODE( UC_SAP_27, UC_SAP_28, UC_SAP_29 )  // UC_SAP_29: 0x8600-0x86FF, UC_SAP_27: 0x8700-0x87FF
  },
  { UC_SAP_6,  // handle 0x8800-0x8FFF
    SAP_DECODE( UC_SAP_30, UC_SAP_31, UC_SAP_32 )  // UC_SAP_32: 0x8800-0x8BFF, UC_SAP_30: 0x8C00-0x8FFF
  },
  { UC_SAP_32, // handle 0x8800-0x8BFF
    SAP_DECODE( UC_SAP_33, UC_SAP_34, UC_SAP_35 )  // UC_SAP_35: 0x8800-0x89FF, UC_SAP_33: 0x8A00-0x8BFF
  },
  { UC_SAP_35, // handle 0x8800-0x89FF
    SAP_DECODE( UC_SAP_36, UC_SAP_37, UC_SAP_38 )  // UC_SAP_38: 0x8800-0x88FF, UC_SAP_36: 0x8900-0x89FF
  },
  { UC_SAP_33, // handle 0x8A00-0x8BFF
    SAP_DECODE( UC_SAP_39, UC_SAP_40, UC_SAP_41 )  // UC_SAP_41: 0x8A00-0x8AFF, UC_SAP_39: 0x8B00-0x8BFF
  },
  { UC_SAP_30, // handle 0x8C00-0x8FFF
    SAP_DECODE( UC_SAP_42, UC_SAP_43, UC_SAP_44 )  // UC_SAP_44: 0x8C00-0x8DFF, UC_SAP_42: 0x8E00-0x8FFF
  },
  { UC_SAP_44, // handle 0x8C00-0x8DFF
    SAP_DECODE( UC_SAP_45, UC_SAP_46, UC_SAP_47 )  // UC_SAP_47: 0x8C00-0x8CFF, UC_SAP_45: 0x8D00-0x8DFF
  },
  { UC_SAP_42, // handle 0x8E00-0x8FFF
    SAP_DECODE( UC_SAP_48, UC_SAP_49, UC_SAP_50 )  // UC_SAP_50: 0x8E00-0x8EFF, UC_SAP_48: 0x8F00-0x8FFF
  },
  { UC_SAP_3,  // handle 0x9000-0x9FFF
    SAP_DECODE( UC_SAP_51, UC_SAP_52, UC_SAP_53 )  // UC_SAP_53: 0x9000-0x97FF, UC_SAP_51: 0x9800-0x9FFF
  },
  { UC_SAP_53, // handle 0x9000-0x97FF
    SAP_DECODE( UC_SAP_54, UC_SAP_55, UC_SAP_56 )  // UC_SAP_56: 0x9000-0x93FF, UC_SAP_54: 0x9400-0x97FF
  },
  { UC_SAP_56, // handle 0x9000-0x93FF
    SAP_DECODE( UC_SAP_57, UC_SAP_58, UC_SAP_59 )  // UC_SAP_59: 0x9000-0x91FF, UC_SAP_57: 0x9200-0x93FF
  },
  { UC_SAP_59, // handle 0x9000-0x91FF
    SAP_DECODE( UC_SAP_60, UC_SAP_61, UC_SAP_62 )  // UC_SAP_62: 0x9000-0x90FF, UC_SAP_60: 0x9100-0x91FF
  },
  { UC_SAP_57, // handle 0x9200-0x93FF
    SAP_DECODE( UC_SAP_63, UC_SAP_64, UC_SAP_65 )  // UC_SAP_65: 0x9200-0x92FF, UC_SAP_63: 0x9300-0x93FF
  },
  { UC_SAP_54, // handle 0x9400-0x97FF
    SAP_DECODE( UC_SAP_66, UC_SAP_67, UC_SAP_68 )  // UC_SAP_68: 0x9400-0x95FF, UC_SAP_66: 0x9600-0x97FF
  },
  { UC_SAP_68, // handle 0x9400-0x95FF
    SAP_DECODE( UC_SAP_69, UC_SAP_70, UC_SAP_71 )  // UC_SAP_71: 0x9400-0x94FF, UC_SAP_69: 0x9500-0x95FF
  },
  { UC_SAP_66, // handle 0x9600-0x97FF
    SAP_DECODE( UC_SAP_72, UC_SAP_73, UC_SAP_74 )  // UC_SAP_74: 0x9600-0x96FF, UC_SAP_72: 0x9700-0x97FF
  },
  { UC_SAP_51, // handle 0x9800-0x9FFF
    SAP_DECODE( UC_SAP_75, UC_SAP_76, UC_SAP_77 )  // UC_SAP_77: 0x9800-0x9BFF, UC_SAP_75: 0x9C00-0x9FFF
  },
  { UC_SAP_77, // handle 0x9800-0x9BFF
    SAP_DECODE( UC_SAP_78, UC_SAP_79, UC_SAP_80 )  // UC_SAP_80: 0x9800-0x99FF, UC_SAP_78: 0x9A00-0x9BFF
  },
  { UC_SAP_80, // handle 0x9800-0x99FF
    SAP_DECODE( UC_SAP_81, UC_SAP_82, UC_SAP_83 )  // UC_SAP_83: 0x9800-0x98FF, UC_SAP_81: 0x9900-0x99FF
  },
  { UC_SAP_78, // handle 0x9A00-0x9BFF
    SAP_DECODE( UC_SAP_84, UC_SAP_85, UC_SAP_86 )  // UC_SAP_86: 0x9A00-0x9AFF, UC_SAP_84: 0x9B00-0x9BFF
  },
  { UC_SAP_75, // handle 0x9C00-0x9FFF
    SAP_DECODE( UC_SAP_87, UC_SAP_88, UC_SAP_89 )  // UC_SAP_89: 0x9C00-0x9DFF, UC_SAP_87: 0x9E00-0x9FFF
  },
  { UC_SAP_89, // handle 0x9C00-0x9DFF
    SAP_DECODE( UC_SAP_90, UC_SAP_91, UC_SAP_92 )  // UC_SAP_92: 0x9C00-0x9CFF, UC_SAP_90: 0x9D00-0x9DFF
  },
  { UC_SAP_87, // handle 0x9E00-0x9FFF
    SAP_DECODE( UC_SAP_93, UC_SAP_94, UC_SAP_95 )  // UC_SAP_95: 0x9E00-0x9EFF, UC_SAP_93: 0x9F00-0x9FFF
  },
  { UC_SAP_17, SAP_WRITE( 0x80 ) }, // write to 0x8000-0x80FF
  { UC_SAP_15, SAP_WRITE( 0x81 ) }, // write to 0x8100-0x81FF
  { UC_SAP_20, SAP_WRITE( 0x82 ) }, // write to 0x8200-0x82FF
  { UC_SAP_18, SAP_WRITE( 0x83 ) }, // write to 0x8300-0x83FF
  { UC_SAP_26, SAP_WRITE( 0x84 ) }, // write to 0x8400-0x84FF
  { UC_SAP_24, SAP_WRITE( 0x85 ) }, // write to 0x8500-0x85FF
  { UC_SAP_29, SAP_WRITE( 0x86 ) }, // write to 0x8600-0x86FF
  { UC_SAP_27, SAP_WRITE( 0x87 ) }, // write to 0x8700-0x87FF
  { UC_SAP_38, SAP_WRITE( 0x88 ) }, // write to 0x8800-0x88FF
  { UC_SAP_36, SAP_WRITE( 0x89 ) }, // write to 0x8900-0x89FF
  { UC_SAP_41, SAP_WRITE( 0x8A ) }, // write to 0x8A00-0x8AFF
  { UC_SAP_39, SAP_WRITE( 0x8B ) }, // write to 0x8B00-0x8BFF
  { UC_SAP_47, SAP_WRITE( 0x8C ) }, // write to 0x8C00-0x8CFF
  { UC_SAP_45, SAP_WRITE( 0x8D ) }, // write to 0x8D00-0x8DFF
  { UC_SAP_50, SAP_WRITE( 0x8E ) }, // write to 0x8E00-0x8EFF
  { UC_SAP_48, SAP_WRITE( 0x8F ) }, // write to 0x8F00-0x8FFF
  { UC_SAP_62, SAP_WRITE( 0x90 ) }, // write to 0x9000-0x90FF
  { UC_SAP_60, SAP_WRITE( 0x91 ) }, // write to 0x9100-0x91FF
  { UC_SAP_65, SAP_WRITE( 0x92 ) }, // write to 0x9200-0x92FF
  { UC_SAP_63, SAP_WRITE( 0x93 ) }, // write to 0x9300-0x93FF
  { UC_SAP_71, SAP_WRITE( 0x94 ) }, // write to 0x9400-0x94FF
  { UC_SAP_69, SAP_WRITE( 0x95 ) }, // write to 0x9500-0x95FF
  { UC_SAP_74, SAP_WRITE( 0x96 ) }, // write to 0x9600-0x96FF
  { UC_SAP_72, SAP_WRITE( 0x97 ) }, // write to 0x9700-0x97FF
  { UC_SAP_83, SAP_WRITE( 0x98 ) }, // write to 0x9800-0x98FF
  { UC_SAP_81, SAP_WRITE( 0x99 ) }, // write to 0x9900-0x99FF
  { UC_SAP_86, SAP_WRITE( 0x9A ) }, // write to 0x9A00-0x9AFF
  { UC_SAP_84, SAP_WRITE( 0x9B ) }, // write to 0x9B00-0x9BFF
  { UC_SAP_92, SAP_WRITE( 0x9C ) }, // write to 0x9C00-0x9CFF
  { UC_SAP_90, SAP_WRITE( 0x9D ) }, // write to 0x9D00-0x9DFF
  { UC_SAP_95, SAP_WRITE( 0x9E ) }, // write to 0x9E00-0x9EFF
  { UC_SAP_93, SAP_WRITE( 0x9F ) }, // write to 0x9F00-0x9FFF
  /*-------------------------------------------------------------------------*/
  { OPCODE_INC,
    LOAD_OPERAND( UC_INC_2 ),     // load operand into TEMP3
  },
  { UC_INC_2,
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_ADDRL( READ_MEMORY ),   // read content of reg into ADDR_L
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR( READ_MEMORY ),   // perform table-lookup
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_SAVE2REG)     // save DATAR to register in TEMP3
  },
  { UC_SAVE2REG,
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump to end of micro-instruction
    WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  { OPCODE_DEC,
    LOAD_OPERAND( UC_DEC_2 ),     // load operand into TEMP3
  },
  { UC_DEC_2,
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_ADDRL( READ_MEMORY ),   // read content of reg into ADDR_L
    WRITE_ADDRH( TABLE_DEC ),     // select table for decrement
    WRITE_DATAR( READ_MEMORY ),   // perform table-lookup
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_SAVE2REG)     // save DATAR to register in TEMP3
  },
  /*-------------------------------------------------------------------------*/
  /* ROL reg / Rotate reg left with carry in FLAG register */
  { OPCODE_ROL,
    LOAD_OPERAND( UC_ROL_2 ),     // load operand into TEMP3
  },
  { UC_ROL_2,
    // load register value and rotate it left through table lookup, write it to DATAR
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_ADDRL( READ_MEMORY ),   // read content of reg into ADDR_L
    WRITE_ADDRH( TABLE_ROL ),     // select ROL table
    WRITE_DATAR( READ_MEMORY ),   // perform table-lookup
    // save register value to TEMP2
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    CONT(UC_ROL_3),
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP2. Bit 0 is inverted.
    // clear bit0 from value in DATAR
    WRITE_NORIN( 0x01 ),          // set NOR-input to 1
    WRITE_DATAR( READ_DATAR ),    // set bit0 in DATAR to 0
    // load FLAG register into NORIN
    WRITE_ADDRL( FLAG ),          // select FLAG for next read
    WRITE_NORIN( READ_MEMORY ),   // load FLAG into NORIN
    WRITE_DATAR( READ_DATAR ),    // set bit0 in DATAR value in FLAG (negated)
    WRITE_DATAR_0( READ_DATAR ),  // correct bit0
    CONT(UC_ROL_4),
    // save result in register
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // move content of TEMP3 into ADDRL
    WRITE_RAM( READ_DATAR ),      // write rotated value into register
    // set the FLAG register to new carry value
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
    WRITE_NORIN( READ_MEMORY ),   // load new (still inverted) flag into NOR input
    WRITE_DATAR( READ_DATAR ),    // dummy, relax timing for next operation
    CONT(UC_ROL_5),
    WRITE_DATAR( 0x00 ),          // clear bits 1-7 in DATAR, correct polarity of bit0
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into FLAG register
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump to end of micro-instruction
    WRITE_NOP, WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* ROR reg / Rotate reg right with carry in FLAG register */
  { OPCODE_ROR,
    LOAD_OPERAND( UC_ROR_2 ),     // load operand into TEMP3
  },
  { UC_ROR_2,
    // load register value into DATAR
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_DATAR( READ_MEMORY ),   // read content of reg into DATAR
    // save register value to TEMP2
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP2. Bit 0 is inverted.
    // clear bit0 from value in DATAR
    WRITE_NORIN( 0x01 ),          // set NOR-input to 1
    WRITE_DATAR( READ_DATAR ),    // set bit0 in DATAR to 0
    CONT(UC_ROR_3),
    // load FLAG register into NORIN
    WRITE_ADDRL( FLAG ),          // select FLAG for next read
    WRITE_NORIN( READ_MEMORY ),   // load FLAG into NORIN
    WRITE_DATAR( READ_DATAR ),    // set bit0 in DATAR value in FLAG (negated)
    WRITE_DATAR_0( READ_DATAR ),  // correct bit0
    // rotate the value in DATAR right
    WRITE_ADDRL( READ_DATAR ),    // set table index
    WRITE_ADDRH( TABLE_ROR ),     // select ROR table
    WRITE_DATAR( READ_MEMORY ),   // perform the table lookup
    WRITE_OPCODE(UC_ROL_4)        // continue in ROL opcode
  },
  /*-------------------------------------------------------------------------*/
  /* Subroutine to initialize loop for AND / OR / XOR */
  { UC_SUB_LP,
    LOAD_OPERAND( UC_SUB_LP_2 )
  },
  { UC_SUB_LP_2,
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_DATAR( READ_MEMORY ),   // read content of reg into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // write value into TEMP1
    WRITE_NOP,
    CONT(UC_SUB_LP_3),
    // initialize loop counter in TEMP2
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_DATAR_CONST( 0x01 ),    // loop init value
    WRITE_RAM( READ_DATAR ),      // write value into TEMP2
    WRITE_ADDRL( JUMP(6) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over NOP
    WRITE_NOP,
    WRITE_ADDRL( RADR2 ),         // select TEMP2 for next read
    WRITE_OPCODE( READ_MEMORY ),  // read return address, jump to begin of loop
  },
  /*-------------------------------------------------------------------------*/
  /* AND reg / perform AND operation on Accu and register */  
  { OPCODE_AND,
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR_CONST(UC_AND_2),  // loop start address
    WRITE_ADDRL( RADR2 ),         // select RADR2 for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR to RAM
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOP
    WRITE_NOP,
    WRITE_OPCODE(UC_SUB_LP),     // call subroutine that initializes the loop
  },
  { UC_AND_2,  // Loop
      // AND-operation on bit0:
      // 1. invert bit0 of TEMP1, write to TEMP3
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_DATAR( READ_MEMORY ),   // read value from TEMP1 into DATAR (=invert bit0)
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next write
      WRITE_RAM( READ_DATAR ),      // write value into TEMP3
      // 2. read ACCU into DATAR, invert bit0
      WRITE_ADDRL( ACCU ),          // select ACCU for next read
      WRITE_DATAR( READ_MEMORY ),   // read ACCU into DATAR
      // 3. read inverted bit0 of reg into NORIN
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      CONT(UC_AND_3),
      WRITE_NORIN( READ_MEMORY ),   // write bit0 into NORIN
      // 4. perform: accu = (accu & 0xFE) or (not ((not accuBit0) or (not regBit0)))
      WRITE_DATAR( READ_DATAR ),    // perform final NOR operation
      // rotate accu (in DATAR) right
      WRITE_ADDRL( READ_DATAR ),    // write ACCU into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR_0( READ_MEMORY ), // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( ACCU ),          // select ACCU for next write
      CONT(UC_AND_4),
      WRITE_RAM( READ_DATAR ),      // write ACCU back to RAM
      // rotate TEMP1 right
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP1 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
      CONT(UC_AND_5),
      WRITE_RAM( READ_DATAR ),      // write TEMP1 back to RAM
      // rotate TEMP2 right
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP2 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
      CONT(UC_AND_6),
      WRITE_RAM( READ_DATAR ),      // write TEMP2 back to RAM
    // end loop
    WRITE_NORIN( READ_DATAR ),
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_LDI_2),       // quit the loop, jump to UC_INCPC_NEXTOP
    WRITE_OPCODE(UC_AND_2)        // jump to begin of the loop
  },
  /*-------------------------------------------------------------------------*/
  /* OR reg / perform OR operation on Accu and register */  
  { OPCODE_OR,
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR_CONST(UC_OR_2),   // loop start address
    WRITE_ADDRL( RADR2 ),         // select RADR2 for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR to RAM
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOP
    WRITE_NOP,
    WRITE_OPCODE(UC_SUB_LP),     // call subroutine that initializes the loop
  },
  { UC_OR_2,  // Loop
      // OR-operation on bit0:
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_NORIN( READ_MEMORY ),   // read TEMP1 bit0 into NORIN
      WRITE_ADDRL( ACCU ),          // select ACCU for next read
      WRITE_DATAR( READ_MEMORY ),   // read ACCU into DATAR
      WRITE_DATAR_0( READ_DATAR ),  // correct bit0
      // rotate accu (in DATAR) right
      WRITE_ADDRL( READ_DATAR ),    // write ACCU into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      CONT(UC_OR_3),
      WRITE_DATAR_0( READ_MEMORY ), // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( ACCU ),          // select ACCU for next write
      WRITE_RAM( READ_DATAR ),      // write ACCU back to RAM
      // rotate TEMP1 right
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP1 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      CONT(UC_OR_4),
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP1 back to RAM
      // rotate TEMP2 right
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP2 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      CONT(UC_OR_5),
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP2 back to RAM
    // end loop
    WRITE_NORIN( READ_DATAR ),
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    WRITE_NOP,
    CONT(UC_OR_6),
    JUMP_IF_ZERO,                 // perform the jump
    WRITE_ADDRL( JUMP(6) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over NOPs
    WRITE_NOP,
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_LDI_2),       // quit the loop, jump to UC_INCPC_NEXTOP
    WRITE_OPCODE(UC_OR_2)         // jump to begin of the loop
  },
  /*-------------------------------------------------------------------------*/
  /* XOR reg / perform XOR operation on Accu and register */  
  { OPCODE_XOR,
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR_CONST(UC_XOR_10), // return address
    WRITE_ADDRL( RADR2 ),         // select RADR2 for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR to RAM
    WRITE_ADDRL( RADR3 ),         // select RADR3 for next write
    WRITE_DATAR_CONST(UC_XOR_11), // return address for XOR operation
    WRITE_RAM( READ_DATAR ),      // write DATAR to RAM
    WRITE_OPCODE(UC_SUB_LP),      // call subroutine that initializes the loop
  },
  { UC_XOR_10,
    // store a copy of the ACCU into TEMP4
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR( READ_MEMORY ),   // read value from ACCU into DATAR (=invert bit0)
    WRITE_DATAR_0( READ_DATAR ),  // correct bit0
    WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into TEMP4
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_XOR_2),       // continue with XOR
  },
  { UC_XOR_2,  // Loop
      // XOR-operation on bit0:
      // 1. TEMP3.0 = (not TEMP4) nor (not REG) = TEMP4 and REG
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_DATAR( READ_MEMORY ),   // read value from TEMP1 into DATAR (=invert bit0)
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next write
      WRITE_RAM( READ_DATAR ),      // write value into TEMP3
      WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next read
      WRITE_DATAR( READ_MEMORY ),   // read TEMP4 into DATAR
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      CONT(UC_XOR_3),
      WRITE_NORIN( READ_MEMORY ),   // write bit0 into NORIN
      WRITE_DATAR( READ_DATAR ),    // perform final NOR operation
      WRITE_RAM( READ_DATAR ),      // write result into TEMP3
      // 2. TEMP4.0 = TEMP4 nor REG
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_NORIN( READ_MEMORY ),   // write bit0 into NORIN
      WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next read
      WRITE_DATAR( READ_MEMORY ),   // perform final NOR operation
      CONT(UC_XOR_4),
      // 3. TEMP4 = TEMP3 nor TEMP4.0  (TEMP3 bits 1-7 contain also TEMP4 bits 1-7)
      WRITE_NORIN( READ_DATAR ),    // write TEMP4.0 into NORIN
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      WRITE_DATAR( READ_MEMORY ),   // perform DATAR = TEMP3 nor TEMP4
      // rotate TEMP4 (in DATAR) right
      WRITE_ADDRL( READ_DATAR ),    // write TEMP4 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR_0( READ_MEMORY ), // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      CONT(UC_XOR_5),
      WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP4 back to RAM
      // rotate TEMP1 right
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP1 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      CONT(UC_XOR_6),
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP1 back to RAM
      // rotate TEMP2 right
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP2 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      CONT(UC_XOR_7),
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP2 back to RAM
    // end loop
    WRITE_NORIN( READ_DATAR ),
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    WRITE_NOP,
    WRITE_OPCODE(UC_XOR_8),       // quit (jump to exit routine)
    WRITE_OPCODE(UC_XOR_2)        // jump to begin of the loop
  },
  { UC_XOR_8,
    WRITE_ADDRL( JUMP(6) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over 4 NOPs
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP,
    CONT(UC_XOR_9),
    WRITE_ADDRL( JUMP(6) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over NOPs
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    // return from subroutine
    WRITE_ADDRL( RADR3 ),         // select RADR3 for next read
    WRITE_OPCODE( READ_MEMORY ),  // read return address, jump to next ucode block
  },
  { UC_XOR_11,
    // copy TEMP4 to ACCU
    WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next read
    WRITE_DATAR( READ_MEMORY ),   // read value from TEMP4 into DATAR (=invert bit0)
    WRITE_DATAR_0( READ_DATAR ),  // correct bit0
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_RAM( READ_DATAR ),      // write DATAR into ACCU
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* CMP reg / Compare ACCU with register value */
  { OPCODE_CMP,
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR_CONST(UC_XOR_10), // return address, calls XOR
    WRITE_ADDRL( RADR2 ),         // select RADR2 for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR to RAM
    WRITE_ADDRL( RADR3 ),         // select RADR3 for next write
    WRITE_DATAR_CONST(UC_CMP_2),  // return address for XOR operation
    WRITE_RAM( READ_DATAR ),      // write DATAR to RAM
    WRITE_OPCODE(UC_SUB_LP),      // call subroutine that initializes the loop
  },
  { UC_CMP_2,
    // check if TEMP4 is zero, set flag
    WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next read
    WRITE_ADDRL( READ_MEMORY ),   // load TEMP4 into ADDRL for table-lookup
    WRITE_ADDRH( TABLE_TST ),     // select table for testing for zero
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // write result into FLAG
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_NOP,
    CONT(UC_CMP_3),               // Note: This label is called from OPCODE_TST
    WRITE_ADDRL( FLAG ),          // select FLAG for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into to RAM
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOPs
    WRITE_NOP,
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* CMPI # / Compare ACCU with immediate value */
  { OPCODE_CMPI,
    LOAD_OPERAND( UC_CMPI_2 )
  },
  { UC_CMPI_2,
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR_CONST(UC_XOR_10), // return address, calls XOR
    WRITE_ADDRL( RADR2 ),         // select RADR2 for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR to RAM
    WRITE_ADDRL( RADR3 ),         // select RADR3 for next write
    WRITE_DATAR_CONST(UC_CMP_2),  // return address for XOR operation
    WRITE_RAM( READ_DATAR ),      // write DATAR to RAM
    CONT(UC_CMPI_3),
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_DATAR( READ_MEMORY ),   // read content of TEMP3 into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // write value into TEMP1
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_SUB_LP_3)     // continue in UC_SUB_LP_3 routine
  },
  /*-------------------------------------------------------------------------*/
  /* ADD reg / Add ACCU and register (8-bit binary addition w/ carry) */
  { OPCODE_ADD,
    LOAD_OPERAND( UC_ADD_2 )
  },
  { UC_ADD_2,
    // load content of OP-code parameter register into TEMP1
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_DATAR( READ_MEMORY ),   // read content of reg into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // write value into TEMP1
    WRITE_NOP,
    CONT(UC_ADD_3),               // here begins the code that is shared with SUB
    // initialize loop counter in TEMP2
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_DATAR_CONST( 0x01 ),    // loop init value
    WRITE_RAM( READ_DATAR ),      // write value into TEMP2
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOPs
    WRITE_NOP,
    WRITE_NOP,
    CONT(UC_ADD_4),               // Loop: Sum all bits of ACCU and TEMP1
      // combine ACCU.0, TEMP1.0 and FLAG.0 to a 3-bit word
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_NORIN( READ_MEMORY ),   // read bit 0 from TEMP1 into NORIN
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate left
      WRITE_DATAR( 0x00 ),          // clear bit 1-7 in DATAR, keep bit0 from NORIN, invert bit0
      WRITE_ADDRL( READ_DATAR ),    // write temp.value into ADDRL for table-lookup
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      // DATAR now:  bit7 = not TEMP1.0, bit0-6 = 0
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      CONT(UC_ADD_5),
      WRITE_ADDRL( ACCU ),          // select ACCU for next read
      WRITE_NORIN( READ_MEMORY ),   // read bit 0 from ACCU into NORIN
      WRITE_DATAR( READ_DATAR ),    // move bit 0 into temp.value
      WRITE_ADDRL( READ_DATAR ),    // write temp.value into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate left
      WRITE_DATAR_0( READ_MEMORY ),   // perform rotation
      // DATAR now:  bit7 = not ACCU.0, bit6 = not TEMP1.0, bit0-5 = 0
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      CONT(UC_ADD_6),
      WRITE_ADDRL( FLAG ),          // select FLAG for next read
      WRITE_NORIN( READ_MEMORY ),   // read bit 0 from FLAG into NORIN
      WRITE_DATAR( READ_DATAR ),    // move bit 0 into temp.value
      // DATAR now:  bit7 = not ACCU.0, bit6 = not TEMP1.0, bit1-5 = 0, bit0 = not FLAG.0
      WRITE_ADDRL( READ_DATAR ),    // write temp.value into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_TST ),     // select table for full-adder operation (stored in TST table)
      WRITE_DATAR_0( READ_MEMORY ), // perform table lookup
      // DATAR now:  bit7 = carry, bit1 = not sum
      // save DATAR to TEMP3
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      CONT(UC_ADD_7),
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      WRITE_RAM( READ_DATAR ),      // write result into TEMP3
      // save carry flag
      WRITE_ADDRL( READ_DATAR ),    // write temp.value into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROL ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation, invert bit0 (FLAG)
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( FLAG ),          // select FLAG for next write
      CONT(UC_ADD_8),
      WRITE_RAM( READ_DATAR ),      // write result to FLAG
      // read ACCU into DATAR, clear bit 0
      WRITE_NORIN( 0x01 ),          // set NORIN to 1
      WRITE_ADDRL( ACCU ),          // select ACCU for next read
      WRITE_DATAR( READ_MEMORY ),   // read ACCU into DATAR, clear bit 0
      // rotate TEMP3 right, write SUM bit to ACCU.0
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP3 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      CONT(UC_ADD_9),
      WRITE_NORIN( READ_MEMORY ),   // read bit 0 from TEMP3 into NORIN (inverted)
      WRITE_DATAR( READ_DATAR ),    // set bit 0 in DATAR to TEMP3.0
      // rotate DATAR right
      WRITE_ADDRL( READ_DATAR ),    // write temp.value into ADDRL for table-lookup
      WRITE_DATAR_0( READ_MEMORY ), // perform rotation
      // save DATAR to ACCU
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( ACCU ),          // select ACCU for next write
      WRITE_RAM( READ_DATAR ),      // write result into ACCU
      CONT(UC_ADD_10),
      // rotate TEMP1 right
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP1 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP1 back to RAM
      CONT(UC_ADD_11),
      // rotate TEMP2 right
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP2 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP2 back to RAM
      CONT(UC_ADD_12),
    // end loop
    WRITE_NORIN( READ_DATAR ),
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    WRITE_ADDRL( JUMP(6) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOP
    WRITE_NOP,
    WRITE_OPCODE(UC_LDI_2),       // quit (jump to exit routine)
    WRITE_OPCODE(UC_ADD_4)        // jump to begin of the loop
  },
  /*-------------------------------------------------------------------------*/
  /* SUB reg / Substract register value from ACCU (8-bit with carry) */
  { OPCODE_SUB,
    LOAD_OPERAND( UC_SUB_2 )
  },
  { UC_SUB_2,
    // load content of OP-code parameter register into WRITE_ADDRL
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_ADDRL( READ_MEMORY ),   // read content of reg into ADDRL
    // perform table-lookup and get the 1's complement
    WRITE_ADDRH( TABLE_NOT ),     // select table for 1's complement
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // write 1's complement of reg value into TEMP1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    CONT(UC_SUB_3),
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // write register content into TEMP1
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOPs
    WRITE_NOP,
    WRITE_NOP,
    WRITE_NOP,
    // initialize loop counter in TEMP2
    WRITE_OPCODE(UC_ADD_3)        // jump to ADD routine
  },
  /*-------------------------------------------------------------------------*/
  /* TST reg / Test if register is zero. Set flag when yes. */  
  { OPCODE_TST,
    LOAD_OPERAND( UC_TEST_2 )
  },
  { UC_TEST_2,
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_ADDRL( READ_MEMORY ),   // read content of reg into ADDRL
    WRITE_ADDRH( TABLE_TST ),     // select test-zero-table for lookup
    WRITE_DATAR( READ_MEMORY ),   // write result from table into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_OPCODE(UC_CMP_3)        // jump to CMP routine
  },
  /*-------------------------------------------------------------------------*/
  /* JPF abs / Jump only when flag is set */  
  { OPCODE_JPF,
    LOAD_2_OPERANDS( UC_JPF_2 ),  // load two operands into TEMP3 and TEMP2
  },
  { UC_JPF_2,
    WRITE_ADDRL( FLAG ),          // select FLAG for next read
    WRITE_NORIN( READ_MEMORY ),   // write FLAG into NORIN
    WRITE_ADDRL( JUMP(6) ),       // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump (continue normal program flow)
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the next micro-instruction
    WRITE_OPCODE(UC_LDI_2),       // continue with next OP code
    WRITE_OPCODE(UC_JPF_3)        // set PC to new address
  },
  { UC_JPF_3,
    WRITE_NOP, WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP, WRITE_NOP,
    WRITE_NOP,
    CONT(UC_JPF_4),
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
    WRITE_DATAR( READ_MEMORY ),   // read content of reg into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_L ),          // select PC_H for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_H
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_DATAR( READ_MEMORY ),   // read content of reg into DATAR
    CONT(UC_JPF_5),
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_H ),          // select PC_L for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_L
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOPs
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_NEXTOP)       // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* JNF abs / Jump only when flag is not set */  
  { OPCODE_JNF,
    LOAD_2_OPERANDS( UC_JNF_2 ),  // load two operands into TEMP3 and TEMP2
  },
  { UC_JNF_2,
    WRITE_ADDRL( FLAG ),          // select FLAG for next read
    WRITE_NORIN( READ_MEMORY ),   // write FLAG into NORIN
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump (jump to new address)
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_LDI_2),       // continue with next OP code
    WRITE_OPCODE(UC_JPF_3)        // set PC to new address
  },
  /*-------------------------------------------------------------------------*/
  /* JMP abs / Jump to absolut address */  
  { OPCODE_JMP,
    LOAD_2_OPERANDS( UC_JPF_4 ),  // jump to new target address
  },
  /*-------------------------------------------------------------------------*/
  /* JSR abs / Call subroutine */  
  { OPCODE_JSR,
    LOAD_2_OPERANDS( UC_JSR_2 ),  // load two operands into TEMP3 and TEMP2
  },
  { UC_JSR_2,
    // LR_L := PC_L + 1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_L ),          // select PC_L for next memory read
    WRITE_ADDRL( READ_MEMORY ),   // set ADDRL to PC_L
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR_0( READ_MEMORY ), // read incremented PC_L from EPROM and write it to DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( LR_L ),          // select LR_L for next memory write
    CONT(UC_JSR_3),
    WRITE_RAM( READ_DATAR ),      // write to LR_L
    // LR_H := PC_H
    WRITE_ADDRL( PC_H ),          // select PC_H for next read
    WRITE_DATAR( READ_MEMORY ),   // read PC_H into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( LR_H ),          // select LR_H for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into LR_H
    // test if LR_L is zero
    WRITE_ADDRL( LR_L ),          // select LR_L for next memory read
    CONT(UC_JSR_4),
    WRITE_ADDRL( READ_MEMORY ),   // read LR_L into ADDRL
    WRITE_ADDRH( TABLE_TST ),     // select table for zero-test
    WRITE_NORIN( READ_MEMORY ),   // read test-value into NORIN
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    WRITE_OPCODE( UC_JSR_5 ),     // jump to code that increments LR_H
    WRITE_OPCODE( UC_JPF_4 ),     // jump to new target address
    },{ UC_JSR_5,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP,
    CONT(UC_JSR_6),
    // INC LR_H
    WRITE_ADDRL( READ_DATAR ),    // set ADDRL to DATAR (=LR_H)
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR_0( READ_MEMORY ), // read incremented PC_L from EEPROM and write it to DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( LR_H ),          // select LR_H register for next memory write
    WRITE_RAM( READ_DATAR ),      // write back LR_H
    WRITE_NOP,
    WRITE_OPCODE( UC_JPF_4 ),     // jump to new target address
  },
  /* RET / Return from subroutine */  
  { OPCODE_RET,
    // copy LR to PC
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( LR_L ),          // select LR_L for next read
    WRITE_DATAR( READ_MEMORY ),   // read LR_L into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_L ),          // select PC_L for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_L
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    CONT(UC_RET_2),
    WRITE_ADDRL( LR_H ),          // select LR_H for next read
    WRITE_DATAR( READ_MEMORY ),   // read LR_L into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_H ),          // select PC_H for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_H
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE( UC_NEXTOP ),    // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* PSH reg / Push register onto stack */
  { OPCODE_PSH,
    LOAD_OPERAND( UC_PSH_2 )
  },
  { UC_PSH_2,
    // load value where PC points to and load the value from this register into DATAR
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_DATAR( READ_MEMORY ),   // read content of reg into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // load stack pointer into ADDRL
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read stack-pointer into ADDRL
    WRITE_ADDRH( STACKPAGE_HI ),  // select stack-page
    CONT(UC_PSH_3),
    // write DATAR into the stack
    WRITE_RAM( READ_DATAR ),      // write DATAR into RAM
    // increment the stack-pointer
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read SP value into ADDRL
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    CONT(UC_PSH_4),
    WRITE_ADDRL( SP ),            // set stack-pointer register for next write
    WRITE_RAM( READ_DATAR ),      // write back the incremented stack-pointer
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOPs
    WRITE_NOP,
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* POP reg / Pull register back from stack */  
  { OPCODE_POP,
    LOAD_OPERAND( UC_POP_2 )
  },
  { UC_POP_2,
    // decrement the stack-pointer
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read SP value into ADDRL
    WRITE_ADDRH( TABLE_DEC ),     // select table for decrement
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next write
    WRITE_RAM( READ_DATAR ),      // write back the incremented stack-pointer
    CONT(UC_POP_3),
    // load stack pointer into ADDRL
    WRITE_ADDRL( READ_DATAR ),    // read stack-pointer into ADDRL
    WRITE_ADDRH( STACKPAGE_HI ),  // select stack-page
    // read the register back from stack and write it into the target register
    WRITE_DATAR( READ_MEMORY ),   // load value from stack
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP3 ),         // TEMP3 register for next read
    WRITE_ADDRL( READ_MEMORY ),   // load register name for next write
    WRITE_OPCODE(UC_SAP_2)        // write DATAR into RAM and execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* PHL / Push link registers LR_L and LR_H to the stack */
  { OPCODE_PHL,
    // load LR_L into DATAR
    WRITE_ADDRL( LR_L ),          // load address of LR_L in RAM
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // read register value into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // load stack pointer into ADDRL
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read stack-pointer into ADDRL
    WRITE_ADDRH( STACKPAGE_HI ),  // select stack-page
    CONT(UC_PHL_2),
    // write DATAR into the stack
    WRITE_RAM( READ_DATAR ),      // write DATAR into RAM
    // increment the stack-pointer
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read SP value into ADDRL
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    CONT(UC_PHL_3),
    WRITE_ADDRL( SP ),            // set stack-pointer register for next write
    WRITE_RAM( READ_DATAR ),      // write back the incremented stack-pointer
    // load LR_H into DATAR
    WRITE_ADDRL( LR_H ),          // load address of LR_H in RAM
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // read register value into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // load stack pointer into ADDRL
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    CONT(UC_PHL_4),
    WRITE_ADDRL( READ_MEMORY ),   // read stack-pointer into ADDRL
    WRITE_ADDRH( STACKPAGE_HI ),  // select stack-page
    // write DATAR into the stack
    WRITE_RAM( READ_DATAR ),      // write DATAR into RAM
    // increment the stack-pointer
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read SP value into ADDRL
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    CONT(UC_PHL_5),
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next write
    WRITE_RAM( READ_DATAR ),      // write back the incremented stack-pointer
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOP
    WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* RTS / Pop link register from stack and return */
  { OPCODE_RTS,
    // decrement the stack-pointer
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read SP value into ADDRL
    WRITE_ADDRH( TABLE_DEC ),     // select table for decrement
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next write
    CONT(UC_RTS_2),
    WRITE_RAM( READ_DATAR ),      // write back the incremented stack-pointer
    // load stack pointer into ADDRL
    WRITE_ADDRL( READ_DATAR ),    // read stack-pointer into ADDRL
    WRITE_ADDRH( STACKPAGE_HI ),  // select stack-page
    // read the LR_H register back from stack
    WRITE_DATAR( READ_MEMORY ),   // load value from stack
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( LR_H ),          // set LR_H register for next write
    CONT(UC_RTS_3),
    WRITE_RAM( READ_DATAR ),      // write DATAR into target register
    // decrement the stack-pointer
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read SP value into ADDRL
    WRITE_ADDRH( TABLE_DEC ),     // select table for decrement
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    CONT(UC_RTS_4),
    WRITE_ADDRL( SP ),            // set stack-pointer register for next write
    WRITE_RAM( READ_DATAR ),      // write back the incremented stack-pointer
    // load stack pointer into ADDRL
    WRITE_ADDRL( READ_DATAR ),    // read stack-pointer into ADDRL
    WRITE_ADDRH( STACKPAGE_HI ),  // select stack-page
    // read the LR_L register back from stack
    WRITE_DATAR( READ_MEMORY ),   // load value from stack
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    CONT(UC_RTS_5),
    WRITE_ADDRL( LR_L ),          // set LR_L register for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into target register
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOPs
    WRITE_NOP,
    WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE(OPCODE_RET)      // continue with RET OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* SEC / Set carry flag */
  { OPCODE_SEC,
    WRITE_DATAR_CONST( 0x01 ),    // set DATAR to 1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write to FLAG register
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOP
    WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* CLC / Clear carry flag */
  { OPCODE_CLC,
    WRITE_DATAR_CONST( 0x00 ),    // set DATAR to 0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write to FLAG register
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOP
    WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* AD / Add with carry flag set to 0 */
  { OPCODE_AD,
    WRITE_DATAR_CONST( 0x00 ),    // set DATAR to 0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write to FLAG register
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOP
    WRITE_NOP,
    WRITE_OPCODE( OPCODE_ADD )    // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* SU / Subtract with carry flag set to 1 */
  { OPCODE_SU,
    WRITE_DATAR_CONST( 0x01 ),    // set DATAR to 1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write to FLAG register
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOP
    WRITE_NOP,
    WRITE_OPCODE( OPCODE_SUB )    // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* RWL / rotate 16-bit word left */
  { OPCODE_RWL,
    LOAD_OPERAND( UC_RWL_2 )
  },
  { UC_RWL_2,
    // load register value and rotate it left through table lookup, write it to DATAR
    WRITE_ADDRL( TEMP3 ),         // set operand register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_ADDRL( READ_MEMORY ),   // load register value into ADDRL
    WRITE_ADDRH( TABLE_ROL ),     // select ROR table
    WRITE_DATAR( READ_MEMORY ),   // perform the table lookup
    // save register value to TEMP2
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    CONT(UC_RWL_3),
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP2. Bit 0 is inverted.
    // clear bit0 from value in DATAR
    WRITE_NORIN( 0x01 ),          // set NOR-input to 1
    WRITE_DATAR( READ_DATAR ),    // set bit0 in DATAR to 0
    // load FLAG register into NORIN
    WRITE_ADDRL( FLAG ),          // select FLAG for next read
    WRITE_NORIN( READ_MEMORY ),   // load FLAG into NORIN
    WRITE_DATAR( READ_DATAR ),    // set bit0 in DATAR value in FLAG (negated)
    WRITE_DATAR_0( READ_DATAR ),  // correct bit0
    CONT(UC_RWL_4),
    // save result in register
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // move content of TEMP3 into ADDRL
    WRITE_RAM( READ_DATAR ),      // write rotated value into register
    // set the FLAG register to new carry value
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
    WRITE_NORIN( READ_MEMORY ),   // load new (still inverted) flag into NOR input
    WRITE_DATAR( READ_DATAR ),    // dummy, relax timing for next operation
    WRITE_DATAR( 0x00 ),          // clear bits 1-7 in DATAR, correct polarity of bit0
    CONT(UC_RWL_5),
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into FLAG register
    // increment register number (rotate high-part now)
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of register and write it into ADDRL for table operation
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    CONT(UC_RWL_6),
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into TEMP3 register
    // continue in ROL OP-Code
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOPs
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP,
    WRITE_OPCODE( UC_ROL_2 )      // execute part 2 of the ROL OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* JLP / decrement R0 and loop while R0 is not zero */
  { OPCODE_JLP,
    // perform decrement on content of register
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( R0 ),            // write register address of R0 into ADDRL
    WRITE_ADDRL( READ_MEMORY ),   // read content of register and write it into ADDRL for table operation
    WRITE_ADDRH( TABLE_DEC ),     // select table for decrement
    WRITE_DATAR( READ_MEMORY ),   // perform table-lookup
    // write decremented value back into the source register
    WRITE_ADDRL( R0 ),            // write register address of R0 into ADDRL
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    CONT(SU_JLP_2),
    WRITE_RAM( READ_DATAR ),      // write incremented value from DATAR back into RAM
    // test R0 for zero
    WRITE_ADDRL( READ_DATAR ),    // write value of R0 into ADDRL
    WRITE_ADDRH( TABLE_TST ),     // select test-zero-table for lookup
    WRITE_DATAR( READ_MEMORY ),   // write result from table into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    CONT(SU_JLP_3),
    WRITE_RAM( READ_DATAR ),      // write content of DATAR into FLAG register
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOPs
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_OPCODE( OPCODE_JNF )    // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* OUT / Ouput register on output port */
  { OPCODE_OUT,
    LOAD_OPERAND( OPCODE_OUT2 )
  },
  { OPCODE_OUT2,
    WRITE_ADDRL( TEMP3 ),         // set operand register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of TEMP3 into ADDR_L
    WRITE_DATAR_0( READ_MEMORY ), // content of reg into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // output value from DATAR on output port
    WRITE_OUT( READ_DATAR ),      // write value from DATAR to OUTPUT port
    WRITE_NOP, WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* IN / read input port into FLAG */
  { OPCODE_IN,
    WRITE_DATAR_CONST( 0 ),       // reset the NOR-Flip-Flop
    WRITE_NOP,
    WRITE_DATAR( READ_INPORT ),   // read port value into NOR-Flip-Flop
    WRITE_DATAR( READ_DATAR ),    // read port value into DATAR   
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page by default
    WRITE_ADDRL( FLAG ),          // select FLAG for next write
    WRITE_RAM( READ_DATAR ),      // write value into FLAG
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* ICO / fast I2C output, send R1 over I2C and keep TXD low */
  { OPCODE_ICO,
    // initialize loop counter in TEMP2
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_DATAR_CONST( 0x01 ),    // loop init value
    WRITE_RAM( READ_DATAR ),      // write value into TEMP2
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    WRITE_NOP,
    WRITE_NOP,
    CONT(UC_ICO_2),               // Loop: send all bits in R1 over I2C
      // begin of loop
      WRITE_ADDRL( R1 ),            // select R1 for next read
      WRITE_ADDRL( READ_MEMORY ),   // read content of R1 into ADDR_L
      WRITE_ADDRH( TABLE_ROL ),     // select ROL table
      WRITE_DATAR( READ_MEMORY ),   // perform table-lookup
      WRITE_ADDRL( R1 ),            // select R1 for next write
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_RAM( READ_DATAR ),      // write DATAR into R1
      CONT(UC_ICO_3),
      WRITE_NORIN( READ_DATAR ),    // load (inverted) bit7 of R1 into NOR input
      WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
      JUMP_IF_ZERO,                 // perform the jump
      WRITE_ADDRL( JUMP(6) ),       // write jump-target-address in ADDRL
      JUMP_ALWAYS,                  // jump over the NOP
      WRITE_NOP,
      WRITE_OPCODE(UC_ICO_4),
      WRITE_OPCODE(UC_ICO_6),
      },{ UC_ICO_4,
      WRITE_NOP, WRITE_NOP, WRITE_NOP,
      WRITE_NOP, WRITE_NOP, WRITE_NOP,
      WRITE_NOP,
      CONT(UC_ICO_5),
      // send a zero over SDA
      WRITE_OUT( 0x00 ),            // TXD=0, SDA=0, SCL=0
      WRITE_ADDRL( RADR3 ),         // select RADR3 for next write
      WRITE_DATAR_CONST(UC_ICO_7),  // return address for ICO instruction
      WRITE_RAM( READ_DATAR ),      // write DATAR to RAM
      WRITE_NOP,
      WRITE_DATAR_CONST( 0x00 ),    // save current output value
      WRITE_OUT( 0x04 ),            // TXD=0, SDA=0, SCL=1
      WRITE_OPCODE( UC_XOR_8 ),     // send bit
      // send a one over SDA
      },{ UC_ICO_6,                     
      WRITE_OUT( 0x02 ),            // TXD=0, SDA=1, SCL=0
      WRITE_ADDRL( RADR3 ),         // select RADR3 for next write
      WRITE_DATAR_CONST(UC_ICO_7),  // return address for ICO instruction
      WRITE_RAM( READ_DATAR ),      // write DATAR to RAM
      WRITE_NOP,
      WRITE_DATAR_CONST( 0x02 ),    // save current output value
      WRITE_OUT( 0x06 ),            // TXD=0, SDA=1, SCL=1
      WRITE_OPCODE( UC_XOR_8 ),     // send bit
      },{ UC_ICO_7,                     
      WRITE_NOP, WRITE_NOP, WRITE_NOP,
      WRITE_NOP, WRITE_NOP, WRITE_NOP,
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
      CONT(UC_ICO_8),
      // continue with next bit
      // rotate TEMP2 right (loop index)
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP2 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_OUT( READ_DATAR ),      // TXD=0, SDA=X, SCL=0
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP2 back to RAM
      CONT(UC_ICO_9),
    // end loop
    WRITE_NORIN( READ_DATAR ),
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    WRITE_OUT( 0x02 ),            // TXD=0, SDA=1, SCL=0
    WRITE_NOP, WRITE_NOP,
    WRITE_OPCODE(UC_LDI_2),       // quit (jump to exit routine)
    WRITE_OPCODE(UC_ICO_2)        // jump to begin of the loop
  },
  /*-------------------------------------------------------------------------*/
  /* ICL / fast I2C input, set SCL low (SDA=1,TXD=0) and call "ROL R0" */
  { OPCODE_ICL,
    WRITE_OUT( 0x02 ),            // TXD=0, SDA=1, SCL=0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next write
    WRITE_DATAR_CONST( R0 ),      // load address of R0 into DATAR
    WRITE_RAM( READ_DATAR ),      // write DATAR into TEMP3 register
    WRITE_NOP, WRITE_NOP,
    WRITE_OPCODE(UC_ROL_2)        // execute "ROL R0"
  },
  /*-------------------------------------------------------------------------*/
  /* ICH / fast I2C input, set SCL high (SDA=1,TXD=0) */
  { OPCODE_ICH,
    WRITE_OUT( 0x06 ),            // TXD=0, SDA=1, SCL=1
    WRITE_ADDRL( JUMP(7) ),       // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // jump over the NOPs
    WRITE_NOP, WRITE_NOP,
    WRITE_NOP, WRITE_NOP,
    WRITE_OPCODE(UC_INCPC_NEXTOP) // execute next OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* SSV / save stack pointer to PAR1 and execute LD opcode */
  { OPCODE_SSV,
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // select SP for next read
    WRITE_DATAR( READ_MEMORY ),   // read content of SP into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PAR1 ),          // select PAR1 for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    WRITE_NOP,
    WRITE_OPCODE(OPCODE_LD)       // execute LD OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* SRT / load stack pointer from a register and do RET */
  { OPCODE_SRT,
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PAR1 ),          // select PAR1 for next read
    WRITE_DATAR( READ_MEMORY ),   // read content of PAR1 into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( SP ),            // select SP for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    WRITE_NOP,
    WRITE_OPCODE(OPCODE_RET)      // execute RET OP-Code
  },
  /*-------------------------------------------------------------------------*/
  /* TJZ / test and jump to 0x1603 if zero */
  { OPCODE_TJZ,
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_DATAR_CONST( 0x03 ),    // low byte of address 0x1603
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next write
    WRITE_DATAR_CONST( 0x16 ),    // high byte of address 0x1603
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    WRITE_OPCODE(UC_JPF_2)        // execute JPF OP-Code
  },
};


/*===========================================================================*/


static void compile_microcode(uint8_t *buffer)
{
  uint32_t j, op, idx, v;
  uint8_t *b, b1, b2;
  uint8_t b1_l;

  for (j = 0; j < NUM_OPCODES; j++)
  {
    op = ucode_table[j][0];
    if ((op == 0x00) && (j != 0))
      break;
    b1_l = 0;
    b = buffer + (op * 16);
    for (idx = 0; idx < 8; idx++)
    {
      v = ucode_table[j][idx+1];
      b1 = (uint8_t)(v >> 8) ^ (CTL_RES_NORFF | CTL_SEL_INPORT);
      b2 = (uint8_t)v;
      if ((v & UFLAG_REGSOURCE) == 0)
      {
        b1 |= CONST_FLAGS;  // no register or memory access, but load constant value
        if ((b1 & CTL_CLK_MASK) == CTL_CLK_RAM)
        {
          printf("Error in OP-Code %02X, line %d: Can not write a constant value directly into RAM\n", op, idx+1);
        }
        if ((b1 & CTL_CLK_MASK) == CTL_CLK_DATAR)
        {
          if (b1_l & CTL_ASEL_MEMORY)
          {
            printf("Error in OP-Code %02X, line %d: No memory operation allowed before writing constant value to DATAR\n", op, idx+1);
          }
        }
      }
      *b++ = b1;
      *b++ = b2;
      b1_l = b1;
    }
  }
}

static void generate_rol_table(uint8_t *buffer)
{
  uint8_t *b = &buffer[TABLE_ROL*256];
  unsigned i, a;

  for (i=0; i<256; i++)
  {
    a = (uint8_t)i;
    b[i] = (a<<1) | (a>>7);
  }
}

static void generate_ror_table(uint8_t *buffer)
{
  uint8_t *b = &buffer[TABLE_ROR*256];
  unsigned i, a;

  for (i=0; i<256; i++)
  {
    a = (uint8_t)i;
    b[i] = ((a<<7) | (a>>1)) ^ 1;
  }
}

static void generate_inc_table(uint8_t *buffer)
{
  uint8_t *b = &buffer[TABLE_INC*256];
  unsigned i;

  for (i=0; i<256; i++)
  {
    b[i] = (uint8_t)(i + 1)^1;
  }
}

static void generate_dec_table(uint8_t *buffer)
{
  uint8_t *b = &buffer[TABLE_DEC*256];
  unsigned i;

  for (i=0; i<256; i++)
  {
    b[i] = (uint8_t)(i - 1)^1;
  }
}

static void generate_not_table(uint8_t *buffer)
{
  uint8_t *b = &buffer[TABLE_NOT*256];
  unsigned i;

  for (i=0; i<256; i++)
  {
    b[i] = (uint8_t)(i^0xFF);
  }
}

static void generate_testzero_table(uint8_t *buffer)
{
  uint8_t v, *b = &buffer[TABLE_TST*256];
  unsigned i, b1, b2, b3, c, s, a;

  for (i=0; i<256; i++)
  {
    b[i] = (i == 0) ? 0x01 : 0x00;
  }

  // implement also a 1-bit full-adder into this table
  for (i=0; i<8; i++)
  {
    b1 = (i>>0) & 1;
    b2 = (i>>1) & 1;
    b3 = (i>>2) & 1;

    s = b1 + b2 + b3;
    c = (s > 1) ? 1 : 0;
    s&= 1;

    c^=1;  // invert carry (inversion needed by microcode)

    v = (uint8_t) ((c << 7) | (s << 1));

    a = (b1 << 7) | (b2 << 6) | b3;
    a^= 0xC1;  // invert address bits (inversion needed by microcode)
    b[a] |= v;
  }
}


/** Generate microcode.
 *  @param buffer   ptr to output buffer with at least 16kb space
 */
void generate_my4th_xs_microcode_eprom(uint8_t *buffer)
{
  compile_microcode(buffer);

  generate_inc_table(buffer);
  generate_dec_table(buffer);
  generate_rol_table(buffer);
  generate_ror_table(buffer);
  generate_not_table(buffer);
  generate_testzero_table(buffer);
}
