#include <stdio.h>
#include <string.h>
#ifdef ARCH_MYNOR
#include "mynor_microcode.h"
#else
#include "my4th_microcode.h"
#endif

/* MACRO: increment PC */
#define MACRO_INC_PC  \
    /* INC PC_L */                                                                                                    \
    WRITE_ADDRH( ZEROPAGE_HI ),   /* select zero-page by default                                                   */ \
    WRITE_ADDRL( PC_L ),          /* select PC_L register for next memory write                                    */ \
    WRITE_ADDRL( READ_MEMORY ),   /* set ADDRL to PC_L                                                             */ \
    WRITE_ADDRH( TABLE_INC   ),   /* select table for increment                                                    */ \
    WRITE_DATAR_0( READ_MEMORY ), /* read incremented PC_L from EEPROM and write it to DATAR, set NOR-input 1 to 0 */ \
    WRITE_ADDRH( ZEROPAGE_HI ),   /* select zero-page by default                                                   */ \
    WRITE_ADDRL( PC_L ),          /* select PC_L register for next memory write                                    */ \
    WRITE_RAM( READ_DATAR ),      /* write back PC_L                                                               */ \
    /* prepare for zero-test of PC_L */                                                                               \
    WRITE_ADDRH( TABLE_TST  ),    /* select table for zero-test                                                    */ \
    WRITE_ADDRL( READ_DATAR ),    /* write value to test for zero in ADDRL                                         */ \
    WRITE_NORIN( READ_MEMORY ),   /* read test-value into NORIN                                                    */ \
    /* prepare jump behind code for incrementing PC_H */                                                              \
    WRITE_ADDRL( JUMPOFS(10) ),   /* write jump-target-address in ADDRL                          */ \
    JUMP_IF_ZERO,                 /* perform the jump                                                              */ \
    /* INC PC_H */                                                                                                    \
    WRITE_ADDRH( ZEROPAGE_HI ),   /* select zero-page by default                                                   */ \
    WRITE_ADDRL( PC_H ),          /* select PC_H register for next memory write                                    */ \
    WRITE_ADDRL( READ_MEMORY ),   /* set ADDRL to PC_L                                                             */ \
    WRITE_ADDRH( TABLE_INC   ),   /* select table for increment                                                    */ \
    WRITE_DATAR_0( READ_MEMORY ), /* read incremented PC_L from EEPROM and write it to DATAR                       */ \
    WRITE_ADDRH( ZEROPAGE_HI ),   /* select zero-page by default                                                   */ \
    WRITE_ADDRL( PC_H ),          /* select PC_H register for next memory write                                    */ \
    WRITE_RAM( READ_DATAR )       /* write back PC_H                                                               */

/* MACRO: Jump to end of uCode to execute next OP-Code */
#define MACRO_NEXTOP  \
    /* Execute next OP. Jump to offset 120. */                                                                         \
    WRITE_ADDRL( 120*2 ),         /* set new address offset                                                        */  \
    JUMP_ALWAYS,                  /* perform the jump                                                              */  \
    UFLAG_NEXTOP                  /* advice compiler to append code to fetch next OP-code                          */

/* MACRO: Load PC into ADDRL / ADDRH register */
#define MACRO_LOAD_PC_ADDR  \
    WRITE_ADDRH( ZEROPAGE_HI ),   /* select zero-page by default                */ \
    WRITE_ADDRL( PC_L ),          /* select PC_L register for next read         */ \
    WRITE_DATAR_0( READ_MEMORY ), /* read memory and write PCL to DATA-register */ \
    WRITE_DATAR( READ_DATAR ),    /* correct bit0                               */ \
    WRITE_ADDRL( PC_H ),          /* select PC_H register for next read         */ \
    WRITE_ADDRH( READ_MEMORY ),   /* set ADDRH to PC_H                          */ \
    WRITE_ADDRL( READ_DATAR )     /* set ADDRL to PC_L                          */

/* MACRO: Fetch next OP-Code. This code must reside at the end of the uCode space! */
#define MACRO_NEXTOP_INLINE \
    WRITE_ADDRH( ZEROPAGE_HI ),   /* select zero-page by default                 */ \
    WRITE_ADDRL( PC_L ),          /* select PC_L register for next read          */ \
    WRITE_DATAR_0( READ_MEMORY ), /* read memory and write PCL to DATA-register  */ \
    WRITE_DATAR( READ_DATAR ),    /* correct bit0                                */ \
    WRITE_ADDRL( PC_H ),          /* select PC_H register for next read          */ \
    WRITE_ADDRH( READ_MEMORY ),   /* set ADDRH to PC_H                           */ \
    WRITE_ADDRL( READ_DATAR ),    /* set ADDRL to PC_L                           */ \
    WRITE_OPCODE( READ_MEMORY ),  /* read OP-Code from memory where PC points to */ \
    // counter roll-over, continue with next OP

static uint32_t nextop_table[8] = {
    MACRO_NEXTOP_INLINE
};


static uint32_t ucode_table[NUM_OPCODES][128+1] =
{
  /* RST / Reset */
  { OPCODE_RST,
    WRITE_NOP,                    // dummy, allow A15 to recover
    WRITE_NOP,                    // dummy, allow A15 to recover
    WRITE_NOP,                    // dummy, allow A15 to recover
    WRITE_NOP,                    // dummy, allow A15 to recover
    WRITE_NOP,                    // dummy, allow A15 to recover
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page by default
    WRITE_DATAR_CONST( 0x00 ),    // set constant value for next memory write
    WRITE_ADDRL( SP ),            // select SP register for next memory write
    WRITE_RAM( READ_DATAR ),      // SP := 0x00
    WRITE_ADDRL( PC_L ),          // select PC_L register for next memory write
    WRITE_RAM( READ_DATAR ),      // PC_L := 0x00
    WRITE_ADDRL( PC_H ),          // select PC_H register for next memory write
    WRITE_DATAR_CONST( (CODE_START>>8) ),    // set constant value for next memory write
    WRITE_RAM( READ_DATAR ),      // PC_H := 0x24
    WRITE_ADDRL( 127*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump (execute first OP-Code)

    // SAP code for RAM address range 0x8800-0x8FFF    
    LOOP1|  // entry-point for decoder of address-range 0x8800-0x8FFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(53) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0x8800 - 0x8BFF, else 0x8C00-0x8FFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0x8800 - 0x89FF, else 0x8A00-0x8BFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x8900-0x89FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x89 ),          // set ADDR_H to 0x89
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x8800-0x88FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x88 ),          // set ADDR_H to 0x88
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0x8A00 - 0x8BFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x8B00-0x8BFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x8B ),          // set ADDR_H to 0x8B
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x8A00-0x8AFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x8A ),          // set ADDR_H to 0x8A
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0x8C00 - 0x8FFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0x8C00 - 0x8DFF, else 0x8E00-0x8FFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x8D00-0x8DFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x8D ),          // set ADDR_H to 0x8D
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x8C00-0x8CFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x8C ),          // set ADDR_H to 0x8C
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0x8E00 - 0x8FFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x8F00-0x8FFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x8F ),          // set ADDR_H to 0x8F
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x8E00-0x8EFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x8E ),          // set ADDR_H to 0x8E
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP,                    // dummy for alignment
    WRITE_NOP,                    // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // execute next OP Code
    WRITE_OPCODE(OPCODE_SAP2)
  },
  /*-------------------------------------------------------------------------*/
  /* LD reg,# / load register with immediate value */
  { OPCODE_LDI,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save dest.addr
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP1
    // increment PC
    MACRO_INC_PC,
    // load second OP-Code parameter into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read content of register pointed by src.addr into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // load dest.addr from TEMP1 and perform move operation
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( READ_MEMORY ),   // read dest.addr into ADDRL
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    // increment PC and execute next OP Code
    MACRO_INC_PC, MACRO_NEXTOP
  },
  /*-------------------------------------------------------------------------*/
  /* LD reg,reg / load register from other register */  
  { OPCODE_LD,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save dest.addr
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP1
    // increment PC
    MACRO_INC_PC,
    // load value from reg where second OP-Code parameter points to
    MACRO_LOAD_PC_ADDR,
    WRITE_ADDRL( READ_MEMORY ),   // read OP-Code parameter from memory as addr.lo for next read
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // read content of register pointed by src.addr into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // load dest.addr from TEMP1 and perform move operation
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read dest.addr into ADDRL
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    // increment PC and execute next OP Code
    MACRO_INC_PC, MACRO_NEXTOP
  },
  /*-------------------------------------------------------------------------*/
  /* LDAI # / Load ACCU with immediate value */
  { OPCODE_LDAI,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // store value in ACCU
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( ACCU ),          // select ACCU for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    // increment PC and execute next OP Code
    MACRO_INC_PC,
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump and execute next OP-Code

    // SAP code for RAM address range 0x9400-0x97FF    
    LOOP1|  // entry-point for decoder of address-range 0x9400-0x97FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0x9400 - 0x95FF, else 0x9600-0x97FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x9500-0x95FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x95 ),          // set ADDR_H to 0x95
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x9400-0x94FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x94 ),          // set ADDR_H to 0x94
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0x9600 - 0x97FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x9700-0x97FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x97 ),          // set ADDR_H to 0x97
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x9600-0x96FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x96 ),          // set ADDR_H to 0x96
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP,                    // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* LDA reg / Load ACCU from register */
  { OPCODE_LDA,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into ADDRL. Then load reg into DATAR.
    MACRO_LOAD_PC_ADDR,
    WRITE_ADDRL( READ_MEMORY ),   // read OP-Code parameter into ADDRL
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // content of reg into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // store value in ACCU
    WRITE_ADDRL( ACCU ),          // select ACCU for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    // increment PC and execute next OP Code
    MACRO_INC_PC,
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump and execute next OP-Code

    // SAP code for RAM address range 0x9000-0x93FF    
    LOOP1|  // entry-point for decoder of address-range 0x9000-0x97FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 114*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0x9000 - 0x93FF, else 0x9400-0x97FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0x9000 - 0x91FF, else 0x9200-0x93FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x9100-0x91FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x91 ),          // set ADDR_H to 0x91
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x9000-0x90FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x90 ),          // set ADDR_H to 0x90
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0x9200 - 0x93FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x9300-0x93FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x93 ),          // set ADDR_H to 0x93
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x9200-0x92FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x92 ),          // set ADDR_H to 0x92
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_OPCODE( OPCODE_LDAI ),  // continue code execution at OP-code LDAI,116  : range 0x9800-0x9FFF
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* LAP / Load Accu indirect through PTR register */  
  { OPCODE_LAP,
    // load PTR_L and PTR_H into ADDRL and ADDRH
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_DATAR_0( READ_MEMORY ), // read content of first ZP-register into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_H ),         // select PTR_H for next read
    WRITE_ADDRH( READ_MEMORY ),   // store content of second ZP-register in ADDRH
    WRITE_ADDRL( READ_DATAR ),    // move DATAR into ADDRL (content of first ZP-register)
    // read memory pointed by PTR into DATAR
    WRITE_DATAR( READ_MEMORY ),   // finally read the value from address pointed by ZP and ZP+1
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // store DATAR to ACCU
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( ACCU ),          // select ACCU for next write
    WRITE_RAM( READ_DATAR ),      // DATAR into the ACCU
    // increment PC and execute next OP Code
    LOOP2|
    MACRO_INC_PC,
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump and execute next OP-Code

    // SAP code for RAM address range 0x9800-0x9BFF    
    LOOP1|  // entry-point for decoder of address-range 0x9800-0x9BFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 117*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0x9800 - 0x9BFF, else 0x9C00-0x9FFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0x9800 - 0x99FF, else 0x9A00-0x9BFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x9900-0x99FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x99 ),          // set ADDR_H to 0x99
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x9800-0x98FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x98 ),          // set ADDR_H to 0x98
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0x9A00 - 0x9BFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x9B00-0x9BFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x9B ),          // set ADDR_H to 0x9B
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x9A00-0x9AFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x9A ),          // set ADDR_H to 0x9A
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_OPCODE( OPCODE_STA ),   // continue code execution at OP-code STA,118  : range 0x9C00-0x9FFF
    WRITE_ADDRL( LOOP( LOOP2 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* STA reg / Store ACCU to register */
  { OPCODE_STA,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR, store register name in TEMP1
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    // load ACCU into DATAR
    WRITE_ADDRL( ACCU ),          // select ACCU for next write
    WRITE_DATAR( READ_MEMORY ),   // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save ACCU in target register
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read address of register into ADDRL
    WRITE_RAM( READ_DATAR ),      // write DATAR into memory
    // increment PC and execute next OP Code
    MACRO_INC_PC,
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump and execute next OP-Code

    // SAP code for RAM address range 0x9C00-0x9FFF
    LOOP1|  // entry-point for decoder of address-range 0x9C00-0x9FFF
    // here: ADDR 0x9800 - 0x9BFF, else 0x9C00-0x9FFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0x9C00 - 0x9DFF, else 0x9E00-0x9FFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x9D00-0x9DFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x9D ),          // set ADDR_H to 0x9D
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x9C00-0x9CFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x9C ),          // set ADDR_H to 0x9C
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0x9E00 - 0x9FFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x9F00-0x9FFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x9F ),          // set ADDR_H to 0x9F
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x9E00-0x9EFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x9E ),          // set ADDR_H to 0x9E
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP,                    // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* SAP / Store Accu to memory pointed by PTR register */  
  { OPCODE_SAP,
    MACRO_INC_PC,
    // load PTR_H 
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PTR_H ),         // select PTR_H for next read
    WRITE_ADDRL( READ_MEMORY ),   // write PTR_H into ADDRL for table lookup
    WRITE_ADDRH( TABLE_ROL ),     // select ROL table
    WRITE_ADDRL( READ_MEMORY ),   // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 2) ^ 0x01  (bit0 = PTR_H.6 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(45) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.6 == 1), handle address range 0xC000-0xFFFF
    // here: ADDR 0x8000 - 0xBFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 3) ^ 0x81  (bit0 = PTR_H.5 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(21) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.5 == 1), handle address range 0xA000-0xBFFF
    // here: ADDR 0x8000 - 0x9FFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 4) ^ 0xC1  (bit0 = PTR_H.4 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(9) ),    // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.4 == 1), handle address range 0x9000-0x9FFF
    // here: ADDR 0x8000 - 0x8FFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 5) ^ 0xE1  (bit0 = PTR_H.3 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 124*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.3 == 1), handle address range 0x8800-0x8FFF
    // here: ADDR 0x8000 - 0x87FF
    WRITE_ADDRL( 117*2 ),
    JUMP_ALWAYS,
    // here: ADDR 0x9000 - 0x9FFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 5) ^ 0xE1  (bit0 = PTR_H.3 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 114*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.3 == 1), handle address range 0x9800-0x9FFF
    // here: ADDR 0x9000 - 0x97FF
    WRITE_ADDRL( 115*2 ),
    JUMP_ALWAYS,
    // here: ADDR 0xA000 - 0xBFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 4) ^ 0xC1  (bit0 = PTR_H.4 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(9) ),    // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.4 == 1), handle address range 0xB000-0xBFFF
    // here: ADDR 0xA000 - 0xAFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 5) ^ 0xE1  (bit0 = PTR_H.3 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 112*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.3 == 1), handle address range 0xA800-0xAFFF
    // here: ADDR 0xA000 - 0xA7FF
    WRITE_ADDRL( 116*2 ),
    JUMP_ALWAYS,
    // here: ADDR 0xB000 - 0xBFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 5) ^ 0xE1  (bit0 = PTR_H.3 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 111*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.3 == 1), handle address range 0xB800-0xBFFF
    // here: ADDR 0xB000 - 0xB7FF
    WRITE_ADDRL( 113*2 ),
    JUMP_ALWAYS,
    // here: ADDR 0xC000 - 0xFFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 3) ^ 0x81  (bit0 = PTR_H.5 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(21) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.5 == 1), handle address range 0xE000-0xFFFF
    // here: ADDR 0xC000 - 0xDFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 4) ^ 0xC1  (bit0 = PTR_H.4 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(9) ),    // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.4 == 1), handle address range 0xD000-0xDFFF
    // here: ADDR 0xC000 - 0xCFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 5) ^ 0xE1  (bit0 = PTR_H.3 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 109*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.3 == 1), handle address range 0xC800-0xCFFF
    // here: ADDR 0xC000 - 0xC7FF
    WRITE_ADDRL( 110*2 ),
    JUMP_ALWAYS,
    // here: ADDR 0xD000 - 0xDFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 5) ^ 0xE1  (bit0 = PTR_H.3 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 123*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.3 == 1), handle address range 0xD800-0xDFFF
    // here: ADDR 0xD000 - 0xD7FF
    WRITE_ADDRL( 122*2 ),
    JUMP_ALWAYS,
    // here: ADDR 0xE000 - 0xFFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 4) ^ 0xC1  (bit0 = PTR_H.4 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 121*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.4 == 1), handle address range 0xF000-0xFFFF
    // here: ADDR 0xE000 - 0xEFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 5) ^ 0xE1  (bit0 = PTR_H.3 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 119*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.3 == 1), handle address range 0xE800-0xEFFF
    // here: ADDR 0xE000 - 0xE7FF
    WRITE_ADDRL( 118*2 ),
    JUMP_ALWAYS,

    // 109*2
    WRITE_OPCODE( OPCODE_RWL ),   // continue code execution at OP-code RWL,110  : range 0xC800-0xCFFF
    // 110*2
    WRITE_OPCODE( OPCODE_JLP ),   // continue code execution at OP-code PHA,111  : range 0xC000-0xC7FF
    // 111*2
    WRITE_OPCODE( OPCODE_RTS ),   // continue code execution at OP-code RTS,112  : range 0xB800-0xBFFF
    // 112*2
    WRITE_OPCODE( OPCODE_RET ),   // continue code execution at OP-code RET,113  : range 0xA800-0xAFFF
    // 113*2
    WRITE_OPCODE( OPCODE_PHL ),   // continue code execution at OP-code PHL,114  : range 0xB000-0xB7FF
    // 114*2
    WRITE_OPCODE( OPCODE_LAP ),   // continue code execution at OP-code LAP,115  : range 0x9800-0x9FFF
    // 115*2
    WRITE_OPCODE( OPCODE_LDA ),   // continue code execution at OP-code LDA,116  : range 0x9000-0x97FF
    // 116*2
    WRITE_OPCODE( OPCODE_ADD ),   // continue code execution at OP-code ADD,117  : range 0xA000-0xA7FF
    // 117*2
    WRITE_OPCODE( OPCODE_SAP2 ),  // continue code execution at OP-code SAP2,118 : range 0x8000-0x87FF
    // 118*2
    WRITE_OPCODE( OPCODE_AD  ),   // continue code execution at OP-code AD,119   : range 0xE000-0xE7FF
    // 119*2
    WRITE_OPCODE( OPCODE_SU  ),   // continue code execution at OP-code SU,120   : range 0xE800-0xEFFF
    // 120*2
    WRITE_NOP,
    // 121*2
    WRITE_OPCODE( OPCODE_OUT ),   // continue code execution at OP-code OUT,122 : range 0xF000-0xFFFF
    // 122*2
    WRITE_OPCODE( OPCODE_SEC ),   // continue code execution at OP-code SEC,123 : range 0xD000-0xD7FF
    // 123*2
    WRITE_OPCODE( OPCODE_CLC ),   // continue code execution at OP-code CLC,124 : range 0xD800-0xDFFF
    // 124*2
    WRITE_OPCODE( OPCODE_RST ),   // continue code execution at OP-code RST,125  : range 0x8800-0x8FFF
  },
  /*-------------------------------------------------------------------------*/
  /* This is the second part of the SAP OP  */  
  { OPCODE_SAP2,
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // execute next OP
    
    LOOP1|  // entry-point for decoder of address-range 0x8000-0x87FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(53) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0x8000 - 0x83FF, else 0x8400-0x8700
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0x8000 - 0x81FF, else 0x8200-0x83FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x8100-0x81FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x81 ),          // set ADDR_H to 0x81
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x8000-0x80FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x80 ),          // set ADDR_H to 0x80
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0x8200 - 0x83FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x8300-0x83FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x83 ),          // set ADDR_H to 0x83
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x8200-0x82FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x82 ),          // set ADDR_H to 0x82
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0x8400 - 0x87FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0x8400 - 0x85FF, else 0x8600-0x87FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x8500-0x85FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x85 ),          // set ADDR_H to 0x85
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x8400-0x84FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x84 ),          // set ADDR_H to 0x84
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0x8600 - 0x87FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0x8700-0x87FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x87 ),          // set ADDR_H to 0x87
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0x8600-0x86FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0x86 ),          // set ADDR_H to 0x86
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP,                    // dummy for alignment
    // the entry-point for the decoder 0x8000-0x87FF is SAP2,118
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* INC reg / Increment value in register */  
  { OPCODE_INC,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to and save this value in TEMP1
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 register for next write
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_RAM( READ_DATAR ),      // write register number into TEMP1
    // perform increment on content of register
    WRITE_ADDRL( READ_DATAR ),    // write register number into ADDRL
    WRITE_ADDRL( READ_MEMORY ),   // read content of register and write it into ADDRL for table operation
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR( READ_MEMORY ),   // perform table-lookup
    // write incremented value back into the source register
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read register number from TEMP1 and use it as ADDRL
    WRITE_RAM( READ_DATAR ),      // write incremented value from DATAR back into RAM
    // increment PC and execute next OP Code
    MACRO_INC_PC,
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump and execute next OP-Code

    // SAP code for RAM address range 0xAC00-0xAFFF
    LOOP1|  // entry-point for decoder of address-range 0xAC00-0xAFFF
    // here: ADDR 0xA800 - 0xABFF, else 0xAC00-0xAFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xAC00 - 0xADFF, else 0xAE00-0xAFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xAD00-0xADFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xAD ),          // set ADDR_H to 0xAD
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xAC00-0xACFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xAC ),          // set ADDR_H to 0xAC
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xAE00 - 0xAFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xAF00-0xAFFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xAF ),          // set ADDR_H to 0xAF
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xAE00-0xAEFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xAE ),          // set ADDR_H to 0xAE
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP,                    // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* DEC reg / Decrement value in register */  
  { OPCODE_DEC,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to and save this value in TEMP1
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 register for next write
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_RAM( READ_DATAR ),      // write register number into TEMP1
    // perform decrement on content of register
    WRITE_ADDRL( READ_DATAR ),    // write register number into ADDRL
    WRITE_ADDRL( READ_MEMORY ),   // read content of register and write it into ADDRL for table operation
    WRITE_ADDRH( TABLE_DEC ),     // select table for decrement
    WRITE_DATAR( READ_MEMORY ),   // perform table-lookup
    // write incremented value back into the source register
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read register number from TEMP1 and use it as ADDRL
    WRITE_RAM( READ_DATAR ),      // write incremented value from DATAR back into RAM
    // increment PC and execute next OP Code
    MACRO_INC_PC,
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump and execute next OP-Code

    // code for OPCODE_OUT reg:
#ifdef ARCH_MYNOR
    LOOP1|
    MACRO_INC_PC,
    // load value where PC points to into ADDRL. Then load reg into DATAR.
    MACRO_LOAD_PC_ADDR,
    WRITE_ADDRL( READ_MEMORY ),   // read OP-Code parameter into ADDRL
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // content of reg into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // output value from DATAR on output port
    WRITE_ADDRH( 0xE0 ),          // load ADDRH with port address of OUTPORT3
    WRITE_RAM  ( READ_DATAR ),    // write value from DATAR to output port 3
#else  // My4TH
    WRITE_NOP,                    // dummy for alignment
    LOOP1|
    MACRO_INC_PC,
    // load value where PC points to into ADDRL. Then load reg into DATAR.
    MACRO_LOAD_PC_ADDR,
    WRITE_ADDRL( READ_MEMORY ),   // read OP-Code parameter into ADDRL
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // content of reg into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // output value from DATAR on output port
    WRITE_OUT( READ_DATAR ),      // write value from DATAR to OUTPUT port
#endif
    WRITE_OPCODE( OPCODE_AND ),   // continue code execution at OP-code AND,99 : INC_PC and execute next op-code
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP,                    // dummy for alignment
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* ROL reg / Rotate reg left with carry in FLAG register */
  { OPCODE_ROL,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save parameter (register name) to TEMP1
    LOOP2|
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP1
    // load register value and rotate it left through table lookup, write it to DATAR
    WRITE_ADDRL( READ_DATAR ),    // set operand register for next read
    WRITE_ADDRL( READ_MEMORY ),   // load register value into ADDRL
    WRITE_ADDRH( TABLE_ROL ),     // select ROR table
    WRITE_DATAR( READ_MEMORY ),   // perform the table lookup
    // save register value to TEMP2
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP2. Bit 0 is inverted.
    // clear bit0 from value in DATAR
    WRITE_NORIN( 0x01 ),          // set NOR-input to 1
    WRITE_DATAR( READ_DATAR ),    // set bit0 in DATAR to 0
    // load FLAG register into NORIN
    WRITE_ADDRL( FLAG ),          // select FLAG for next read
    WRITE_NORIN( READ_MEMORY ),   // load FLAG into NORIN
    WRITE_DATAR( READ_DATAR ),    // set bit0 in DATAR value in FLAG (negated)
    WRITE_DATAR_0( READ_DATAR ),  // correct bit0
    // save result in register
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
    WRITE_ADDRL( READ_MEMORY ),   // move content of TEMP1 into ADDRL
    WRITE_RAM( READ_DATAR ),      // write rotated value into register
    // set the FLAG register to new carry value
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
    WRITE_NORIN( READ_MEMORY ),   // load new (still inverted) flag into NOR input
    WRITE_DATAR( READ_DATAR ),    // dummy, relax timing for next operation
    WRITE_DATAR( 0x00 ),          // clear bits 1-7 in DATAR, correct polarity of bit0
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into FLAG register
    // increment PC and execute next OP Code
    MACRO_INC_PC,
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump and execute next OP-Code

    // OPCODE_RWL part 2 :
    LOOP1|
    // load value where PC points to into DATAR
    WRITE_ADDRL( READ_DATAR ),    // set ADDRL to PC_L
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save parameter (register name) to TEMP1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP1
    // load register value and rotate it left through table lookup, write it to DATAR
    WRITE_ADDRL( READ_DATAR ),    // set operand register for next read
    WRITE_ADDRL( READ_MEMORY ),   // load register value into ADDRL
    WRITE_ADDRH( TABLE_ROL ),     // select ROR table
    WRITE_DATAR( READ_MEMORY ),   // perform the table lookup
    // save register value to TEMP2
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP2. Bit 0 is inverted.
    // clear bit0 from value in DATAR
    WRITE_NORIN( 0x01 ),          // set NOR-input to 1
    WRITE_DATAR( READ_DATAR ),    // set bit0 in DATAR to 0
    // load FLAG register into NORIN
    WRITE_ADDRL( FLAG ),          // select FLAG for next read
    WRITE_NORIN( READ_MEMORY ),   // load FLAG into NORIN
    WRITE_DATAR( READ_DATAR ),    // set bit0 in DATAR value in FLAG (negated)
    WRITE_DATAR_0( READ_DATAR ),  // correct bit0
    // save result in register
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
    WRITE_ADDRL( READ_MEMORY ),   // move content of TEMP1 into ADDRL
    WRITE_RAM( READ_DATAR ),      // write rotated value into register
    // set the FLAG register to new carry value
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
    WRITE_NORIN( READ_MEMORY ),   // load new (still inverted) flag into NOR input
    WRITE_DATAR( READ_DATAR ),    // dummy, relax timing for next operation
    WRITE_DATAR( 0x00 ),          // clear bits 1-7 in DATAR, correct polarity of bit0
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into FLAG register
    // increment register number (rotate high-part now)
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
    WRITE_ADDRL( READ_MEMORY ),   // read content of register and write it into ADDRL for table operation
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup
    WRITE_ADDRL( LOOP( LOOP2 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump

    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment

    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // execute next OP from jump target address
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* ROR reg / Rotate reg right with carry in FLAG register */
  { OPCODE_ROR,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save parameter (register name) to TEMP1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP1
    // load register value to DATAR
    WRITE_ADDRL( READ_DATAR ),    // set operand register for next read
    WRITE_DATAR( READ_MEMORY ),   // load register value into DATAR
    // save register value to TEMP2
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP2. Bit 0 is inverted.
    // clear bit0 from value in DATAR
    WRITE_NORIN( 0x01 ),          // set NOR-input to 1
    WRITE_DATAR( READ_DATAR ),    // set bit0 in DATAR to 0
    // load FLAG register into NORIN
    WRITE_ADDRL( FLAG ),          // select FLAG for next read
    WRITE_NORIN( READ_MEMORY ),   // load FLAG into NORIN
    WRITE_DATAR( READ_DATAR ),    // set bit0 in DATAR value in FLAG (negated)
    WRITE_DATAR_0( READ_DATAR ),  // correct bit0
    // rotate the value in DATAR right
    WRITE_ADDRL( READ_DATAR ),    // set table index
    WRITE_ADDRH( TABLE_ROR ),     // select ROR table
    WRITE_DATAR( READ_MEMORY ),   // perform the table lookup
    // save result in register
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
    WRITE_ADDRL( READ_MEMORY ),   // move content of TEMP1 into ADDRL
    WRITE_RAM( READ_DATAR ),      // write rotated value into register
    // set the FLAG register to new carry value
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
    WRITE_NORIN( READ_MEMORY ),   // load new (still inverted) flag into NOR input
    WRITE_DATAR( READ_DATAR ),    // dummy, relax timing for next operation
    WRITE_DATAR( 0x00 ),          // clear bits 1-7 in DATAR, correct polarity of bit0
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into FLAG register
    // increment PC and execute next OP Code
    MACRO_INC_PC,

    // Space for 44 mikro-instructions of misc. code 

    MACRO_NEXTOP
  },
  /*-------------------------------------------------------------------------*/
  /* AND reg / perform AND operation on Accu and register */  
  { OPCODE_AND,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into TEMP1
    MACRO_LOAD_PC_ADDR,
    WRITE_ADDRL( READ_MEMORY ),   // read register name into DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // read register content into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // write register content into TEMP1
    // initialize loop counter in TEMP2
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_DATAR_CONST( 0x01 ),    // loop init value
    WRITE_RAM( READ_DATAR ),      // write value into TEMP2
    // Loop:
    LOOP1|
      // AND-operation on bit0:
      // 1. invert bit0 of TEMP1, write to TEMP3
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_DATAR( READ_MEMORY ),   // read value from TEMP1 into DATAR (=invert bit0)
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next write
      WRITE_RAM( READ_DATAR ),      // write value into TEMP3
      // 2. read ACCU into DATAR, invert bit0
      WRITE_ADDRL( ACCU ),          // select ACCU for next read
      WRITE_DATAR( READ_MEMORY ),   // read ACCU into DATAR
      // 3. read inverted bit0 of reg into NORIN
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      WRITE_NORIN( READ_MEMORY ),   // write bit0 into NORIN
      // 4. perform: accu = (accu & 0xFE) or (not ((not accuBit0) or (not regBit0)))
      WRITE_DATAR( READ_DATAR ),    // perform final NOR operation
      // rotate accu (in DATAR) right
      WRITE_ADDRL( READ_DATAR ),    // write ACCU into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR_0( READ_MEMORY ), // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( ACCU ),          // select ACCU for next write
      WRITE_RAM( READ_DATAR ),      // write ACCU back to RAM
      // rotate TEMP1 right
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP1 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP1 back to RAM
      // rotate TEMP2 right
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP2 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP2 back to RAM
    // end loop
    WRITE_NORIN( READ_DATAR ),
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    WRITE_ADDRL( 99*2 ),
    JUMP_ALWAYS,

    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
//UFLAG_PRINTINDEX|
    // this address (99) is called from OPCODE_OUT
    MACRO_INC_PC,
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* OR reg / perform OR operation on Accu and register */  
  { OPCODE_OR,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into TEMP1
    MACRO_LOAD_PC_ADDR,
    WRITE_ADDRL( READ_MEMORY ),   // read register name into DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // read register content into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // write register content into TEMP1
    // initialize loop counter in TEMP2
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_DATAR_CONST( 0x01 ),    // loop init value
    WRITE_RAM( READ_DATAR ),      // write value into TEMP2
    // Loop:
    LOOP1|
      // OR-operation on bit0:
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_NORIN( READ_MEMORY ),   // read TEMP1 bit0 into NORIN
      WRITE_ADDRL( ACCU ),          // select ACCU for next read
      WRITE_DATAR( READ_MEMORY ),   // read ACCU into DATAR
      WRITE_DATAR_0( READ_DATAR ),  // correct bit0
      // rotate accu (in DATAR) right
      WRITE_ADDRL( READ_DATAR ),    // write ACCU into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR_0( READ_MEMORY ), // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( ACCU ),          // select ACCU for next write
      WRITE_RAM( READ_DATAR ),      // write ACCU back to RAM
      // rotate TEMP1 right
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP1 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP1 back to RAM
      // rotate TEMP2 right
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP2 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP2 back to RAM
    // end loop
    WRITE_NORIN( READ_DATAR ),
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    MACRO_INC_PC, MACRO_NEXTOP
  },
  /*-------------------------------------------------------------------------*/
  /* XOR reg / perform XOR operation on Accu and register */  
  { OPCODE_XOR,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into TEMP1
    MACRO_LOAD_PC_ADDR,
    WRITE_ADDRL( READ_MEMORY ),   // read register name into DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // read register content into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // write register content into TEMP1
    // initialize loop counter in TEMP2
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_DATAR_CONST( 0x01 ),    // loop init value
    WRITE_RAM( READ_DATAR ),      // write value into TEMP2
    // Loop:
    LOOP1|
      // XOR-operation on bit0:
      // 1. TEMP3.0 = (not ACCU) nor (not REG) = ACCU and REG
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_DATAR( READ_MEMORY ),   // read value from TEMP1 into DATAR (=invert bit0)
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next write
      WRITE_RAM( READ_DATAR ),      // write value into TEMP3
      WRITE_ADDRL( ACCU ),          // select ACCU for next read
      WRITE_DATAR( READ_MEMORY ),   // read ACCU into DATAR
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      WRITE_NORIN( READ_MEMORY ),   // write bit0 into NORIN
      WRITE_DATAR( READ_DATAR ),    // perform final NOR operation
      WRITE_RAM( READ_DATAR ),      // write result into TEMP3
      // 2. TEMP4.0 = ACCU nor REG
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_NORIN( READ_MEMORY ),   // write bit0 into NORIN
      WRITE_ADDRL( ACCU ),          // select ACCU for next read
      WRITE_DATAR( READ_MEMORY ),   // perform final NOR operation
      // 3. ACCU = TEMP3 nor TEMP4.0  (TEMP3 bits 1-7 contain also ACCU bits 1-7)
      WRITE_NORIN( READ_DATAR ),    // write TEMP4.0 into NORIN
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      WRITE_DATAR( READ_MEMORY ),   // perform DATAR = TEMP3 nor TEMP4
      // rotate accu (in DATAR) right
      WRITE_ADDRL( READ_DATAR ),    // write ACCU into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR_0( READ_MEMORY ), // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( ACCU ),          // select ACCU for next write
      WRITE_RAM( READ_DATAR ),      // write ACCU back to RAM
      // rotate TEMP1 right
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP1 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP1 back to RAM
      // rotate TEMP2 right
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP2 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP2 back to RAM
    // end loop
    WRITE_NORIN( READ_DATAR ),
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    MACRO_INC_PC, MACRO_NEXTOP
  },
  /*-------------------------------------------------------------------------*/
  /* ADD reg / Add ACCU and register (8-bit binary addition w/ carry) */
  { OPCODE_ADD,
    LOOP2|
    MACRO_INC_PC,                 // increment PC
    MACRO_LOAD_PC_ADDR,           // load PC from RAM into ADDRL and ADDRH
    // load content of OP-code parameter register into TEMP1
    WRITE_ADDRL( READ_MEMORY ),   // read register name into DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // read register content into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // write register content into TEMP1
    // initialize loop counter in TEMP2
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_DATAR_CONST( 0x01 ),    // loop init value
    WRITE_RAM( READ_DATAR ),      // write value into TEMP2
    MACRO_INC_PC,                 // increment PC
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    // jump to next OP-Code space
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // end of OP-code, wrap-around into next OP-code
    
    // SAP code for RAM address range 0xA000-0xA3FF    
    LOOP1|  // entry-point for decoder of address-range 0xA000-0xA7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 124*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0xA000 - 0xA3FF, else 0xA400-0xA7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xA000 - 0xA1FF, else 0xA200-0xA3FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xA100-0xA1FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xA1 ),          // set ADDR_H to 0xA1
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 119*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xA000-0xA0FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xA0 ),          // set ADDR_H to 0xA0
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 119*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xA200 - 0A93FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xA300-0xA3FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xA3 ),          // set ADDR_H to 0xA3
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 119*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xA200-0xA2FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xA2 ),          // set ADDR_H to 0xA2
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 119*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_OPCODE( OPCODE_SAP2 ),  // continue code execution at OP-code SAP2,120 : execute next op-code
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_OPCODE( OPCODE_SUB ),   // continue code execution at OP-code SUB,125 : range 0xA400-0xA7FFF
    WRITE_ADDRL( LOOP( LOOP2 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_OPCODE( OPCODE_ADD2 )   // continue code execution at OP-code OPCODE_ADD2
  },
  /*-------------------------------------------------------------------------*/
  /* SUB reg / Substract register value from ACCU (8-bit with carry) */
  { OPCODE_SUB,
    LOOP2|
    MACRO_INC_PC,                 // increment PC
    MACRO_LOAD_PC_ADDR,           // load value where PC points to into TEMP1
    // write content of OP-code parameter reg into ADDRL
    WRITE_ADDRL( READ_MEMORY ),   // read register name into DATAR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( READ_MEMORY ),   // read register content into DATAR
    // perform table-lookup and get the 1's complement
    WRITE_ADDRH( TABLE_NOT ),     // select table for 1's complement
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // write 2's complement of reg value into TEMP1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // write register content into TEMP1
    // initialize loop counter in TEMP2
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_DATAR_CONST( 0x01 ),    // loop init value
    WRITE_RAM( READ_DATAR ),      // write value into TEMP2
    MACRO_INC_PC,                 // increment PC
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    // jump to next OP-Code space
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // end of OP-code, wrap-around into next OP-code

    // SAP code for RAM address range 0xA400-0xA7FF    
    LOOP1|  // entry-point for decoder of address-range 0xA400-0xA7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xA400 - 0xA5FF, else 0xA600-0xA7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xA500-0xA5FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xA5 ),          // set ADDR_H to 0xA5
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 119*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xA400-0xA4FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xA4 ),          // set ADDR_H to 0xA4
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 119*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xA600 - 0xA7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xA700-0xA7FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xA7 ),          // set ADDR_H to 0xA7
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 119*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xA600-0xA6FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xA6 ),          // set ADDR_H to 0xA6
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 119*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_OPCODE( OPCODE_SAP2 ),  // continue code execution at OP-code SAP2,120 : execute next op-code
    WRITE_NOP,                    // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP2 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_OPCODE( OPCODE_ADD2 )   // continue code execution at OP-code OPCODE_ADD2
  },
  /*-------------------------------------------------------------------------*/
  /* Second part of ADD / SUB. Uses a LUT for doing 1-bit addition in a loop */ 
  { OPCODE_ADD2,
    // Loop:   work with ACCU, TEMP1(reg), FLAG
    LOOP1|
      // combine ACCU.0, TEMP1.0 and FLAG.0 to a 3-bit word
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_NORIN( READ_MEMORY ),   // read bit 0 from TEMP1 into NORIN
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate left
      WRITE_DATAR( 0x00 ),          // clear bit 1-7 in DATAR, keep bit0 from NORIN, invert bit0
      WRITE_ADDRL( READ_DATAR ),    // write temp.value into ADDRL for table-lookup
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      // DATAR now:  bit7 = not TEMP1.0, bit0-6 = 0
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( ACCU ),          // select ACCU for next read
      WRITE_NORIN( READ_MEMORY ),   // read bit 0 from ACCU into NORIN
      WRITE_DATAR( READ_DATAR ),    // move bit 0 into temp.value
      WRITE_ADDRL( READ_DATAR ),    // write temp.value into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate left
      WRITE_DATAR_0( READ_MEMORY ),   // perform rotation
      // DATAR now:  bit7 = not ACCU.0, bit6 = not TEMP1.0, bit0-5 = 0
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( FLAG ),          // select FLAG for next read
      WRITE_NORIN( READ_MEMORY ),   // read bit 0 from FLAG into NORIN
      WRITE_DATAR( READ_DATAR ),    // move bit 0 into temp.value
      // DATAR now:  bit7 = not ACCU.0, bit6 = not TEMP1.0, bit1-5 = 0, bit0 = not FLAG.0
      WRITE_ADDRL( READ_DATAR ),    // write temp.value into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_TST ),     // select table for full-adder operation (stored in TST table)
      WRITE_DATAR_0( READ_MEMORY ), // perform table lookup
      // DATAR now:  bit7 = carry, bit1 = not sum
      // save DATAR to TEMP3
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      WRITE_RAM( READ_DATAR ),      // write result into TEMP3
      // save carry flag
      WRITE_ADDRL( READ_DATAR ),    // write temp.value into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROL ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation, invert bit0 (FLAG)
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( FLAG ),          // select FLAG for next write
      WRITE_RAM( READ_DATAR ),      // write result to FLAG
      // read ACCU into DATAR, clear bit 0
      WRITE_NORIN( 0x01 ),          // set NORIN to 1
      WRITE_ADDRL( ACCU ),          // select ACCU for next read
      WRITE_DATAR( READ_MEMORY ),   // read ACCU into DATAR, clear bit 0
      // rotate TEMP3 right, write SUM bit to ACCU.0
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP3 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_NORIN( READ_MEMORY ),   // read bit 0 from TEMP3 into NORIN (inverted)
      WRITE_DATAR( READ_DATAR ),    // set bit 0 in DATAR to TEMP3.0
      // rotate DATAR right
      WRITE_ADDRL( READ_DATAR ),    // write temp.value into ADDRL for table-lookup
      WRITE_DATAR_0( READ_MEMORY ), // perform rotation
      // save DATAR to ACCU
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( ACCU ),          // select ACCU for next write
      WRITE_RAM( READ_DATAR ),      // write result into ACCU
      // rotate TEMP1 right
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP1 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP1 back to RAM
      // rotate TEMP2 right
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP2 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP2 back to RAM
    // end loop
    WRITE_NORIN( READ_DATAR ),
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    MACRO_NEXTOP
  },
  /*-------------------------------------------------------------------------*/
  /* CMPI # / Compare ACCU with immediate value */
  { OPCODE_CMPI,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into TEMP1
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // write register content into TEMP1
    // copy ACCU into TEMP4
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR( READ_MEMORY ),   // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next write
    WRITE_RAM( READ_DATAR ),      // write ACCU into TEMP4
    // initialize loop counter in TEMP2
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_DATAR_CONST( 0x01 ),    // loop init value
    WRITE_RAM( READ_DATAR ),      // write value into TEMP2
    // Loop:
    LOOP1|
      // XOR-operation on bit0:
      // 1. TEMP3.0 = (not TEMP4) nor (not REG) = TEMP4 and REG
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_DATAR( READ_MEMORY ),   // read value from TEMP1 into DATAR (=invert bit0)
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next write
      WRITE_RAM( READ_DATAR ),      // write value into TEMP3
      WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next read
      WRITE_DATAR( READ_MEMORY ),   // read TEMP4 into DATAR
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      WRITE_NORIN( READ_MEMORY ),   // write bit0 into NORIN
      WRITE_DATAR( READ_DATAR ),    // perform final NOR operation
      WRITE_RAM( READ_DATAR ),      // write result into TEMP3
      // 2. DATAR.0 = TEMP4 nor REG
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_NORIN( READ_MEMORY ),   // write bit0 into NORIN
      WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next read
      WRITE_DATAR( READ_MEMORY ),   // perform final NOR operation
      // 3. TEMP4 = TEMP3 nor DATAR.0  (TEMP3 bits 1-7 contain also TEMP4 bits 1-7)
      WRITE_NORIN( READ_DATAR ),    // write DATAR.0 into NORIN
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      WRITE_DATAR( READ_MEMORY ),   // perform DATAR = TEMP3 nor DATAR
      // rotate TEMP4 (in DATAR) right
      WRITE_ADDRL( READ_DATAR ),    // write TEMP4 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR_0( READ_MEMORY ), // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP4 back to RAM
      // rotate TEMP1 right
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP1 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP1 back to RAM
      // rotate TEMP2 right
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP2 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP2 back to RAM
    // end loop
    WRITE_NORIN( READ_DATAR ),
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    // check if TEMP4 is zero, set flag
    WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next read
    WRITE_ADDRL( READ_MEMORY ),   // load TEMP4 into ADDRL for table-lookup
    WRITE_ADDRH( TABLE_TST ),     // select table for testing for zero
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // write result into FLAG
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into to RAM
    MACRO_INC_PC, MACRO_NEXTOP
  },
  /*-------------------------------------------------------------------------*/
  /* CMP reg / Compare ACCU with register value */
  { OPCODE_CMP,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into TEMP1
    MACRO_LOAD_PC_ADDR,
    WRITE_ADDRL( READ_MEMORY ),   // read OP-Code parameter into ADDRL
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // content of reg into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // write register content into TEMP1
    // copy ACCU into TEMP4
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR( READ_MEMORY ),   // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next write
    WRITE_RAM( READ_DATAR ),      // write ACCU into TEMP4
    // initialize loop counter in TEMP2
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_DATAR_CONST( 0x01 ),    // loop init value
    WRITE_RAM( READ_DATAR ),      // write value into TEMP2
    // Loop:
    LOOP1|
      // XOR-operation on bit0:
      // 1. TEMP3.0 = (not TEMP4) nor (not REG) = TEMP4 and REG
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_DATAR( READ_MEMORY ),   // read value from TEMP1 into DATAR (=invert bit0)
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next write
      WRITE_RAM( READ_DATAR ),      // write value into TEMP3
      WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next read
      WRITE_DATAR( READ_MEMORY ),   // read TEMP4 into DATAR
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      WRITE_NORIN( READ_MEMORY ),   // write bit0 into NORIN
      WRITE_DATAR( READ_DATAR ),    // perform final NOR operation
      WRITE_RAM( READ_DATAR ),      // write result into TEMP3
      // 2. DATAR.0 = TEMP4 nor REG
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_NORIN( READ_MEMORY ),   // write bit0 into NORIN
      WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next read
      WRITE_DATAR( READ_MEMORY ),   // perform final NOR operation
      // 3. TEMP4 = TEMP3 nor DATAR.0  (TEMP3 bits 1-7 contain also TEMP4 bits 1-7)
      WRITE_NORIN( READ_DATAR ),    // write DATAR.0 into NORIN
      WRITE_ADDRL( TEMP3 ),         // select TEMP3 for next read
      WRITE_DATAR( READ_MEMORY ),   // perform DATAR = TEMP3 nor DATAR
      // rotate TEMP4 (in DATAR) right
      WRITE_ADDRL( READ_DATAR ),    // write TEMP4 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR_0( READ_MEMORY ), // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP4 back to RAM
      // rotate TEMP1 right
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP1 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP1 back to RAM
      // rotate TEMP2 right
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
      WRITE_ADDRL( READ_MEMORY ),   // load TEMP2 into ADDRL for table-lookup
      WRITE_ADDRH( TABLE_ROR ),     // select table for rotate right
      WRITE_DATAR( READ_MEMORY ),   // perform rotation
      WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
      WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
      WRITE_RAM( READ_DATAR ),      // write TEMP2 back to RAM
    // end loop
    WRITE_NORIN( READ_DATAR ),
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    // check if TEMP4 is zero, set flag
    WRITE_ADDRL( TEMP4 ),         // select TEMP4 for next read
    WRITE_ADDRL( READ_MEMORY ),   // load TEMP4 into ADDRL for table-lookup
    WRITE_ADDRH( TABLE_TST ),     // select table for testing for zero
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // write result into FLAG
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into to RAM
    MACRO_INC_PC, MACRO_NEXTOP
  },
  /*-------------------------------------------------------------------------*/
  /* TST reg / Test if register is zero. Set flag when yes. */  
  { OPCODE_TST,
    MACRO_INC_PC,
    MACRO_LOAD_PC_ADDR,
    WRITE_ADDRL( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_ADDRH( TABLE_TST ),     // select test-zero-table for lookup
    WRITE_DATAR( READ_MEMORY ),   // write result from table into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write content of DATAR into FLAG register
    // increment PC and execute next OP Code
    MACRO_INC_PC,
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump and execute next OP-Code

    // OPCODE_JLP :
    LOOP1|
    // perform decrement on content of register
    WRITE_ADDRL( R0 ),            // write register address of R0 into ADDRL
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page by default
    WRITE_ADDRL( READ_MEMORY ),   // read content of register and write it into ADDRL for table operation
    WRITE_ADDRH( TABLE_DEC ),     // select table for decrement
    WRITE_DATAR( READ_MEMORY ),   // perform table-lookup
    // write decremented value back into the source register
    WRITE_ADDRL( R0 ),            // write register address of R0 into ADDRL
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page by default
    WRITE_RAM( READ_DATAR ),      // write incremented value from DATAR back into RAM
    // test R0 for zero
    WRITE_ADDRL( READ_DATAR ),    // write value of R0 into ADDRL
    WRITE_ADDRH( TABLE_TST ),     // select test-zero-table for lookup
    WRITE_DATAR( READ_MEMORY ),   // write result from table into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write content of DATAR into FLAG register
    WRITE_ADDRL( 117*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)

    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment

    WRITE_OPCODE( OPCODE_JNF ),   // continue code execution at OP-code JNF,118
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
//UFLAG_PRINTINDEX|
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* JPF abs / Jump only when flag is set */  
  { OPCODE_JPF,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save to TEMP1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP1
    // increment PC
    MACRO_INC_PC,
    // load FLAG register
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG for next read
    WRITE_NORIN( READ_MEMORY ),   // write FLAG into NORIN
    // conditional jump
    WRITE_ADDRL( 78*2 ),          // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save to PC_H
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_H ),          // select PC_H for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_H
    // copy TEMP1 into PC_L
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_L ),          // select PC_L for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_L
    // execute next OP from jump target address
    WRITE_ADDRL( 120*2 ),         //
    JUMP_ALWAYS,
    MACRO_INC_PC, MACRO_NEXTOP
  },
  /*-------------------------------------------------------------------------*/
  /* JNF abs / Jump only when flag is not set */  
  { OPCODE_JNF,
    // increment PC
    LOOP1|
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save to TEMP1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP1
    // increment PC
    MACRO_INC_PC,
    // load FLAG register
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG for next read
    WRITE_DATAR( READ_MEMORY ),   // load FLAG into DATAR
    WRITE_NORIN( READ_DATAR ),    // write inverted FLAG into NORIN
    // conditional jump
    WRITE_ADDRL( 79*2 ),          // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save to PC_H
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_H ),          // select PC_H for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_H
    // copy TEMP1 into PC_L
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_L ),          // select PC_L for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_L
    // execute next OP from jump target address
    WRITE_ADDRL( 120*2 ),         //
    JUMP_ALWAYS,
    MACRO_INC_PC,
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump and execute next OP-Code
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* JMP abs / Jump to absolut address */  
  { OPCODE_JMP,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save to TEMP1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP1
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save to PC_H
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_H ),          // select PC_H for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_H
    // copy TEMP1 into PC_L
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_L ),          // select PC_L for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_L
    // increment PC and execute next OP Code
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump and execute next OP-Code

    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP,                    // dummy for alignment

    // OPCODE_RWL part 1 :
    LOOP1|
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    // MACRO_LOAD_PC_ADDR part 1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page by default
    WRITE_ADDRL( PC_L ),          // select PC_L register for next read
    WRITE_DATAR_0( READ_MEMORY ), // read memory and write PCL to DATA-register
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_H ),          // select PC_H register for next read
    WRITE_ADDRH( READ_MEMORY ),   // set ADDRH to PC_H

    // continue in OPCODE_ROL
    WRITE_OPCODE( OPCODE_ROL ),   // continue code execution at OP-code ROL,118
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // execute next OP from jump target address
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* JSR abs / Call subroutine */  
  { OPCODE_JSR,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save to TEMP1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP1
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // save to TEMP2
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into TEMP2
    // increment PC
    MACRO_INC_PC,
    // copy PC to LR
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( PC_L ),          // select PC_L for next read
    WRITE_DATAR( READ_MEMORY ),   // read PC_L into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( LR_L ),          // select LR_L for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into LR_L
    WRITE_ADDRL( PC_H ),          // select PC_H for next read
    WRITE_DATAR( READ_MEMORY ),   // read PC_H into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( LR_H ),          // select LR_H for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into LR_H
    // copy TEMP1/TEMP2 into PC
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read
    WRITE_DATAR( READ_MEMORY ),   // read TEMP1 into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_L ),          // select PC_L for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_L
    WRITE_ADDRL( TEMP2 ),         // select TEMP2 for next read
    WRITE_DATAR( READ_MEMORY ),   // read TEMP2 into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_H ),          // select PC_H for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_H
    // execute next OP from jump target address
    MACRO_NEXTOP
  },
  /*-------------------------------------------------------------------------*/
  /* RET / Return from subroutine */  
  { OPCODE_RET,
    // copy LR to PC
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( LR_L ),          // select LR_L for next read
    WRITE_DATAR( READ_MEMORY ),   // read LR_L into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_L ),          // select PC_L for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_L
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( LR_H ),          // select LR_H for next read
    WRITE_DATAR( READ_MEMORY ),   // read LR_L into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PC_H ),          // select PC_H for next write
    WRITE_RAM( READ_DATAR ),      // store DATAR into PC_H
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // execute next OP from return address

    // SAP code for RAM address range 0xA800-0xABFF    
    LOOP1|  // entry-point for decoder of address-range 0xA800-0xABFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 117*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0xA800 - 0xABFF, else 0xAC00-0xAFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xA800 - 0xA9FF, else 0xAA00-0xABFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xA900-0xA9FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xA9 ),          // set ADDR_H to 0xA9
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xA800-0xA8FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xA8 ),          // set ADDR_H to 0xA8
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xAA00 - 0xABFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xAB00-0xABFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xAB ),          // set ADDR_H to 0xAB
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xAA00-0xAAFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xAA ),          // set ADDR_H to 0xAA
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP,                    // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_ADDRL( 0 ),             // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_OPCODE( OPCODE_INC ),   // continue code execution at OP-code INC,118  : range 0xAC00-0xAFFF
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* PSH reg / Push register onto stack */
  { OPCODE_PSH,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to and load the value from this register into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_ADDRL( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // read register value into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // load stack pointer into ADDRL
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read stack-pointer into ADDRL
    WRITE_ADDRH( STACKPAGE_HI ),  // select stack-page
    // write DATAR into the stack
    WRITE_RAM( READ_DATAR ),      // write DATAR into RAM
    // increment the stack-pointer
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read SP value into ADDRL
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next write
    WRITE_RAM( READ_DATAR ),      // write back the incremented stack-pointer

    // increment PC and execute next OP Code
    LOOP2|  
    MACRO_INC_PC,
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump and execute next OP-Code

    // microcode of the instruction PHL (push link register to stack)
    LOOP1|
    // load LR_L into DATAR
    WRITE_ADDRL( LR_L ),          // load address of LR_L in RAM
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // read register value into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // load stack pointer into ADDRL
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read stack-pointer into ADDRL
    WRITE_ADDRH( STACKPAGE_HI ),  // select stack-page
    // write DATAR into the stack
    WRITE_RAM( READ_DATAR ),      // write DATAR into RAM
    // increment the stack-pointer
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read SP value into ADDRL
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next write
    WRITE_RAM( READ_DATAR ),      // write back the incremented stack-pointer
    // load LR_H into DATAR
    WRITE_ADDRL( LR_H ),          // load address of LR_H in RAM
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_DATAR( READ_MEMORY ),   // read register value into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    // load stack pointer into ADDRL
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read stack-pointer into ADDRL
    WRITE_ADDRH( STACKPAGE_HI ),  // select stack-page
    // write DATAR into the stack
    WRITE_RAM( READ_DATAR ),      // write DATAR into RAM
    // increment the stack-pointer
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read SP value into ADDRL
    WRITE_ADDRH( TABLE_INC ),     // select table for increment
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next write
    WRITE_RAM( READ_DATAR ),      // write back the incremented stack-pointer
    WRITE_ADDRL( LOOP( LOOP2 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP,                    // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* POP reg / Pull register back from stack */  
  { OPCODE_POP,
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into TEMP1
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read register name into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // TEMP1 register for next write
    WRITE_RAM( READ_DATAR ),      // write register name into TEMP1
    // decrement the stack-pointer
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read SP value into ADDRL
    WRITE_ADDRH( TABLE_DEC ),     // select table for decrement
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next write
    WRITE_RAM( READ_DATAR ),      // write back the incremented stack-pointer
    // load stack pointer into ADDRL
    WRITE_ADDRL( READ_DATAR ),    // read stack-pointer into ADDRL
    WRITE_ADDRH( STACKPAGE_HI ),  // select stack-page
    // read the register back from stack and write it into the target register
    WRITE_DATAR( READ_MEMORY ),   // load value from stack
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( TEMP1 ),         // TEMP1 register for next read
    WRITE_ADDRL( READ_MEMORY ),   // load register name for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into target register
    // increment PC and execute next OP Code
    MACRO_INC_PC,
    WRITE_ADDRL( 120*2 ),         // set new address offset
    JUMP_ALWAYS,                  // perform the jump and execute next OP-Code

    // microcode of the instruction RTS (pop link register from stack and return from subroutine)
    LOOP1|
    // decrement the stack-pointer
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read SP value into ADDRL
    WRITE_ADDRH( TABLE_DEC ),     // select table for decrement
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next write
    WRITE_RAM( READ_DATAR ),      // write back the incremented stack-pointer
    // load stack pointer into ADDRL
    WRITE_ADDRL( READ_DATAR ),    // read stack-pointer into ADDRL
    WRITE_ADDRH( STACKPAGE_HI ),  // select stack-page
    // read the LR_H register back from stack
    WRITE_DATAR( READ_MEMORY ),   // load value from stack
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( LR_H ),          // set LR_H register for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into target register
    // decrement the stack-pointer
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next read
    WRITE_ADDRL( READ_MEMORY ),   // read SP value into ADDRL
    WRITE_ADDRH( TABLE_DEC ),     // select table for decrement
    WRITE_DATAR( READ_MEMORY ),   // perform the table-lookup
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( SP ),            // set stack-pointer register for next write
    WRITE_RAM( READ_DATAR ),      // write back the incremented stack-pointer
    // load stack pointer into ADDRL
    WRITE_ADDRL( READ_DATAR ),    // read stack-pointer into ADDRL
    WRITE_ADDRH( STACKPAGE_HI ),  // select stack-page
    // read the LR_L register back from stack
    WRITE_DATAR( READ_MEMORY ),   // load value from stack
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( LR_L ),          // set LR_L register for next write
    WRITE_RAM( READ_DATAR ),      // write DATAR into target register
    WRITE_ADDRL( 114*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_OPCODE( OPCODE_RET ),   // continue code execution at OP-code RET,115 
    WRITE_NOP,                    // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* PHL / Push link registers LR_L and LR_H to the stack */
  { OPCODE_PHL,
    WRITE_ADDRL( 117*2 ),         // set new address offset
    JUMP_ALWAYS,                  // jump to code of PHL instruction

    // SAP code for RAM address range 0xB000-0xB7FF    
    LOOP1|  // entry-point for decoder of address-range 0xB000-0xB7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(53) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0xB000 - 0xB3FF, else 0xB400-0xB700
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xB000 - 0xB1FF, else 0xB200-0xB3FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xB100-0xB1FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xB1 ),          // set ADDR_H to 0xB1
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xB000-0xB0FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xB0 ),          // set ADDR_H to 0xB0
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xB200 - 0xB3FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xB300-0xB3FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xB3 ),          // set ADDR_H to 0xB3
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xB200-0xB2FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xB2 ),          // set ADDR_H to 0xB2
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xB400 - 0xB7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xB400 - 0xB5FF, else 0xB600-0xB7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xB500-0xB5FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xB5 ),          // set ADDR_H to 0xB5
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xB400-0xB4FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xB4 ),          // set ADDR_H to 0xB4
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xB600 - 0xB7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xB700-0xB7FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xB7 ),          // set ADDR_H to 0xB7
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xB600-0xB6FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xB6 ),          // set ADDR_H to 0xB6
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP,                    // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP,                    // dummy for alignment
// UFLAG_PRINTINDEX|
    WRITE_OPCODE( OPCODE_PSH ),   // continue code execution at OP-code PSH,118 
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* RTS / Pop link register from stack and return */
  { OPCODE_RTS,
    WRITE_ADDRL( 117*2 ),         // set new address offset
    JUMP_ALWAYS,                  // jump to code of RTS instruction

    // SAP code for RAM address range 0xB800-0xBFFF    
    LOOP1|  // entry-point for decoder of address-range 0xB800-0xBFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(53) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0xB800 - 0xBBFF, else 0xBC00-0xBFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xB800 - 0xB9FF, else 0xBA00-0xBBFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xB900-0xB9FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xB9 ),          // set ADDR_H to 0xB9
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xB800-0xB8FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xB8 ),          // set ADDR_H to 0xB8
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xBA00 - 0xBBFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xBB00-0xBBFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xBB ),          // set ADDR_H to 0xBB
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xBA00-0xBAFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xBA ),          // set ADDR_H to 0xBA
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xBC00 - 0xBFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xBC00 - 0xBDFF, else 0xBE00-0xBFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xBD00-0xBDFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xBD ),          // set ADDR_H to 0xBD
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xBC00-0xBCFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xBC ),          // set ADDR_H to 0xBC
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xBE00 - 0xBFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xBF00-0xBFFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xBF ),          // set ADDR_H to 0xBF
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xBE00-0xBEFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xBE ),          // set ADDR_H to 0xBE
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP,                    // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP,                    // dummy for alignment
    WRITE_OPCODE( OPCODE_POP ),   // continue code execution at OP-code POP,118 
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* JLP / decrement R0 and loop while R0 is not zero */
  { OPCODE_JLP,
    WRITE_ADDRL( 117*2 ),         // set new address offset
    JUMP_ALWAYS,                  // jump to code of PHL instruction

    // SAP code for RAM address range 0xC000-0xC7FF    
    LOOP1|  // entry-point for decoder of address-range 0xC000-0xC7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(53) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0xC000 - 0xC3FF, else 0xC400-0xC700
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xC000 - 0xC1FF, else 0xC200-0xC3FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xC100-0xC1FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xC1 ),          // set ADDR_H to 0xB1
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xC000-0xC0FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xC0 ),          // set ADDR_H to 0xC0
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xC200 - 0xC3FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xC300-0xC3FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xC3 ),          // set ADDR_H to 0xB3
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xC200-0xC2FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xC2 ),          // set ADDR_H to 0xB2
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xC400 - 0xC7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xC400 - 0xC5FF, else 0xC600-0xC7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xC500-0xC5FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xC5 ),          // set ADDR_H to 0xC5
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xC400-0xC4FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xC4 ),          // set ADDR_H to 0xC4
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xC600 - 0xC7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xC700-0xC7FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xC7 ),          // set ADDR_H to 0xC7
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xC600-0xC6FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xC6 ),          // set ADDR_H to 0xC6
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_OPCODE( OPCODE_TST ),   // continue code execution at OP-code TST,118 
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* RWL / rotate 16-bit word left */
  { OPCODE_RWL,
    WRITE_ADDRL( 117*2 ),         // set new address offset
    JUMP_ALWAYS,                  // jump to code of RTS instruction

    // SAP code for RAM address range 0xC800-0xCFFF    
    LOOP1|  // entry-point for decoder of address-range 0xC800-0xCFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(53) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0xC800 - 0xCBFF, else 0xCC00-0xCFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xC800 - 0xC9FF, else 0xCA00-0xCBFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xC900-0xC9FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xC9 ),          // set ADDR_H to 0xC9
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xC800-0xC8FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xC8 ),          // set ADDR_H to 0xC8
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xCA00 - 0xCBFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xCB00-0xCBFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xCB ),          // set ADDR_H to 0xCB
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xCA00-0xCAFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xCA ),          // set ADDR_H to 0xCA
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xCC00 - 0xCFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xCC00 - 0xCDFF, else 0xCE00-0xCFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xCD00-0xCDFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xCD ),          // set ADDR_H to 0xCD
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xCC00-0xCCFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xCC ),          // set ADDR_H to 0xCC
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xCE00 - 0xCFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xCF00-0xCFFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xCF ),          // set ADDR_H to 0xCF
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xCE00-0xCEFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xCE ),          // set ADDR_H to 0xCE
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 120*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP,                    // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP,                    // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_OPCODE( OPCODE_JMP ),   // continue code execution at OP-code JMP,118
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    // execute next OP Code
    MACRO_NEXTOP_INLINE
  },
  /*-------------------------------------------------------------------------*/
  /* SEC / Set carry flag */
  { OPCODE_SEC,
    WRITE_DATAR_CONST( 0x01 ),    // set DATAR to 1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write to FLAG register
    WRITE_ADDRL( 117*2 ),         // set new address offset
    JUMP_ALWAYS,                  // jump to code INC_PC / NEXTOP

    // SAP code for RAM address range 0xD000-0xD7FF
    LOOP1|  // entry-point for decoder of address-range 0xD000-0xD7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(53) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0xD000 - 0xD3FF, else 0xD400-0xD700
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xD000 - 0xD1FF, else 0xD200-0xD3FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xD100-0xD1FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xD1 ),          // set ADDR_H to 0xD1
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xD000-0xD0FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xD0 ),          // set ADDR_H to 0xD0
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xD200 - 0xD3FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xD300-0xD3FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xD3 ),          // set ADDR_H to 0xD3
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xD200-0xD2FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xD2 ),          // set ADDR_H to 0xD2
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xD400 - 0xD7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xD400 - 0xD5FF, else 0xD600-0xD7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xD500-0xD5FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xD5 ),          // set ADDR_H to 0xD5
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xD400-0xD4FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xD4 ),          // set ADDR_H to 0xD4
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xD600 - 0xD7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xD700-0xD7FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xD7 ),          // set ADDR_H to 0xD7
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xD600-0xD6FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xD6 ),          // set ADDR_H to 0xD6
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_OPCODE( OPCODE_LAP ),   // continue code execution at OP-code LAP,118 -> INC_PC / NEXTOP
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP,                    // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    // execute next OP Code
    WRITE_OPCODE( OPCODE_SAP2 ),  // continue code execution in SAP2 -> execute next OP
  },
  /*-------------------------------------------------------------------------*/
  /* CLC / Clear carry flag */
  { OPCODE_CLC,
    WRITE_DATAR_CONST( 0x00 ),    // set DATAR to 0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write to FLAG register
    WRITE_ADDRL( 117*2 ),         // set new address offset
    JUMP_ALWAYS,                  // jump to code INC_PC / NEXTOP

    // SAP code for RAM address range 0xD800-0xDFFF    
    LOOP1|  // entry-point for decoder of address-range 0xC800-0xCFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(53) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0xD800 - 0xDBFF, else 0xDC00-0xDFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xD800 - 0xD9FF, else 0xDA00-0xDBFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xD900-0xD9FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xD9 ),          // set ADDR_H to 0xD9
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xD800-0xD8FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xD8 ),          // set ADDR_H to 0xD8
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xDA00 - 0xDBFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xDB00-0xDBFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xDB ),          // set ADDR_H to 0xDB
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xDA00-0xDAFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xDA ),          // set ADDR_H to 0xDA
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xDC00 - 0xDFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xDC00 - 0xDDFF, else 0xDE00-0xDFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xDD00-0xDDFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xDD ),          // set ADDR_H to 0xDD
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xDC00-0xDCFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xDC ),          // set ADDR_H to 0xDC
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xDE00 - 0xDFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xDF00-0xDFFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xDF ),          // set ADDR_H to 0xDF
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xDE00-0xDEFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xDE ),          // set ADDR_H to 0xDE
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_OPCODE( OPCODE_LAP ),   // continue code execution at OP-code LAP,118 -> INC_PC / NEXTOP
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP,                    // dummy for alignment
    // execute next OP Code
    WRITE_OPCODE( OPCODE_SAP2 ),  // continue code execution in SAP2 -> execute next OP
  },
  /*-------------------------------------------------------------------------*/
  /* AD / Add with carry flag set to 0 */
  { OPCODE_AD,
    WRITE_DATAR_CONST( 0x00 ),    // set DATAR to 0
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write to FLAG register
    WRITE_ADDRL( 124*2 ),         // set new address offset
    JUMP_ALWAYS,                  // jump to opcode ADD

    // SAP code for RAM address range 0xE000-0xE7FF
    LOOP1|  // entry-point for decoder of address-range 0xE000-0xE7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(53) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0xE000 - 0xE3FF, else 0xE400-0xE700
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xE000 - 0xE1FF, else 0xE200-0xE3FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xE100-0xE1FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xE1 ),          // set ADDR_H to 0xE1
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xE000-0xE0FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xE0 ),          // set ADDR_H to 0xE0
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xE200 - 0xE3FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xE300-0xE3FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xE3 ),          // set ADDR_H to 0xE3
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xE200-0xE2FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xE2 ),          // set ADDR_H to 0xE2
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xE400 - 0xE7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xE400 - 0xE5FF, else 0xE600-0xE7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xE500-0xE5FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xE5 ),          // set ADDR_H to 0xE5
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xE400-0xE4FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xE4 ),          // set ADDR_H to 0xE4
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xE600 - 0xE7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xE700-0xE7FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xE7 ),          // set ADDR_H to 0xE7
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xE600-0xE6FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xE6 ),          // set ADDR_H to 0xE6
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP,                    // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_OPCODE( OPCODE_ADD ),   // continue code execution in OP-code ADD,125
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    // execute next OP Code
    WRITE_OPCODE( OPCODE_SAP2 ),  // continue code execution in SAP2 -> execute next OP
  },
  /*-------------------------------------------------------------------------*/
  /* SU / Subtract with carry flag set to 1 */
  { OPCODE_SU,
    WRITE_DATAR_CONST( 0x01 ),    // set DATAR to 1
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( FLAG ),          // select FLAG register for next write
    WRITE_RAM( READ_DATAR ),      // write to FLAG register
    WRITE_ADDRL( 122*2 ),         // set new address offset
    JUMP_ALWAYS,                  // jump to opcode SUB

    // SAP code for RAM address range 0xE800-0xEFFF    
    LOOP1|  // entry-point for decoder of address-range 0xE800-0xEFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(53) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0xE800 - 0xEBFF, else 0xEC00-0xEFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xE800 - 0xE9FF, else 0xEA00-0xEBFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xE900-0xE9FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xE9 ),          // set ADDR_H to 0xE9
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xE800-0xE8FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xE8 ),          // set ADDR_H to 0xE8
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xEA00 - 0xEBFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xEB00-0xEBFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xEB ),          // set ADDR_H to 0xEB
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xEA00-0xEAFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xEA ),          // set ADDR_H to 0xEA
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xEC00 - 0xEFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xEC00 - 0xEDFF, else 0xEE00-0xEFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xED00-0xEDFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xED ),          // set ADDR_H to 0xED
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xEC00-0xECFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xEC ),          // set ADDR_H to 0xEC
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xEE00 - 0xEFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xEF00-0xEFFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xEF ),          // set ADDR_H to 0xEF
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xEE00-0xEEFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xEE ),          // set ADDR_H to 0xEE
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP,                    // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_OPCODE( OPCODE_SUB ),   // continue code execution in OP-code SUB,123
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    // execute next OP Code
    WRITE_OPCODE( OPCODE_SAP2 ),  // continue code execution in SAP2 -> execute next OP
  },
  /*-------------------------------------------------------------------------*/
  /* OUT / Ouput ACCU on output port */
  { OPCODE_OUT,
    WRITE_ADDRL( 116*2 ),         // set new address offset
    JUMP_ALWAYS,                  // jump to opcode OUT

    LOOP1|  // entry-point for decoder of address-range 0xF000-0xFFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 3) ^ 0x81  (bit0 = PTR_H.5 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( 124*2 ),         // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.5 == 1), handle address range 0xF000-0xFFFF
    // here: ADDR 0xF000 - 0xF7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(53) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0xF000 - 0xF3FF, else 0xF400-0xF700
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xF000 - 0xF1FF, else 0xF200-0xF3FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xF100-0xF1FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xF1 ),          // set ADDR_H to 0xF1
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xF000-0xF0FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xF0 ),          // set ADDR_H to 0xF0
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xF200 - 0xF3FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xF300-0xF3FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xF3 ),          // set ADDR_H to 0xF3
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xF200-0xF2FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xF2 ),          // set ADDR_H to 0xF2
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xF400 - 0xF7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xF400 - 0xF5FF, else 0xF600-0xF7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xF500-0xF5FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xF5 ),          // set ADDR_H to 0xF5
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xF400-0xF4FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xF4 ),          // set ADDR_H to 0xF4
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xF600 - 0xF7FF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xF700-0xF7FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xF7 ),          // set ADDR_H to 0xF7
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xF600-0xF6FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xF6 ),          // set ADDR_H to 0xF6
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_OPCODE( OPCODE_DEC ),   // continue code execution in OP-code DEC,117 -> code of OPCODE_OUT
    WRITE_OPCODE( OPCODE_LAP ),   // continue code execution in OP-code LAP,118 -> INC_PC / NEXTOP
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_OPCODE( OPCODE_IN   ),  // continue code execution in IN,125
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    // execute next OP Code
    WRITE_OPCODE( OPCODE_SAP2 ),  // continue code execution in SAP2 -> execute next OP
  },
  /*-------------------------------------------------------------------------*/
  /* IN / read input port into ACCU */
  { OPCODE_IN,
#ifdef ARCH_MYNOR
    WRITE_ADDRH( 0xC0 ),          // set port-address of input port
    WRITE_DATAR_0( READ_MEMORY ), // read port value into DATAR
#else  // My4TH
    WRITE_ADDRH( 0xC0 ),          // set port-address of input port
    WRITE_DATAR_0( READ_INPORT ), // read port value into DATAR
#endif
    WRITE_DATAR( READ_DATAR ),    // correct bit0   
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page by default
    WRITE_ADDRL( ACCU ),          // select ACCU for next write
    WRITE_RAM  ( READ_DATAR ),    // write value into ACCU
    WRITE_ADDRL( 117*2 ),         // set new address offset
    JUMP_ALWAYS,                  // continue code execution in OP-code LAP,118 -> INC_PC / NEXTOP

    // SAP code for RAM address range 0xF800-0xFFFF    
    LOOP1|  // entry-point for decoder of address-range 0xF800-0xFFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 6) ^ 0xF1  (bit0 = PTR_H.2 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(53) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.2 == 1)
    // here: ADDR 0xF800 - 0xFBFF, else 0xFC00-0xFFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xF800 - 0xF9FF, else 0xFA00-0xFBFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xF900-0xF9FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xF9 ),          // set ADDR_H to 0xF9
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xF800-0xF8FF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xF8 ),          // set ADDR_H to 0xF8
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xFA00 - 0xFBFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xFB00-0xFBFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xFB ),          // set ADDR_H to 0xFB
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xFA00-0xFAFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xFA ),          // set ADDR_H to 0xFA
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xFC00 - 0xFFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_DATAR_0( READ_MEMORY ), // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xF9  (bit0 = PTR_H.1 ^ 1)
    WRITE_NORIN( READ_DATAR ),    // use bit0 for jump decision
    WRITE_ADDRL( JUMPOFS(25) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.1 == 1)
    // here: ADDR 0xFC00 - 0xFDFF, else 0xFE00-0xFFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xFD00-0xFDFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xFD ),          // set ADDR_H to 0xFD
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xFC00-0xFCFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xFC ),          // set ADDR_H to 0xFC
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // here: ADDR 0xFE00 - 0xFFFF
    WRITE_ADDRL( READ_DATAR ),    // rotate left
    WRITE_NORIN( READ_MEMORY ),   // perform table-lookup, DATAR = (PTR_H rol 7) ^ 0xFD  (bit0 = PTR_H.0)
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page
    WRITE_ADDRL( JUMPOFS(11) ),   // write jump-target-address in ADDRL
    JUMP_IF_ZERO,                 // perform the jump if (PTR_H.0 == 0)
    // write Accu to RAM 0xFF00-0xFFFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xFF ),          // set ADDR_H to 0xFF
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // write Accu to RAM 0xFE00-0xFEFF
    WRITE_ADDRL( ACCU ),          // select ACCU for next read
    WRITE_DATAR_0( READ_MEMORY ), // read ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0
    WRITE_ADDRL( PTR_L ),         // select PTR_L for next read
    WRITE_ADDRL( READ_MEMORY ),   // read PTR_L into ADDR_L
    WRITE_ADDRH( 0xFE ),          // set ADDR_H to 0xFE
    WRITE_RAM( READ_DATAR ),      // write ACCU into RAM pointed by PTR_L
    WRITE_ADDRL( 127*2 ),         // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_OPCODE( OPCODE_LAP ),   // continue code execution in OP-code LAP,118 -> INC_PC / NEXTOP
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP, WRITE_NOP,         // dummy for alignment
    WRITE_NOP,                    // dummy for alignment
    WRITE_ADDRL( LOOP( LOOP1 ) ), // write jump-target-address in ADDRL
    JUMP_ALWAYS,                  // perform the jump
    // execute next OP Code
    WRITE_OPCODE( OPCODE_SAP2 ),  // continue code execution in SAP2 -> execute next OP
  },
  /*-------------------------------------------------------------------------*/
#ifdef ARCH_MYNOR
#ifdef OPCODE_OUT2
  /* OUT2 / Write IO-Ports */
  { OPCODE_OUT2,
    //   OUT2 0xA0 : write ACCU to Output-Port 1
    //   OUT2 0xC0 : write ACCU to Output-Port 2
    //   OUT2 0xE0 : write ACCU to Output-Port 3
    // increment PC
    MACRO_INC_PC,
    // load value where PC points to into DATAR
    MACRO_LOAD_PC_ADDR,
    WRITE_DATAR( READ_MEMORY ),   // read OP-Code parameter from memory where PC points to
    WRITE_DATAR( READ_DATAR ),    // correct bit0   
    //DATAR is now: 0xA0=OUT-1, 0xC0=OUT-2, 0xE0=OUT-3
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page by default
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next write
    WRITE_RAM( READ_DATAR ),      // DATAR into the TEMP1
    // IO OUTPUT
    WRITE_ADDRH( ZEROPAGE_HI ),   // select zero-page by default
    WRITE_ADDRL( ACCU ),          // select ACCU for next read 
    WRITE_DATAR( READ_MEMORY ),   // load ACCU into DATAR
    WRITE_DATAR( READ_DATAR ),    // correct bit0   
    WRITE_ADDRL( TEMP1 ),         // select TEMP1 for next read 
    WRITE_ADDRH( READ_MEMORY ),   // load TEMP1 into ADDRH - select OUTPUT port
    WRITE_RAM  ( READ_DATAR ),    // write value from DATAR to OUTPUT port
    // increment PC and execute next OP Code
    MACRO_INC_PC, MACRO_NEXTOP
  }
#endif
#endif
};


static void compile_nextop_code(uint8_t *buffer)
{
  uint32_t idx, v;
  uint8_t b1, b2;
  
  for (idx = 0; idx < 8; idx++)
  {
    v = nextop_table[idx];
    b1 = (uint8_t)(v >> 8) ^ CTL_RES_NORFF;
    b2 = (uint8_t)v;
    if ((v & UFLAG_REGSOURCE) == 0)
    {
      b1 |= CONST_FLAGS;  // no register or memory access, but load constant value
    }
    if ((v & UFLAG_OFFSET) != 0)
    {
      b2 += (uint8_t)idx;
      b2 *= 2;
    }
    *buffer++ = b1;
    *buffer++ = b2;
  }
}


static void compile_microcode(uint8_t *buffer)
{
  uint8_t lidx[4] = {0,};
  uint32_t j, op, idx, v;
  uint8_t *b, b1, b2;
  uint8_t b1_l;
  
  for (j = 0; j < NUM_OPCODES; j++)
  {
    b1_l = 0;
    op = ucode_table[j][0];
    b = buffer + (op * 256);
    for (idx = 0; idx < 128; idx++)
    {
      v = ucode_table[j][idx+1];
      b1 = (uint8_t)(v >> 8) ^ CTL_RES_NORFF;
      b2 = (uint8_t)v;
      lidx[(v & LABEL_MASK) >> LABEL_SHIFT] = (uint8_t)idx;
      if (v & UFLAG_PRINTINDEX)
      {
        printf("OP 0x%02X Index = %d\n", op, idx);
      }
      if (v & UFLAG_NEXTOP)
      {
        if (idx >= 120)
        {
          printf("Error in OP-Code %02X: No space left for NEXTOP code\n", op);
        }
//        else
//        {
//          printf("OP 0x%02x: %d u-ops free\n", op, 119-idx);
//        }
        compile_nextop_code(buffer + (op * 256) + (2*120));
        break;
      }
      if ((v & UFLAG_REGSOURCE) == 0)
      {
        b1 |= CONST_FLAGS;  // no register or memory access, but load constant value
        if ((b1 & CTL_CLK_MASK) == CTL_CLK_RAM)
        {
          printf("Error in OP-Code %02X, line %d: Can not write a constant value directly into RAM\n", op, idx+1);
        }
        if ((b1 & CTL_CLK_MASK) == CTL_CLK_DATAR)
        {
          if (b1_l & CTL_ASEL_MEMORY)
          {
            printf("Error in OP-Code %02X, line %d: No memory operation allowed before writing constant value to DATAR\n", op, idx+1);
          }
        }
      }
      if ((v & UFLAG_OFFSET) != 0)
      {
        b2 += (uint8_t)idx;
        b2 *= 2;
      } else
      if ((v & UFLAG_LABEL) != 0)
      {
        b2 = lidx[b2 & (LABEL_MASK>>LABEL_SHIFT)] * 2;
      }
      
      *b++ = b1;
      *b++ = b2;
      b1_l = b1;
    }
  }
}


static void generate_rol_table(uint8_t *buffer)
{
  uint8_t *b = &buffer[TABLE_ROL*256];
  unsigned i, a;
  
  for (i=0; i<256; i++)
  {
    a = (uint8_t)i;
    b[i] = (a<<1) | (a>>7);
  }
}

static void generate_ror_table(uint8_t *buffer)
{
  uint8_t *b = &buffer[TABLE_ROR*256];
  unsigned i, a;
  
  for (i=0; i<256; i++)
  {
    a = (uint8_t)i;
    b[i] = ((a<<7) | (a>>1)) ^ 1;
  }
}

static void generate_inc_table(uint8_t *buffer)
{
  uint8_t *b = &buffer[TABLE_INC*256];
  unsigned i;
  
  for (i=0; i<256; i++)
  {
    b[i] = (uint8_t)(i + 1)^1;
  }
}

static void generate_dec_table(uint8_t *buffer)
{
  uint8_t *b = &buffer[TABLE_DEC*256];
  unsigned i;
  
  for (i=0; i<256; i++)
  {
    b[i] = (uint8_t)(i - 1)^1;
  }
}

static void generate_not_table(uint8_t *buffer)
{
  uint8_t *b = &buffer[TABLE_NOT*256];
  unsigned i;
  
  for (i=0; i<256; i++)
  {
    b[i] = (uint8_t)(i^0xFF);
  }
}

static void generate_testzero_table(uint8_t *buffer)
{
  uint8_t v, *b = &buffer[TABLE_TST*256];
  unsigned i, b1, b2, b3, c, s, a;
  
  for (i=0; i<256; i++)
  {
    b[i] = (i == 0) ? 0x01 : 0x00;
  }

  // implement also a 1-bit full-adder into this table
  for (i=0; i<8; i++)
  {
    b1 = (i>>0) & 1;
    b2 = (i>>1) & 1;
    b3 = (i>>2) & 1;

    s = b1 + b2 + b3;
    c = (s > 1) ? 1 : 0;
    s&= 1;

    c^=1;  // invert carry (inversion needed by microcode)

    v = (uint8_t) ((c << 7) | (s << 1));

    a = (b1 << 7) | (b2 << 6) | b3;
    a^= 0xC1;  // invert address bits (inversion needed by microcode)
    b[a] |= v;
  }
}

static void add_infotext(uint8_t *buffer, const char *text)
{
  const int limit = 256*(LAST_OPCODE+1) - 1;
  int l = strlen(text);
  int i, j;
  
  for (i=0; i<limit; i+=2)
  {
    if (((buffer[i] & 0x7F) == 0x7F) && (buffer[i+1] == 0xFF))
    {
      for (j=i; j<limit; j+=2)
      {
        if (((buffer[j] & 0x7F) != 0x7F) || (buffer[j+1] != 0xFF))
          break;
      }
      j -= i;
      if (j > l)
      {
        memcpy(buffer+i, text, l);
        break;
      }
    }
  }
}


/** Add a visual effect to the microcode.
 *  A LED can be connected to the unused bit7 of the control-word-latch.
 *  The LED is lit when MyNOR executes instructions that use the
 *  NOR gate a lot.
 */
static void add_LED_effect(uint8_t *buffer)
{
#ifdef ARCH_MYNOR
  static uint8_t led100_instructions[] = {
    OPCODE_AND, OPCODE_OR, OPCODE_XOR, OPCODE_ADD, OPCODE_SUB, OPCODE_ADD2,
    OPCODE_CMPI, OPCODE_CMP,
  0 };

  static uint8_t led50_instructions[] = {
    OPCODE_ROR, OPCODE_ROL,
  0 };

  static uint8_t led25_instructions[] = {
    OPCODE_TST, OPCODE_IN, OPCODE_OUT, // only because LED shall glow a bit when MyNOR is idle
  0 };

  uint8_t *op;
  unsigned i, addr;

  for (i = 0; i < 256*(LAST_OPCODE+1); i+= 2)
  {
    buffer[i] |= 0x80;    
  }
  
  for (op = led100_instructions; *op != 0; op++)
  {
    addr = ((unsigned)*op) << 8;
    for (i = 0; i < 256; i+= 2)
    {
      buffer[addr+i] ^= 0x80;
    }
  }

  for (op = led50_instructions; *op != 0; op++)
  {
    addr = ((unsigned)*op) << 8;
    for (i = 0; i < 256; i+= 8)
    {
      buffer[addr+i] ^= 0x80;
    }
  }

  for (op = led25_instructions; *op != 0; op++)
  {
    addr = ((unsigned)*op) << 8;
    for (i = 0; i < 256; i+= 16)
    {
      buffer[addr+i] ^= 0x80;
    }
  }
#endif
}


/** Generate microcode.
 *  @param buffer   ptr to output buffer with at least 16kb space
 */
#ifdef ARCH_MYNOR
void generate_mynor_microcode_eprom(uint8_t *buffer)
#else
void generate_my4th_microcode_eprom(uint8_t *buffer)
#endif
{
  compile_microcode(buffer);

  generate_inc_table(buffer);
  generate_dec_table(buffer);
  generate_rol_table(buffer);
  generate_ror_table(buffer);
  generate_not_table(buffer);
  generate_testzero_table(buffer);
  add_LED_effect(buffer);

  add_infotext(buffer, "***** My4TH by Dennis Kuschel / Germany / www.mynor.org *****");
  add_infotext(buffer, "  This project is open source, license CC BY-SA 4.0  ");
}
