#ifndef _MY4TH_MICROCODE_H
#define _MY4TH_MICROCODE_H
#include <stdint.h>


/* Control outputs from EPROM */
#define CTL_SRC_MEMORY    (0<<0)
#define CTL_SRC_DATAR     (1<<0)
#define CTL_CLK_MASK      (7<<1)
#define CTL_CLK_OPCODE    (0<<1)
#define CTL_CLK_OUT       (1<<1)  // Only My4TH, not available on MyNOR
#define CTL_CLK_NORIN     (2<<1)
#define CTL_CLK_DATAR     (3<<1)
#define CTL_CLK_ADDRL     (4<<1)
#define CTL_CLK_ADDRH     (5<<1)
#define CTL_CLK_RAM       (6<<1)
#define CTL_CLK_NONE      (7<<1)
#define CTL_ASEL_UCODE    (0<<4)
#define CTL_ASEL_MEMORY   (1<<4)
#define CTL_ENA_LDCTR     (1<<5)
#define CTL_RES_NORFF     (1<<6)
#define CTL_SEL_INPORT    (1<<7)

/* Compiler internal flags */
#define UFLAG_REGSOURCE   (1<<16)
#define UFLAG_OFFSET      (1<<17)
#define UFLAG_LABEL       (1<<18)
#define UFLAG_NEXTOP      (1<<19)
#define UFLAG_PRINTINDEX  (1<<20)
#define CONST_FLAGS       (CTL_SRC_MEMORY | CTL_ASEL_UCODE)

/* Jump label defines for loops */
#define LABEL_SHIFT       24
#define LABEL_MASK        (3<<LABEL_SHIFT)
#define LOOP1             (1<<LABEL_SHIFT)
#define LOOP2             (2<<LABEL_SHIFT)
#define LOOP3             (3<<LABEL_SHIFT)

/* Data-Sources used with WRITE_xxx - macros */
#define READ_DATAR        (UFLAG_REGSOURCE | ((CTL_SRC_DATAR  | CTL_ASEL_UCODE                  ) << 8))
#define READ_MEMORY       (UFLAG_REGSOURCE | ((CTL_SRC_MEMORY | CTL_ASEL_MEMORY                 ) << 8))
#define READ_INPORT       (UFLAG_REGSOURCE | ((CTL_SRC_MEMORY | CTL_ASEL_MEMORY | CTL_SEL_INPORT) << 8))

/* Data-Sinks. Example: WRITE_RAM(READ_DATAR) or WRITE_ADDRH(0x04) */
#define WRITE_OPCODE(src)       ((CTL_CLK_OPCODE << 8) | (src))
#define WRITE_OUT(src)          ((CTL_CLK_OUT    << 8) | (src))
#define WRITE_NORIN(src)        ((CTL_CLK_NORIN  << 8) | (src))
#define WRITE_DATAR(src)        ((CTL_CLK_DATAR  << 8) | (src))
#define WRITE_DATAR_0(src)      (((CTL_CLK_DATAR|CTL_RES_NORFF) << 8) | (src))
#define WRITE_DATAR_CONST(src)  (((CTL_CLK_DATAR|CTL_RES_NORFF) << 8) | ((src)^1))
#define WRITE_ADDRL(src)        ((CTL_CLK_ADDRL  << 8) | (src))
#define WRITE_ADDRH(src)        ((CTL_CLK_ADDRH  << 8) | (src))
#define WRITE_RAM(src)          (((CTL_CLK_RAM|CTL_ASEL_MEMORY) << 8) | (src))
#define WRITE_NOP               (CTL_CLK_NONE << 8) 

/* Jump instruction. Load program counter when bit0 of data source is set. */
#define JUMP_IF_ZERO            (((CTL_ENA_LDCTR|CTL_ASEL_MEMORY|CTL_CLK_NONE) << 8) | (READ_DATAR))
#define JUMP_ALWAYS             (((CTL_ENA_LDCTR|CTL_ASEL_MEMORY|CTL_CLK_NONE|CTL_RES_NORFF) << 8) | (READ_DATAR))

/* offset for jump target calculation */
#define JUMPOFS(offset)         (UFLAG_OFFSET | ((uint8_t)(offset)))
#define LOOP(label)             (UFLAG_LABEL  | ((uint8_t)((label)>>LABEL_SHIFT)))

/* Table indexes */
#define TABLE_INC     0x27    // increment input value by one
#define TABLE_DEC     0x28    // decrement input value by one
#define TABLE_TST     0x29    // test for zero. set bit 0 if input is zero.
#define TABLE_ROL     0x2A    // rotate left
#define TABLE_ROR     0x2B    // rotate left
#define TABLE_NOT     0x2C    // invert bits (used for SUB OP-Code)

/* OP-Code binary index values */
#define OPCODE_RST    0x00
#define OPCODE_LDI    0x01
#define OPCODE_LD     0x02
#define OPCODE_LDAI   0x03
#define OPCODE_LDA    0x04
#define OPCODE_LAP    0x05
#define OPCODE_STA    0x06
#define OPCODE_SAP    0x07
#define OPCODE_SAP2   0x08
#define OPCODE_INC    0x09
#define OPCODE_DEC    0x0A
#define OPCODE_ROL    0x0B
#define OPCODE_ROR    0x0C
#define OPCODE_AND    0x0D
#define OPCODE_OR     0x0E
#define OPCODE_XOR    0x0F
#define OPCODE_ADD    0x10
#define OPCODE_SUB    0x11
#define OPCODE_ADD2   0x12
#define OPCODE_CMPI   0x13
#define OPCODE_CMP    0x14
#define OPCODE_TST    0x15
#define OPCODE_JPF    0x16
#define OPCODE_JNF    0x17
#define OPCODE_JMP    0x18
#define OPCODE_JSR    0x19
#define OPCODE_RET    0x1A
#define OPCODE_PSH    0x1B
#define OPCODE_POP    0x1C
#define OPCODE_IN     0x1D
#define OPCODE_PHL    0x1E
#define OPCODE_RTS    0x1F
#define OPCODE_JLP    0x20
#define OPCODE_RWL    0x21
#define OPCODE_SEC    0x22
#define OPCODE_CLC    0x23
#define OPCODE_AD     0x24
#define OPCODE_SU     0x25
#define OPCODE_OUT    0x26

#define NUM_OPCODES   39
#define LAST_OPCODE   OPCODE_OUT
#define CODE_START    0x2D00  // at this address starts the assembly code

/* Zero-page registers */
#define ZEROPAGE_HI   0x80  // first page in SRAM, thus it is named "page zero"
#define STACKPAGE_HI  0x81  // memory page for the stack
#define TEMP1   0x00
#define TEMP2   0x01
#define TEMP3   0x02
#define TEMP4   0x03
#define TEMP5   0x04
#define PAR1    0x05  // free to use by user program
#define PAR2    0x06  // free to use by user program
#define SP      0x07  // Stack Pointer
#define PC_L    0x08  // Program Counter, low
#define PC_H    0x09  // Program Counter, high
#define LR_L    0x0A  // Link Register, low
#define LR_H    0x0B  // Link Register, high
#define PTR_L   0x0C  // Memory Pointer Register, low
#define PTR_H   0x0D  // Memory Pointer Register, high
#define ACCU    0x0E  // Accumulator
#define FLAG    0x0F  // Flag Register for zero- and carry-flag
#define R0      0x10



/** Generate microcode.
 *  @param buffer   ptr to output buffer with at least 16kb space
 */
void generate_my4th_microcode_eprom(uint8_t *buffer);


#endif
