
#include "opsystem.h"


/****************************************************************************
 *                                                                          *
 *  MS Windows (WIN32) Implementation                                       *
 *                                                                          *
 ****************************************************************************/

#ifdef _WIN32

#include <sys/stat.h>
#include <sys/types.h>

#include <windows.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

struct _sOSSEMAHANDLE
{
  HANDLE sema;
};

struct _sOSTASKHANDLE
{
  HANDLE task;
  DWORD  id;
};


OSSEM_t OsSemaCreate(int count)
{
  OSSEM_t sem;

  sem = (OSSEM_t)calloc(1, sizeof(struct _sOSSEMAHANDLE));
  if (sem == NULL)
    return NULL;

  sem->sema = CreateSemaphore(NULL,count,count+1000,NULL);
  if (sem->sema == NULL)
  {
    free(sem);
    return NULL;
  }
  
  return sem;
} 


int OsSemaDelete(OSSEM_t hsem)
{
  if (hsem == NULL)
    return -1;

  CloseHandle(hsem->sema);
  free(hsem);
  return 0;
}


int OsSemaSignal(OSSEM_t hsem)
{
  if (hsem == NULL)
    return -1;

  return (ReleaseSemaphore(hsem->sema, 1, NULL) != 0) ? 0 : -1;
}


int OsSemaWait(OSSEM_t hsem)
{
  if (hsem == NULL)
    return -1;

  return (WaitForSingleObject(hsem->sema, INFINITE) == WAIT_FAILED) ? -1 : 0;
}


int OsSemaWaitTimeout(OSSEM_t hsem, int milliseconds)
{
  DWORD ret;

  if (hsem == NULL)
    return -1;

  ret = WaitForSingleObject(hsem->sema, (DWORD)milliseconds);
  if (ret == WAIT_TIMEOUT)
    return 1;  // timeout
  
  return (ret == WAIT_OBJECT_0) ? 0 : -1;   // ok(0) or error(-1)
}


OSTASK_t OsTaskCreate(OSTASKFUNC_t pFunc, void* pArgs)
{
  OSTASK_t htask;

  htask = (OSTASK_t)calloc(1, sizeof(struct _sOSTASKHANDLE));
  if (htask == NULL)
    return NULL;

  htask->task = CreateThread(NULL, 0,
                             (LPTHREAD_START_ROUTINE)pFunc,
                             pArgs, 0, &htask->id);

  if (htask->task == NULL)
  {
    free(htask);
    return NULL;
  }

  return htask;
}


void OsTaskExit(int exitcode)
{
  ExitThread((DWORD)exitcode);
}


#endif /* _WIN32 */





/****************************************************************************
 *                                                                          *
 *  LINUX Implementation                                                    *
 *                                                                          *
 ****************************************************************************/

#ifdef _LINUX

#include <sys/time.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <stdarg.h>
#include <pthread.h>
#include <semaphore.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/stat.h>


struct _sOSSEMAHANDLE
{
  sem_t  sem;
};

struct _sOSTASKHANDLE
{
  pthread_t       hthread;
  OSTASKFUNC_t    func;
  void            *arg;
  sem_t           sem;
};


OSSEM_t OsSemaCreate(int count)
{
  OSSEM_t sem;
  
  sem = (OSSEM_t)calloc(1, sizeof(struct _sOSSEMAHANDLE));
  if (sem == NULL)
    return NULL;
  
  if (sem_init(&sem->sem, 0, (unsigned int)count) != 0)
  {
    free(sem);
    return NULL;
  }

  return sem;
} 


int OsSemaDelete(OSSEM_t hsem)
{
  if (hsem == NULL)
    return -1;
  
  sem_destroy(&hsem->sem);
  free(hsem);
  return 0;
}


int OsSemaSignal(OSSEM_t hsem)
{
  if (hsem == NULL)
    return -1;

  sem_post(&hsem->sem);
  return 0;
}


int OsSemaWait(OSSEM_t hsem)
{
  if (hsem == NULL)
    return -1;

  sem_wait(&hsem->sem);
  return 0;
}


int OsSemaWaitTimeout(OSSEM_t hsem, int milliseconds)
{
  struct timespec ts;
  int ret;

  if (hsem == NULL)
    return -1;

  clock_gettime(CLOCK_REALTIME, &ts);
  
  ts.tv_sec += milliseconds / 1000;
  milliseconds %= 1000;
  ts.tv_nsec += milliseconds * 1000000;
  if (ts.tv_nsec >= 1000000000)
  {
    ts.tv_nsec -= 1000000000;
    ts.tv_sec++;
  }

  ret = sem_timedwait(&hsem->sem, &ts);
  if ((ret == -1) && (errno == ETIMEDOUT))
    return 1; // timeout
  
  if (ret != 0)
    return -1;  // any other error

  return 0;  // ok
}


static void* pthread_new_task(void *arg)
{
  OSTASK_t htask = (OSTASK_t)arg;
  long ret;
  sem_post(&htask->sem);
  ret = (long) (htask->func)(htask->arg);
  return (void*) ret;
}
  

OSTASK_t OsTaskCreate(OSTASKFUNC_t pFunc, void* pArgs)
{
  OSTASK_t htask;
  
  htask = (OSTASK_t)calloc(1, sizeof(struct _sOSTASKHANDLE));
  if (htask == NULL)
    return NULL;

  htask->func = pFunc;
  htask->arg = pArgs;
  
  if (sem_init(&htask->sem, 0, 0) != 0)
  {
    free(htask);
    return NULL;
  }

  if (pthread_create(&htask->hthread, NULL, pthread_new_task, (void*)htask) != 0)
  {
    free(htask);
    return NULL;
  }

  sem_wait(&htask->sem);
  sem_destroy(&htask->sem);
  return htask;
}


void OsTaskExit(int exitcode)
{
  long rc = (long)exitcode;
  pthread_exit((void*)rc);
}


#endif /* _LINUX */
