#include "timing.h"


#ifdef _LINUX
/*------------- LINUX --------------*/

#include <time.h>
#include <unistd.h>


/* Get 32-bit microseconds-counter
 */
uint32_t get_usec(void)
{
  static uint32_t cur_usec = 0;
  static struct timespec last_ts = {0,0};
  struct timespec ts;

  clock_gettime(CLOCK_REALTIME, &ts);
  if (ts.tv_sec == last_ts.tv_sec) {
    cur_usec += (uint32_t)((ts.tv_nsec - last_ts.tv_nsec) / 1000);
  } else {
    if (ts.tv_sec == last_ts.tv_sec + 1) {
      cur_usec += (uint32_t) ((1000000000 - last_ts.tv_nsec + ts.tv_nsec) / 1000);
    } else {
      cur_usec++;
    }
  }
  last_ts = ts;
  return cur_usec;
}


static void sleep_usec(uint32_t usec)
{
  usleep(usec);
}


void sleep_msec(uint32_t msec)
{
  while (msec > 100)
  {
    usleep(100000);
    msec -= 100;
  }
  
  usleep(msec * 1000);
}


#else
/*------------- WINDOWS ------------*/

#include <windows.h>
#include <stdio.h>


/*
 * NOTE:
 * Timing on Windows is horrible!
 * The My4TH simulation can not be 100% timing exact on a Windows system.
 * If you want correct timing, install and use Linux :-)
 *
 */


/* Get 32-bit microseconds-counter */
uint32_t get_usec(void)
{
  static int inited = 0;
  static LARGE_INTEGER  ticks_per_100usec;
  static LARGE_INTEGER  lastPerfCounter;
  static uint32_t cur_usec = 0;
  LARGE_INTEGER pc, perfDiff, us100;

  if (!inited)
  {
    inited = 1;
    if (!QueryPerformanceFrequency(&pc))
    {
      fprintf(stderr, "ERROR: Your system does not support the Performance Counter!\n");
      exit(1);
    }
    ticks_per_100usec.QuadPart = pc.QuadPart / 10000;
    QueryPerformanceCounter(&lastPerfCounter);
  }
 
  QueryPerformanceCounter(&pc);
  perfDiff.QuadPart = pc.QuadPart - lastPerfCounter.QuadPart;
  
  if (perfDiff.QuadPart >= ticks_per_100usec.QuadPart)
  {
    us100.QuadPart = perfDiff.QuadPart / ticks_per_100usec.QuadPart;
    lastPerfCounter.QuadPart += us100.QuadPart * ticks_per_100usec.QuadPart;
    cur_usec += (uint32_t) (us100.QuadPart * 100);
  }

  return cur_usec;
}


static void sleep_usec(uint32_t usec)
{
  static uint32_t sleepsum = 0;

  sleepsum += usec;
  if (sleepsum >= 1000)
  {
    Sleep(1);
    sleepsum = 0;
  }
  else
  {
    Sleep(0);
  }
}


void sleep_msec(uint32_t msec)
{
  Sleep(msec);
}

#endif


int isTimeOver(uint32_t *timer, uint32_t exptime, uint32_t now, uint32_t *rem)
{
  uint32_t dif;

/*  
	dif = now - *timer;
	dif-= exptime;
  
  changed to:
*/
  uint64_t a;
  a = 0x200000000ULL;
  a+= now;
  a-= *timer;
  a-= exptime;
  a&= 0x0FFFFFFFFULL;
  dif = (uint32_t)a;

  if (dif & (1L<<((sizeof(uint32_t)*8)-1)))
  {
    *rem = 0 - dif;
    return 0;
  }
  else
  {
    *rem = 0;
    *timer = now;
    return 1;
  }
}


/* Timing Control */
void sleep_until_next_timing_grid(void)
{
  static int inited = 0;
  static uint32_t us_timer = 0;
  uint32_t us_remain;

  if (!inited)
  {
    us_timer = get_usec();
    inited = 1;
  }
  
  while (!isTimeOver(&us_timer, TIMING_GRID/1000, get_usec(), &us_remain))
  {
    sleep_usec(us_remain);
  }
}
