/*
 *  File Transfer between the host and the My4TH computer
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "timing.h"
#include "forthfmt.h"
#include "xfer-uart.h"
#include "transfer.h"
#include "version.h"
#include "input.h"

int default_baudrate_g = 14400;
int fast_baudrate_g = 14400;
int slow_baudrate_g = 9600;

#define SLOW_BAUDRATE   slow_baudrate_g
#define FAST_BAUDRATE   fast_baudrate_g
#define DEFAULT_BAUDRATE default_baudrate_g
#define XFCMD_NOP       0x00
#define XFCMD_LISTEN    0x01
#define XFCMD_EXECUTE   0x02
#define XFCMD_READ1K    0x12
#define XFCMD_WRITE1K   0x13

#define XFREP_NACK      0x04
#define XFREP_ACK       0x05
#define XFREP_COK       0x0E
#define XFREP_CERR      0x0F
#define XFREP_BUSY      0x10
#define XFREP_CONT      0x11
#define XFREP_BINMODE   0x86

#define ESC_00_1F       0x15  // Escape for 0x00 - 0x1F, followed by an ASCII character where only bits 0-4 are used
#define ESC_80_9F       0x16  // Escape for 0x80 - 0x9F, followed by an ASCII character where only bits 0-4 are used
#define ESC_A0_FF       0x17  // Escape for 0xA0 - 0xFF, followed by an ASCII character in the range 0x20 - 0x7F

#define ERR_OK    0
#define ERR_FAIL  -1
#define ERR_CONN  -2

#define RX_TIMEOUT      100  // default short timeout
#define DEFAULT_TIMEOUT  5  // seconds

#ifdef _LINUX
  #define EXAMPLE_DEV  "/dev/ttyS0"
  #define LINEEND "\n"
#else
  #define EXAMPLE_DEV  "com1:"
  #define LINEEND "\r\n"
#endif

static int cfg_fast_baudrate_g = 0;


struct screenmem;
typedef struct screenmem screenmem_t;
struct screenmem {
  screenmem_t *next;
  int used;
  uint8_t data[1024];
};


/** Free list of screens
 */
static void free_screenmem(screenmem_t *sm)
{
  screenmem_t *next;
  
  while (sm != NULL)
  {
    next = sm->next;
    free(sm);
    sm = next;
  }
}


/** Update a CRC-16 value
 *  @param crc    ptr to CRC checksum variable
 *  @param byte   byte to add to the checksum
 */
static void update_crc16(uint16_t *crc, uint8_t byte)
{
  uint32_t c = *crc;
  int i;
  
  c = c ^ (((uint32_t)byte) << 8);

  for (i=0; i<8; i++)
  {
    c = c << 1;
    if (c & 0x10000)
      c = (c ^ 0x1021) & 0xFFFF;
  }
  
  *crc = (uint16_t)c;
}


/** Try to connect to My4TH.
 *  @param loops  number of connect tries
 *  @return zero if connected, ERR_CONN if any error
 */
static int try_connect(int loops)
{
  int i, rx;
  int chng = 0;

  for(i = 0; i < loops; i++)
  {
    if ((i % 20 == 0) && cfg_fast_baudrate_g && !chng)
    {
      xfuart_baudrate(SLOW_BAUDRATE);
//printf("Setting SLOW SPEED\n");        
      sleep_msec(RX_TIMEOUT);
      cfg_fast_baudrate_g = 0;
    }
    chng = 0;
    
    while (xfuart_receive(RX_TIMEOUT) >= 0);

    if (xfuart_send(0x00) != 0)
      return ERR_CONN;
    
    sleep_msec(RX_TIMEOUT);

    if (xfuart_send(XFCMD_LISTEN) != 0)
      return ERR_CONN;

    rx = xfuart_receive(RX_TIMEOUT);
    if ((rx & 0xFF) == 0xF9)
    {
      if (!cfg_fast_baudrate_g)
      {
        cfg_fast_baudrate_g = 1;
        xfuart_baudrate(FAST_BAUDRATE);
//printf("Setting FAST SPEED\n");    
//        sleep_msec(RX_TIMEOUT);
//        xfuart_send(0x00);
//        sleep_msec(RX_TIMEOUT);

        chng = 1;
      }
    }
    else
    if ((rx & 0xFF) == XFREP_BINMODE)
    {
      return ERR_OK;
    }
  }
  
  return ERR_CONN;
}


/** Connect to My4TH, push firmware into binary connected mode.
 *  @return ERR_OK on success, otherwise ERR_CONN.
 */
int xfer_connect(void)
{
  int s;
  
  printf("Connecting... ");
  fflush(stdout);
  
  /* clean the line */
  while (xfuart_receive(RX_TIMEOUT) >= 0);
  if (try_connect(20) != 0)
  {
    printf("\rFailed to connect to My4TH. Check connection, push reset button and try again.\n");
    
    for (s = 10; s > 0; s--)
    {
      printf("\rStill trying %d seconds ", s);
      fflush(stdout);
      while (xfuart_receive(1) >= 0);
      xfuart_send(0x00);
      sleep_msec(100);
      xfuart_send(0x00);
      sleep_msec(100);
      if (try_connect(5) == 0) // 5 * 3*RX_TIMEOUT ms
        break;
      sleep_msec(800 - 3*5*RX_TIMEOUT);
    }
    printf("\rStill trying %d seconds ", s);
    fflush(stdout);
    if ((s == 0) && (try_connect(5) != 0))
    {
      printf("\rFailed to connect to My4TH\n");
      return ERR_CONN;
    }
  }
  printf("\rConnected to My4TH     \n");
  return ERR_OK;
}


/** Execute a command string on My4TH.
 *  @param str      Forth string to execute, max 160 characters.
 *  @param timeout  Time in [s] the command execution is allowed to last.
 *                  Default is 0 wich means 5 seconds.
 *  @return zero on success. ERR_FAIL if the command failed on My4TH.
 *          ERR_CONN is returned when the connection is broken.
 *  @note xfer_connect must be called before this function
 */
int transfer_execute_string(char *str, int timeout)
{
  int rx, i, j, rep = 0, nl = 0;
  uint16_t crc = 0;
  char csum[16];
  
  if (strlen(str) > 160)
  {
    fprintf(stderr, "Line too long! Max. 160 characters allowed!\n");
    return ERR_FAIL;
  }
  
  timeout *= 1000;
  if (timeout <= 0)
    timeout = 5000;
  
  for (i=0; str[i]!=0; i++)
  {
    update_crc16(&crc, (uint8_t)str[i]);
  }
  sprintf(csum, "%04X", crc);

  for (j=0;;j++)
  {
    if (j>4)
      return ERR_CONN;

    for (i=0;;i++)
    {
      if (i>4)
        return ERR_CONN;

//printf("sending XFCMD_EXECUTE\n");
// sleep_msec(1000);
      if (xfuart_send(XFCMD_EXECUTE) != 0)
        return ERR_CONN;
      
//printf("Receiving...\n");
      rx = xfuart_receive(RX_TIMEOUT);
      if (rx == -2)
        return ERR_CONN;
//printf("Got 0x%02X\n", rx);      
      if (rx == XFREP_CONT)
        break;

      if (xfer_connect() != 0)
        return ERR_CONN;
    }
    
    printf("< %s\n", str);
    for (i=0; i<4; i++)
    {
      if (xfuart_send(csum[i]) != 0)
        return ERR_CONN;
    }
    for (i=0; str[i]!=0; i++)
    {
      if (xfuart_send(str[i]) != 0)
        return ERR_CONN;
    }
    if (xfuart_send(0) != 0)
      return ERR_CONN;

    rx = xfuart_receive(300);
    if (rx < 0)
      return ERR_CONN;
    
    if (rx == XFREP_ACK)
      break;
  }
  
  for(;;)
  {
    rx = xfuart_receive(timeout);
    if (rx < 0)
      return ERR_CONN;
    
    if ((rx == XFREP_COK) || (rx == XFREP_CERR))
    {
      if (rep)
        printf("\n");
      break;
    }
    
    if (rx >= 0x20)
    {
      if (nl)
      {
        printf("\n");
        nl = 0;
        rep = 0;
      }
      if (!rep)
      {
        printf("> ");
        rep = 1;
      }
      
      printf("%c", rx);
      fflush(stdout);
    }
    else
    if (rx == 0x0A)
    {
      if (rep)
        nl = 1;
    }
  }
 
  return (rx == XFREP_COK) ? ERR_OK : ERR_FAIL;
}


/** Download a Forth-Screen from EEPROM.
 *  @param buffer   ptr to a buffer that must have a size of at least 1kB.
 *  @param screen   the screen number of the screen to download
 *  @return zero on success. ERR_FAIL if the command failed on My4TH.
 *          ERR_CONN is returned when the connection is broken.
 *  @note xfer_connect must be called before this function
 */
int transfer_download_screen(uint8_t *buffer, int screen)
{
  int i, j, k, p, rx, esc, cs, a;
  uint16_t crc = 0;
  unsigned scrnum;
  char scrcmd[16];
  char hexstr[16];
  
  scrnum = ((unsigned)screen) & 0xFFF;
  scrnum|= ((scrnum << 4) ^ (scrnum << 8) ^ (scrnum << 12)) & 0xF000;
  sprintf(scrcmd, "%04X", scrnum);
  
  for (k=0;;k++)
  {
    for (j=0;;j++)
    {
      if (j>4)
        return ERR_CONN;

      for (i=0;;i++)
      {
        if (i>4)
          return ERR_CONN;

        if (xfuart_send(XFCMD_READ1K) != 0)
          return ERR_CONN;
        
        rx = xfuart_receive(RX_TIMEOUT);
        if (rx == -2)
          return ERR_CONN;
        
        if (rx == XFREP_CONT)
          break;

        if (xfer_connect() != 0)
          return ERR_CONN;
      }
      
      for (i=0; i<5; i++)
      {
        if (xfuart_send(scrcmd[i]) != 0)
          return ERR_CONN;
      }

      rx = xfuart_receive(300);
      if (rx < 0)
        return ERR_CONN;

      if (rx == XFREP_CERR)
        return ERR_FAIL; // the requested screen is unavailable

      if (rx == XFREP_ACK)
        break;
    }

    crc = 0;
    p = 0;
    esc = 0;
    for(;;)
    {
      if (p == 0)
      {
        // My4TH needs some time to read the 1K from EEPROM,
        // so we use a long timeout for the first byte.
        // Also the first byte may be zero when My4TH starts reading the EEPROM.
        do {
          rx = xfuart_receive(4000);
          if (rx == 0) {
            printf("My4TH reads the EEPROM ");
            fflush(stdout);
            rx = xfuart_receive(4000);
          }
        } while (rx == 0);
      }
      else
      {
        rx = xfuart_receive(300);
      }
      if (rx == 0)
      {
        if (p == 1024+4)
        {
          hexstr[4] = 0;
          cs = 0;
          if (sscanf(hexstr, "%X", &cs) == 1)
          {
            if (((uint16_t)cs) == crc)
            {
              printf(" OK\n");
              return ERR_OK;
            }
          }
        }
        break;  // error, wrong checksum
      }
      
      if (p >= 1024+4)
      {
        while (xfuart_receive(RX_TIMEOUT) >= 0);
        break;  // error, too much data received
      }
      
      if (p >= 1024)
      {
        hexstr[p-1024] = (char)rx;
        p++;
      }
      else
      if ((rx == ESC_00_1F) || (rx == ESC_80_9F) || (rx == ESC_A0_FF))
      {
        esc = rx;
      }
      else
      if ((rx < 32) || (rx > 127))
      {
        break;  // error, illegal byte reveived
      }
      else
      {
        if (esc == ESC_00_1F) {
          rx &= 0x1F;
        } else
        if (esc == ESC_80_9F) {
          rx = (rx & 0x1F) | 0x80;
        } else
        if (esc == ESC_A0_FF) {
          rx |= 0x80;
        }
        esc = 0;
        buffer[p++] = rx;
        update_crc16(&crc, (uint8_t)rx);
        a = p * 100;
        printf("\rReceiving screen %d, %d%%", screen, a / 1024);
        fflush(stdout);
      }
    }
    
    if (++k <= 3) {
      printf("\nCommunication error, retrying...\n");
    } else {
      printf("\nCommunication error, giving up.\n");
      break;
    }
  }
  
  return ERR_CONN;
}



/** Upload a Forth-Screen to EEPROM.
 *  @param buffer   ptr to a buffer that must have a size of at least 1kB.
 *  @param screen   the screen number of the screen to upload
 *  @return zero on success. ERR_FAIL if the command failed on My4TH.
 *          ERR_CONN is returned when the connection is broken.
 *  @note xfer_connect must be called before this function
 */
int transfer_upload_screen(uint8_t *buffer, int screen)
{
  int i, j, rx;
  uint16_t crc = 0;
  unsigned scrnum;
  char scrcmd[16];
  uint8_t c;
  
  scrnum = ((unsigned)screen) & 0xFFF;
  scrnum|= ((scrnum << 4) ^ (scrnum << 8) ^ (scrnum << 12)) & 0xF000;
  sprintf(scrcmd, "%04X", scrnum);

  for (i=0; i<1024; i++)
  {
    update_crc16(&crc, buffer[i]);
  }
  sprintf(scrcmd+4, "%04X", crc);


  for (j=0;;j++)
  {
    if (j>4)
    {
      printf("\rScreen %d: FAILED, giving up.             \n", screen);
      return ERR_CONN;
    }
    else
    if (j>0)
    {
      printf("\rScreen %d: FAILED, retrying               \n", screen);
      increase_send_delay();
    }

    for (i=0;;i++)
    {
      if (i>4)
        return ERR_CONN;

      if (xfuart_send(XFCMD_WRITE1K) != 0)
        return ERR_CONN;
      
      rx = xfuart_receive(RX_TIMEOUT);
      if (rx == -2)
        return ERR_CONN;
      
      if (rx == XFREP_CONT)
        break;

      if (xfer_connect() != 0)
        return ERR_CONN;
    }
    
    for (i=0; i<8; i++)
    {
      if (xfuart_send(scrcmd[i]) != 0)
        return ERR_CONN;
    }
    
    printf("Screen %d: (1/3) sending data...", screen);
    fflush(stdout);
    for (i=0; i<1024; i++)
    {
      c = buffer[i];
      if (c < 0x20) {
        if (xfuart_send(ESC_00_1F) != 0)
          return ERR_CONN;
        if (xfuart_send((c & 0x1F) + 0x40) != 0)
          return ERR_CONN;
      } else
      if (c < 0x80) {
        if (xfuart_send(c) != 0)
          return ERR_CONN;
      } else
      if (c < 0xA0) {
        if (xfuart_send(ESC_80_9F) != 0)
          return ERR_CONN;
        if (xfuart_send((c & 0x1F) + 0x40) != 0)
          return ERR_CONN;
      } else {
        if (xfuart_send(ESC_A0_FF) != 0)
          return ERR_CONN;
        if (xfuart_send((c & 0x7F) + 0x80) != 0)
          return ERR_CONN;
      }
    }
    if (xfuart_send(0) != 0)
      return ERR_CONN;

    rx = xfuart_receive(100);
    if (rx == -2)
      return ERR_CONN;

    if (rx != XFREP_BUSY)
    {
      if (xfuart_send(0) != 0)
        return ERR_CONN;

      while (xfuart_receive(3000) >= 0);
      continue;  // timeout error, try again
    }

    printf("\rScreen %d: (2/3) My4TH tests checksum... ", screen);
    fflush(stdout);

    rx = xfuart_receive(3000);
    
    if (rx == -2)
      return ERR_CONN;

    if (rx == XFREP_CERR)
      return ERR_FAIL; // the requested screen is unavailable

    if (rx != XFREP_ACK)
    {
      continue;  // wrong checksum
    }

    printf("\rScreen %d: (3/3) My4TH writes EEPROM... ", screen);
    fflush(stdout);

    do {
      rx = xfuart_receive(3000);
    } while (rx == 0);
    
    if (rx == -2)
      return ERR_CONN;

    if (rx == XFREP_COK)
    {
      printf("\rScreen %d:  OK                           \n", screen);
      return ERR_OK;
    }
  }

  return ERR_FAIL;
}


/*---------------------------------------------------------------------------*/


static void lower_baudrate(void)
{
  fast_baudrate_g = slow_baudrate_g;
  slow_baudrate_g /= 2;
  default_baudrate_g = fast_baudrate_g;
}


static void quit_with_error(char *errormsg)
{
  fprintf(stderr, "%s\n", errormsg);
  exit(1);
}


static char* get_next_string(int argc, char *argv[], int *i)
{
  if (*i >= argc)
    quit_with_error("ERROR: Missing argument\n");
  (*i)++;
  return argv[*i-1];
}


static void open_serial_device(char *devname)
{
#ifdef _LINUX
  char buf[500], *p;

  strncpy(buf, devname, 499);
  p = buf;
  while (*p && (*p != ','))
    p++;

  if (*p == ',')
  {
    *p++ = 0;
    if (xfuart_init_pipe(buf, p) != 0)
      quit_with_error("Pipe error");
  }
  else
#endif
  if (xfuart_init_rs232(devname, DEFAULT_BAUDRATE) != 0)
    quit_with_error("RS232 device error");
}



/** The terminal subprogram
 */
static int tf_term(int argc, char *argv[], int i)
{
  int rx, tx;
  char *devname, *p;

#ifdef _WINDOWS
  printf("\nNOTE: The serial terminal is not supported on MS Windows. Use Linux instead.\n");
  return ERR_FAIL;
#endif

  if (argc != 0)
  {
    devname = get_next_string(argc, argv, &i);
    if (i < argc)
    {
      p = get_next_string(argc, argv, &i);
      if (!strcmp(p, "-l"))
      {
        lower_baudrate();
      }
      else
      if (!strcmp(p, "-b"))
      {
        p = get_next_string(argc, argv, &i);
        default_baudrate_g = atoi(p);
        fast_baudrate_g = default_baudrate_g;
        if ((default_baudrate_g < 1) || (default_baudrate_g > 2000000))
          quit_with_error("ERROR: Illegal baud set, valid range 1 - 2000000\n");
      }        
      else
      {
        quit_with_error("ERROR: Invalid argument\n");
      }
    }

    open_serial_device(devname);

    if (argc > i)
      quit_with_error("ERROR: Too many arguments\n");
  }


  printf("\nSerial terminal started. Press CTRL+C to quit.\n\n");
  
  for(;;)
  {
    rx = xfuart_receive(5);
    if (rx == -2)
    {
      printf("\nERROR: Serial connection broken.\n");
      return ERR_CONN;
    }
    else
    if (rx > 0)
    {
      putchar(rx);
      fflush(stdout);
    }
    while (keyboard_hit())
    {
      tx = keyboard_getch();
      if (tx == 3)
        break;
      if (tx == 10)
        tx = 13;
      if (tx > 0){
        xfuart_send((uint8_t)tx);
        usleep(10000);	//sleep 10ms per character
        if (tx==13){
          usleep(90000);	//sleep 100ms per line
        }
      }

    }
  }
  
  return 0;
}



/** The read subprogram
 */
static int tf_read(int argc, char *argv[], int i)
{
  char *devname, *p, *end, *filename;
  uint8_t *buffer;
  int binmode = 0;
  int first = 0, last = 0;
  int j, s, rc, len;
  int num_screens;
  FILE *f;

  devname = get_next_string(argc, argv, &i);
  p = get_next_string(argc, argv, &i);
  if (!strcmp(p, "-l"))
  {
    lower_baudrate();
    p = get_next_string(argc, argv, &i);
  }
  else
  if (!strcmp(p, "-b"))
  {
    p = get_next_string(argc, argv, &i);
    default_baudrate_g = atoi(p);
    fast_baudrate_g = default_baudrate_g;
    if ((default_baudrate_g < 1) || (default_baudrate_g > 2000000))
      quit_with_error("ERROR: Illegal baud set, valid range 1 - 2000000\n");
    p = get_next_string(argc, argv, &i);
  }        
        
  if (!strcmp(p,"binary")) {
    binmode = 1;
    p = get_next_string(argc, argv, &i);
  } 
  first = strtol(p, &end, 10);
  if ((first < 0) || (*end != 0))
    quit_with_error("ERROR: Invalid first screen number.");
  p = get_next_string(argc, argv, &i);
  last = strtol(p, &end, 10);
  if ((last < first) || (*end != 0))
    quit_with_error("ERROR: Invalid last screen number.");
  filename = get_next_string(argc, argv, &i);
  if (argc > i)
    quit_with_error("ERROR: Too many arguments\n");
  num_screens = last - first + 1;
  buffer = (uint8_t*)malloc(num_screens * 1024);
  if (buffer == NULL)
    quit_with_error("Failed to allocate memory\n");
  
  open_serial_device(devname);
  if (xfer_connect() != 0)
    exit(1);

  for (j = 0; j < num_screens; j++)
  {
    rc = transfer_download_screen(buffer + (j * 1024), first + j);
    if (rc == ERR_CONN)
    {
      free(buffer);
      quit_with_error("Serial connection error.");
    }
    if (rc == ERR_FAIL)
    {
      free(buffer);
      quit_with_error("Failed to download screens.");
    }
  }

  if (binmode)
  {
    f = fopen(filename, "wb");
    if (f == NULL)
    {
      free(buffer);
      quit_with_error("ERROR: Could not write to output file.");
    }
    if (fwrite(buffer, 1, num_screens * 1024, f) != num_screens * 1024)
    {
      fclose(f);
      free(buffer);
      quit_with_error("ERROR: Could not write to output file.");
    }
    fclose(f);
  }
  else
  {
    f = fopen(filename, "wb");
    if (f == NULL)
    {
      free(buffer);
      quit_with_error("ERROR: Could not write to output file.");
    }
    
    p = (char*)buffer;
    for (s = first; s <= last; s++)
    {
      fprintf(f, "----[%03d]-------------------------------------------------------" LINEEND, s);
      for (j = 0; j < 16; j++)
      {
        ffmt_make_printable(p, 64);
        len = ffmt_remove_trailing_whitespaces(p, 64);
        fwrite(p, 1, len, f);
        fprintf(f, LINEEND);
        p += 64;
      }
    }
    fprintf(f, "----[EOF]-------------------------------------------------------" LINEEND);
    fclose(f);
  }

  free(buffer);
  return 0;
}


/** The write subprogram
 */
static int tf_write(int argc, char *argv[], int i)
{
  screenmem_t *smem_root, *sm;
  char *devname, *p, *end, *filename;
  uint8_t *buffer;
  int binmode = 0;
  int textmode = 0;
  int screen = 0;
  int binsize, rc, j;
  FILE *f;

  devname = get_next_string(argc, argv, &i);
  p = get_next_string(argc, argv, &i);
  if (!strcmp(p, "-l"))
  {
    lower_baudrate();
    p = get_next_string(argc, argv, &i);
  }
  else
  if (!strcmp(p, "-b"))
  {
    p = get_next_string(argc, argv, &i);
    default_baudrate_g = atoi(p);
    fast_baudrate_g = default_baudrate_g;
    if ((default_baudrate_g < 1) || (default_baudrate_g > 2000000))
      quit_with_error("ERROR: Illegal baud set, valid range 1 - 2000000\n");
    p = get_next_string(argc, argv, &i);
  }         
  if (!strcmp(p,"-d")) {
    increase_send_delay();
    p = get_next_string(argc, argv, &i);    
  }
  
  if (!strcmp(p,"binary")) {
    binmode = 1;
    p = get_next_string(argc, argv, &i);
  } else
  if (!strcmp(p,"text")) {
    textmode = 1;
    p = get_next_string(argc, argv, &i);
  }
  screen = strtol(p, &end, 10);
  if ((screen < 0) || (*end != 0))
    quit_with_error("ERROR: Invalid screen number.");
  filename = get_next_string(argc, argv, &i);
  if (argc > i)
    quit_with_error("ERROR: Too many arguments\n");
  
  if (binmode)
  {
    f = fopen(filename, "rb");
    if (f == NULL)
      quit_with_error("ERROR: Could not open input file.");
    rc = fseek(f, 0, SEEK_END);
    if (rc >= 0)
      rc = ftell(f);
    fseek(f, SEEK_SET, 0);
    if (rc <= 0)
    {
      fclose(f);
      quit_with_error("ERROR: Failed to read input file.");
    }
    binsize = (rc + 0x3FF) & ~0x3FF;
    buffer = (uint8_t*)calloc(binsize, 1);
    if (buffer == NULL)
    {
      fclose(f);
      quit_with_error("ERROR: Failed to allocate memory.");
    }
    if (fread(buffer, 1, rc, f) != rc)
    {
      free(buffer);
      fclose(f);
      quit_with_error("ERROR: Failed to read input file.");
    }
    fclose(f);

    open_serial_device(devname);
    if (xfer_connect() != 0)
      exit(1);

    for (j = 0; j < binsize/1024; j++)
    {
      rc = transfer_upload_screen(buffer + (j * 1024), screen + j);
      if (rc == ERR_CONN)
      {
        free(buffer);
        quit_with_error("Serial connection error.");
      }
      if (rc == ERR_FAIL)
      {
        free(buffer);
        quit_with_error("Failed to upload screens.");
      }
    }
    free(buffer);
  }
  else
  if (textmode)
  {
    // unformatted text mode
    char *text, *t, *d;
    int flags = 0, sline = 0;
    int screen_cnt, line_cnt;
    int one_screen;

    text = ffmt_load_and_compress_file(filename, &flags);
    if (text == NULL)
      exit(1);
    
    if (*text == 0)
    {
      printf("No source code to upload.\n");
      return 0;
    }

    if (flags & FFMT_SCRFMT)
    {
      printf("\nNote: The file that you are uploading contains pre-formatted screens,\n");
      printf("      but you have chosen to upload unformatted source code.\n");
    }

    if (!(flags & FFMT_RUN))
    {
      printf("\nHint: There is no word RUN in the source code.\n");
      printf("      If you insert the word RUN in your code, your program will\n");
      printf("      be started automatically after loading from the EEPROM.\n");
    }
    
    // create screens from text file
    smem_root = (screenmem_t*)malloc(sizeof(screenmem_t));
    if (smem_root == NULL)
      quit_with_error("ERROR: Failed to allocate memory.");
    sm = smem_root;
    sm->next = NULL;
    sm->used = 1;
    memset(sm->data, 0x20, 1024);
    
    line_cnt = 0;
    for (t = text; *t != 0; t++) {
      if (*t == '\n')
        line_cnt++;
    }

    one_screen = (line_cnt <= 15) || ((line_cnt == 16) && !(flags & FFMT_RUN));
    sline = one_screen ? 0 : 5;
    screen_cnt = 1;
    t = text;
    while (*t != 0)
    {
      if (sline >= 16)
      {
        sm->next = (screenmem_t*)malloc(sizeof(screenmem_t));
        if (sm->next == NULL)
        {
          free_screenmem(smem_root);
          quit_with_error("ERROR: Failed to allocate memory.");
        }
        sm = sm->next;
        sm->next = NULL;
        sline = 0;
        screen_cnt++;
        memset(sm->data, 0x20, 1024);
      }
      sm->used = 1;
      d = (char*)&sm->data[sline * 64];
      while ((*t != 0) && (*t != '\n'))
        *d++ = *t++;
      if (*t == '\n') t++;
      sline++;
    }
    free(text);
    
    if (one_screen)
    {
      if (flags & FFMT_RUN)
      {
        if (sline > 15)
        {
          fprintf(stderr, "Program error, sline > 15 but one_screen = true\n");
          exit(1);
        }
        memcpy(&sm->data[sline * 64], "run", 3);
      }
    }
    else
    {
      // add loader code to first screen:
      #define PSLINE(n)  ((char*)&smem_root->data[n * 64])
      #define REMNLN(n)  PSLINE(n)[strlen(PSLINE(n))] = 0x20
      sprintf(PSLINE(0), "decimal : _fldrf c\" _ldrf\" find swap drop ; : _dorun c\" run\""); REMNLN(0);
      sprintf(PSLINE(1), "find if execute else drop then quit ; : _doldr dup 1 > if drop"); REMNLN(1);
      sprintf(PSLINE(2), "_dorun else 0= if %d %d thru _dorun then then ;", screen, screen + screen_cnt - 1); REMNLN(2);
      sprintf(PSLINE(3), ": _cldrf if s\" forget _fldrf\" evaluate then ;"); REMNLN(3);
      sprintf(PSLINE(4), "_fldrf create _ldrf 0 , _cldrf _ldrf @ dup 1+ _ldrf ! _doldr"); REMNLN(4);
    }

    if (flags & FFMT_RUN)
    {
      printf("\nHint:\n");
      printf("To load and run your program on My4TH, enter \" %d LOAD \" at the FORTH prompt.\n", screen);
    }
    else
    {
      printf("\nTo load your program on My4TH, enter \" %d LOAD \" at the FORTH prompt.\n", screen);
      printf("The program will not start automatically; this needs further actions by you.\n");
    }
    printf("\nUploading %d screens now:\n", screen_cnt);
    goto send_screens;
  }
  else
  { // screen mode
    char readbuf[200];

    smem_root = (screenmem_t*)malloc(sizeof(screenmem_t));
    if (smem_root == NULL)
      quit_with_error("ERROR: Failed to allocate memory.");
    sm = smem_root;
    sm->next = NULL;
    sm->used = 0;

    f = fopen(filename, "r");
    if (f == NULL)
    {
      fprintf(stderr, "FAILED to open source file '%s'\n", p);
      exit(1);
    }
    ffmt_screen_parser_init(filename);
    while (fgets(readbuf, 199, f) != NULL)
    {
      rc = ffmt_screen_parse_line(readbuf, sm->data, &sm->used);
      if (rc < 0)
      {
        fclose(f);
        free_screenmem(smem_root);
        exit(1);
      }
      if (rc > 0)
      {
        sm->used = 1;
        sm->next = (screenmem_t*)malloc(sizeof(screenmem_t));
        if (sm->next == NULL)
        {
          fclose(f);
          free_screenmem(smem_root);
          quit_with_error("ERROR: Failed to allocate memory.");
        }
        sm = sm->next;
        sm->next = NULL;
        sm->used = 0;
      }
    }
    fclose(f);
    
send_screens:
/*-----------------------------------------*/
/*
    for (sm = smem_root; sm != NULL; sm = sm->next)
    {
      if (sm->used != 0)
      {
        printf("==========\n");
        for (j=0; j < 16; j++)
        {
          int k;
          for (k=0; k<64; k++)
          {
            printf("%c", sm->data[j*64+k]);
          }
          printf("\n");
        }
        printf("==========\n");
      }
    }
    exit(1);
*/
/*-----------------------------------------*/

    open_serial_device(devname);
    if (xfer_connect() != 0)
      exit(1);

    for (sm = smem_root; sm != NULL; sm = sm->next)
    {
      if (sm->used != 0)
      {
        rc = transfer_upload_screen(sm->data, screen++);
        if (rc == ERR_CONN)
        {
          free_screenmem(smem_root);
          quit_with_error("Serial connection error.");
        }
        if (rc == ERR_FAIL)
        {
          free_screenmem(smem_root);
          quit_with_error("Failed to upload screens.");
        }
      }
    }

    free_screenmem(smem_root);
  }
  printf("The file was uploaded successfully.\n");
  return 0;
}


/** The exec subprogram
 */
static int tf_exec(int argc, char *argv[], int i)
{
  char *devname, *p, *o, *end;
  int timeout = DEFAULT_TIMEOUT;
  int linemode = 0;
  int terminalmode = 0;
  int ret;
  
  devname = get_next_string(argc, argv, &i);
  p = get_next_string(argc, argv, &i);
  if (!strcmp(p, "-l"))
  {
    lower_baudrate();
    p = get_next_string(argc, argv, &i);
  }
  else
  if (!strcmp(p, "-b"))
  {
    p = get_next_string(argc, argv, &i);
    default_baudrate_g = atoi(p);
    fast_baudrate_g = default_baudrate_g;
    if ((default_baudrate_g < 1) || (default_baudrate_g > 2000000))
      quit_with_error("ERROR: Illegal baud set, valid range 1 - 2000000\n");
    p = get_next_string(argc, argv, &i);
  }       
    
  if (!strcmp(p, "timeout"))
  {
    p = get_next_string(argc, argv, &i);
    timeout = strtol(p, &end, 10);
    if ((timeout <= 0) || (*end != 0))
      quit_with_error("ERROR: Invalid timeout option.");
    p = get_next_string(argc, argv, &i);
  }
  if (!strcmp(p,"line")) {
    linemode = 1;
  } else
  if (!strcmp(p,"file")) {
    linemode = 0;
  } else {
    quit_with_error("ERROR: Invalid option, must be 'line' or 'file'");
  }
  p = get_next_string(argc, argv, &i);

  if (argc > i)
  {
    o = get_next_string(argc, argv, &i);
    if (!strcmp(o, "-t")) {
      terminalmode = 1;
    } else {
      quit_with_error("ERROR: Unsupported option detected\n");
    }
  }

  if (argc > i)
    quit_with_error("ERROR: Too many arguments\n");

  ret = 0;
  if (linemode)
  {
    ffmt_cleanup_string(p);
    if (*p != 0)
    {
      open_serial_device(devname);
      if (xfer_connect() != 0)
        exit(1);

      ret = transfer_execute_string(p, timeout);
    }
  }
  else
  { /* file mode */
    FILE *f;
    char readbuf[300];

    open_serial_device(devname);
    if (xfer_connect() != 0)
      exit(1);

    f = fopen(p, "r");
    if (f == NULL)
    {
      fprintf(stderr, "FAILED to open source file '%s'\n", p);
      exit(1);
    }
    while (fgets(readbuf, 299, f) != NULL)
    {
      if (!ffmt_is_internal_comment_line(readbuf))
      {
        ffmt_cleanup_string(readbuf);
        if (readbuf[0] != 0)
        {
          ret = transfer_execute_string(readbuf, timeout);
          if (ret != 0)
            break;
        }
      }
    }
    fclose(f);
    if (ret == 0)
    {
      printf("The file was uploaded successfully.\n");
    }
  }

  if (ret == 0) {
    if (terminalmode)
      tf_term(0, NULL, 0);
  } else  
  if (ret == ERR_CONN) {
    quit_with_error("ERROR: Serial connection\n");
  } else
  if (ret == ERR_FAIL) {
    quit_with_error("ERROR: The command failed on My4TH.\n");
  }

  return 0;
}



/** Transfer main program
 */
int transfer_main(int argc, char *argv[])
{
  int ret = 1, help = 0, i = 1;
  
  if (argc >= i+2)
  {
    if (!strcmp(argv[i+1], "--help") || !strcmp(argv[i+1], "/?")) {
      ret = 0;
      help = 1;
    }
  }

  if ((argc >= i+1) && !help)
  {
    if (!strcmp(argv[i], "read")) {
      return tf_read(argc, argv, i+1);
    } else
    if (!strcmp(argv[i], "write")) {
      return tf_write(argc, argv, i+1);
    } else
    if (!strcmp(argv[i], "exec")) {
      return tf_exec(argc, argv, i+1);
    } else
    if (!strcmp(argv[i], "term")) {
      return tf_term(argc, argv, i+1);
    } else
    if (!strcmp(argv[i], "--help") || !strcmp(argv[i], "/?")) {
      ret = 0;
    }
  }
  
  if (ret)
    fprintf(stderr, "ERROR: Missing arguments\n");
  
  printf("\nMy4TH data transfer tool " VER_VERSION ", (c) by Dennis Kuschel in " VER_YEAR ", " VER_URL "\n");
#ifdef _LINUX
  printf("\nUsage: my4th read|write|exec|term serial-device [options]\n\n");
#else
  printf("\nUsage: my4th read|write|exec serial-device [options]\n\n");
#endif
  printf(" read : Read a block of memory (a screen) from the EEPROM storage.\n");
  printf(" write: Write a block of memory (a screen) to the EEPROM storage.\n");
  printf(" exec : Send a line of text or a text file to My4TH for execution.\n");
#ifdef _LINUX
  printf(" term : Start a very simple terminal emulation.\n");
#endif
  
  printf("\n my4th exec device [timeout sec] line \"commands\"\n");
  printf("   send one line with commands to My4TH, and My4TH executes it.\n");
  printf("   The option 'timeout' can be used to set an other timeout than the\n");
  printf("   default timeout of 5 seconds. Example:\n");
  printf("   $ my4th exec " EXAMPLE_DEV " line \"16 dup * .\"\n");
  printf("   $ my4th exec " EXAMPLE_DEV " timeout 20 line \"5 block\"\n");

#ifdef _LINUX
  printf("\n my4th exec device [timeout sec] file filename [-t]\n");
#else
  printf("\n my4th exec device [timeout sec] file filename\n");
#endif
  printf("   send the content of a file line-by-line to My4TH for execution. Example:\n");
  printf("   $ my4th exec " EXAMPLE_DEV " file my_first_forth_program.txt\n");
#ifdef _LINUX
  printf("   If the flag -t is set, the program stays in terminal mode after file upload.\n");
#endif
  
  printf("\n my4th read device [binary] firstscreen lastscreen filename\n");
  printf("   Read screen data from EEPROM storage. firstscreen is the number of the\n");
  printf("   first screen to read, and lastscreen is the last screen to read.\n");
  printf("   The screens are stored in text format in the file given with filename.\n");
  printf("   But when the special option 'binary' is set, the data is read and stored\n");
  printf("   in binary format. Examples:\n");
  printf("   $ my4th read " EXAMPLE_DEV " 3 12 screens-3-to-12.txt\n");
  printf("   $ my4th read " EXAMPLE_DEV " binary 0 255 eeprom-backup.bin\n");

  printf("\n my4th write device [text|binary] firstscreen filename\n");
  printf("   Write screen data to EEPROM storage. firstscreen is the number of the\n");
  printf("   first screen to be written. How many screens are written to the EEPROM\n");
  printf("   depends on the data size in the source file with the given filename.\n");
  printf("   When the option 'text' or 'binary' is not set, the data in the source\n");
  printf("   file is treated as pre-formatted FORTH source code screens. When 'text'\n");
  printf("   is set, you can upload an unformatted text file containing FORTH source\n");
  printf("   code to My4TH. When 'binary' is set, a binary memory image is uploaded\n");
  printf("   to the EEPROM. Examples:\n");
  printf("   $ my4th write " EXAMPLE_DEV " 20 forthscreens.txt\n");
  printf("   $ my4th write " EXAMPLE_DEV " text 12 testprogram.fth\n");
  printf("   $ my4th write " EXAMPLE_DEV " binary 4 customdata.bin\n");
  
  printf("\nNote: To lower the baudrate to 9600/4800 baud, add option -l directly after\n");
  printf("      the serial device name, e.g. \x22 " EXAMPLE_DEV " -l \x22.\n");
  printf("\n      To set the default baudrate, add option -b baud directly after\n");
  printf("      the serial device name, e.g. \x22 " EXAMPLE_DEV " -b 19200\x22.\n");
  printf("\n      To increase send delay (for 16MHz), add option -d directly after\n");
  printf("      the serial device name or baud set, e.g. \x22 " EXAMPLE_DEV " [-b 19200] -d\x22.\n");  
  
#ifdef _LINUX
  printf("\n      For the serial device you can also use named pipes. You need one pipe\n");
  printf("      for RX and one pipe for TX. Example:\n");
  printf("      $ my4th exec rxpipe,txpipe line \"32 5 3 + * .\"\n");
#endif

  printf("\n");
  return ret;
}


