/*
 *  File Transfer between the host and the My4TH computer
 */

#ifndef _TRANSFER_H
#define _TRANSFER_H

#include <stdint.h>


#define ERR_OK    0
#define ERR_FAIL  -1
#define ERR_CONN  -2

extern int default_baudrate_g;
extern int fast_baudrate_g;
extern int slow_baudrate_g;

/** Connect to My4TH, push firmware into binary connected mode.
 *  @return ERR_OK on success, otherwise ERR_CONN.
 */
int xfer_connect(void);


/** Execute a command string on My4TH.
 *  @param str      Forth string to execute, max 250 characters.
 *  @param timeout  Time in [ms] the command execution is allowed to last.
 *                  Default is 0 wich means 5 seconds.
 *  @return zero on success. ERR_FAIL if the command failed on My4TH.
 *          ERR_CONN is returned when the connection is broken.
 *  @note xfer_connect must be called before this function
 */
int transfer_execute_string(char *str, int timeout);


/** Download a Forth-Screen from EEPROM.
 *  @param buffer   ptr to a buffer that must have a size of at least 1kB.
 *  @param screen   the screen number of the screen to download
 *  @return zero on success. ERR_FAIL if the command failed on My4TH.
 *          ERR_CONN is returned when the connection is broken.
 *  @note xfer_connect must be called before this function
 */
int transfer_download_screen(uint8_t *buffer, int screen);


/** Upload a Forth-Screen to EEPROM.
 *  @param buffer   ptr to a buffer that must have a size of at least 1kB.
 *  @param screen   the screen number of the screen to upload
 *  @return zero on success. ERR_FAIL if the command failed on My4TH.
 *          ERR_CONN is returned when the connection is broken.
 *  @note xfer_connect must be called before this function
 */
int transfer_upload_screen(uint8_t *buffer, int screen);


/** Transfer main program.
 */
int transfer_main(int argc, char *argv[]);


#endif /* _TRANSFER_H */
