/*
 *   This file emulates a serial UART that connects
 *   to Inport1 Bit 0 and Output1 Bit 0 of the My4TH computer
 */

#include <stdio.h>
#include "opsystem.h"
#include "ioports.h"
#include "timing.h"
#include "xfer-uart.h"
#include "uart.h"

#ifdef _LINUX
#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>

static int pipe_rxd_g = -1;
static int pipe_txd_g = -1;
#endif


static volatile uint8_t     txbyte_g = 0;
static volatile int         haveTxByte_g = 0;
static volatile uarttxcb_t  txcallback_g = NULL;
static volatile uartrxcb_t  rxCallback_g = NULL;
static volatile uartrxcb_t  rxCallback2_g = NULL;


static void uart_transmitter(void)
{
  static int clkdiv = 0;
  static int bitstate = 0;
  uarttxcb_t txcb;
  
  if (++clkdiv >= 4)
  {
    clkdiv = 0;
    if (bitstate == 0)
    {
      if (haveTxByte_g != 0)
      {
        // send start-bit
        set_rxd(0);
        bitstate = 1;
      }
      else
      {
        set_rxd(1);
      }
    }
    else
    {
      bitstate++;
      if (bitstate < 10)
      {
        // send data bits
        set_rxd(txbyte_g & 1);
        txbyte_g >>= 1;
      }
      else
      { 
        // send stop-bit
        set_rxd(1);
        bitstate = 0;
        haveTxByte_g = 0;

        txcb = txcallback_g;
        txcallback_g = NULL;
        if (txcb != NULL)
          txcb();
      }
    }
  }
}


static void uart_receiver(void)
{
  static int clkdiv = 0;
  static int bitstate = 0;
  static uint8_t rxbyte = 0;
  static uint8_t last_rxd = 0;
  uint8_t rxd = get_txd();

  if (++clkdiv >= 4)
    clkdiv = 0;

  if (bitstate == 0)
  {
    /* wait for the start bit */
    if ((last_rxd == 1) && (rxd == 0))
    {
      clkdiv = 0;
      bitstate = 1;
    }
  }
  else
  {
    if (clkdiv == 0)
      bitstate++;
    
    if (clkdiv == 1)
    {
      if (bitstate < 10)
      {
        // 8 data bits
        rxbyte >>= 1;
        rxbyte |= (rxd << 7);
      }
      else
      {
        if (rxd == 1)  // stop bit
        {
          int f = 0;
          if (rxCallback2_g != NULL)
            f = (rxCallback2_g)(rxbyte);
          if ((f == 0) && (rxCallback_g != NULL))
            (rxCallback_g)(rxbyte);
#ifdef _LINUX
          if (pipe_txd_g >= 0)
          {
            if (write(pipe_txd_g, &rxbyte, 1) == 1)
              fsync(pipe_txd_g);
          }
#endif
        }
        bitstate = 0;
      }
    }
  }  
    
  last_rxd = rxd;
}


/*---------------------------------------------------------------------------*/


/** Send a byte over the UART
 *  @param b  byte to send
 *  @return nonzero if byte could be sent
 */
int uart_sendbyte(uint8_t b)
{
  if (!haveTxByte_g)
  {
    txbyte_g = b;
    haveTxByte_g = 1;
//printf("SEND------------------------------------ 0x%02X\n", b);
    return 1;
  }
  
  return 0;
}


/** Send a byte over the UART, callback a function
 *  when byte was sent out.
 *  @param b   byte to send
 *  @param cb  callback function
 *  @return nonzero if byte could be sent
 */
int uart_sendbyte_callback(uint8_t b, uarttxcb_t cb)
{
  if (haveTxByte_g)
    return 1;

  txcallback_g = cb;
  txbyte_g = b;
  haveTxByte_g = 1;
  return 0;
}


/** Checks if transmission is finished.
 *  @return true if transmission is finished.
 */
int uart_isTxFinished(void)
{
  return (haveTxByte_g == 0);
}


/** This function must be called at a rate of 9600 HZ (for 2400 Baud)
 */
void uart_timetick(void)
{
  uart_receiver();
  uart_transmitter();
  
#ifdef _LINUX
  if (pipe_rxd_g >= 0)
  {
    if (uart_isTxFinished())
    {
      uint8_t rxbuf[16];
      if (read(pipe_rxd_g, rxbuf, 1) > 0)
      {
        uart_sendbyte(rxbuf[0]);
      }
    }
  }
#endif
}


/** Register a UART data receiver callback
 *  @param cb   ptr to callback-function
 */
void uart_register_callback(uartrxcb_t cb)
{
  rxCallback_g = cb;
}


/** Register a second UART data receiver callback.
 *  The first callback gets no more called when the
 *  second callback has already consumed the data. That the
 *  data was consumed is signalized by returning nonzero
 *  from the callback function.
 *  @param cb   ptr to callback-function, NULL to de-register
 */
void uart_overload_callback(uartrxcb_t cb)
{
  rxCallback2_g = cb;
}


#ifdef _LINUX
static void closeUartPipes(void)
{
  close(pipe_rxd_g);
  close(pipe_txd_g);
}
#endif


/** Open a pipe for UART data transfer (Linux only)
 *  UART RX data is also taken from the pipe, and
 *  UART TX data is also put into the pipe.
 *  This means, that there are two ways for the
 *  UART data, that is the usual terminal and the pipe.
 *  @param rxpipe   name of the pipe used for UART RX
 *  @param txpipe   name of the pipe used for UART TX
 *  @return zero on success
 */
int uart_open_pipe(char *rxpipe, char *txpipe)
{
#ifdef _LINUX
  pipe_rxd_g = open(rxpipe, O_RDWR);
  if (pipe_rxd_g < 0)
  {
    fprintf(stderr, "ERROR: Failed to open input pipe '%s'\n", rxpipe);
    return 1;
  }

  pipe_txd_g = open(txpipe, O_RDWR);
  if (pipe_txd_g < 0)
  {
    fprintf(stderr, "ERROR: Failed to open input pipe '%s'\n", txpipe);
    close(pipe_rxd_g);
    return 1;
  }
  
  if ((fcntl(pipe_rxd_g, F_SETFL, O_NONBLOCK) < 0) ||
      (fcntl(pipe_txd_g, F_SETFL, O_NONBLOCK) < 0))
  {
    fprintf(stderr, "ERROR: Failed to set pipes to non-blocking mode\n");
    close(pipe_rxd_g);
    close(pipe_txd_g);
    return 1;
  }

  atexit(closeUartPipes);
  printf("Pipes opened.\n");

  return 0;
#else
  printf("UART pipe not supported, use Linux!\n");
  return -1;
#endif
}
