#ifndef _UART_H
#define _UART_H
#include <stdint.h>



typedef int (*uartrxcb_t)(uint8_t rxbyte);
typedef void (*uarttxcb_t)(void);


/** Send a byte over the UART
 *  @param b  byte to send
 *  @return nonzero if byte could be sent
 */
int uart_sendbyte(uint8_t b);


/** Send a byte over the UART, callback a function
 *  when byte was sent out.
 *  @param b   byte to send
 *  @param cb  callback function
 *  @return nonzero if byte could be sent
 */
int uart_sendbyte_callback(uint8_t b, uarttxcb_t cb);


/** Checks if transmission is finished.
 *  @return true if transmission is finished.
 */
int uart_isTxFinished(void);


/** This function must be called at a rate of 9600 HZ (for 2400 Baud)
 */
void uart_timetick(void);


/** Register a UART data receiver callback
 *  @param cb   ptr to callback-function
 */
void uart_register_callback(uartrxcb_t cb);


/** Register a second UART data receiver callback.
 *  The first callback gets no more called when the
 *  second callback has already consumed the data. That the
 *  data was consumed is signalized by returning nonzero
 *  from the callback function.
 *  @param cb   ptr to callback-function, NULL to de-register
 */
void uart_overload_callback(uartrxcb_t cb);


/** Open a pipe for UART data transfer (Linux only)
 *  UART RX data is also taken from the pipe, and
 *  UART TX data is also put into the pipe.
 *  This means, that there are two ways for the
 *  UART data, that is the usual terminal and the pipe.
 *  @param rxpipe   name of the pipe used for UART RX
 *  @param txpipe   name of the pipe used for UART TX
 *  @return zero on success
 */
int uart_open_pipe(char *rxpipe, char *txpipe);


#endif /* _UART_H */
