/*
 *  Generate a VCD wave output file
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <memory.h>
#include <time.h>
#include "vcd.h"


static FILE *vcdfile_g = NULL;

typedef enum sigtype {
  st_wire = 0,
  st_dbus,
  st_abus
} sigtype_t;


typedef struct vcdsig vcdsig_t;

struct vcdsig 
{
  vcdsig_t   *next;
  char        name[32];
  sigtype_t   type;
  void       *ptr;
  char        id[8];
  union {
    signal_t  wire;
    dbus_t    dbus;
    abus_t    abus;
  } last;
};

static vcdsig_t  *signallist_root_g = NULL;
static vcdsig_t  *signallist_end_g = NULL;
static unsigned  timescale_ctr_g = 0;



/** Convert a binary number to a string
 *  @param buf  string buffer
 *  @param val  value to convert
 *  @param bits number of bits
 *  @return the string
 */
static char* bin2str(char *buf, uint32_t val, int bits)
{
  uint32_t m = 1<<(bits-1);
  char *b = buf;

  while ((m > 1) && ((val & m) == 0))
    m >>= 1;

  *b++ = 'b';
  while (m != 0)
  {
    *b++ = (val & m) ? '1' : '0';
    m >>= 1;
  }
  *b = 0;
  return buf;
}


/** Create a new ID for a VCD signal
 *  @return  ID string
 */
static char* newid(void)
{
  static unsigned idctr = 0;
  static char buf[16];
  char *b;
  
  sprintf(buf, "%x", idctr++);
  for (b = buf; *b != 0; b++)
  {
    *b = ((unsigned)*b < 0x3A) ? 'A' + (*b & 0x0F) : 'J' + (*b & 0x0F);
  }
  return buf;
}


/** Create a new vcd signal watch object
 */
static void new_vcdsig(const char *name, sigtype_t type, void *p)
{
  vcdsig_t *v;
  int i;
  
  v = (vcdsig_t*) malloc(sizeof(vcdsig_t));
  if (v != NULL)
  {
    memset(v, 0, sizeof(vcdsig_t));
    i = strlen(name) + 1;
    if (i > sizeof(v->name)) {
      memcpy(v->name, name, sizeof(v->name)-1);
      v->name[sizeof(v->name)-1] = 0;
    } else {
      strcpy(v->name, name);
    }
    v->next = NULL;
    v->type = type;
    v->ptr = p;
    strcpy(v->id, newid());

    switch(type)
    {
      case st_wire:  v->last.wire = ~*((signal_t*)p); break;
      case st_dbus:  v->last.dbus = ~*((dbus_t*)p); break;
      case st_abus:  v->last.abus = ~*((abus_t*)p); break;
      default: break;
    }
    
    if (signallist_root_g == NULL)
    {
      signallist_root_g = v;
      signallist_end_g = v;
    }
    else
    {
      signallist_end_g->next = v;
      signallist_end_g = v;
    }
  }
}


/** Add a wire to the VCD view
 *  @param name   display name of the wire
 *  @param sig    ptr to the signal wire
 */
void vcd_add_wire(const char *name, signal_t *sig)
{
  new_vcdsig(name, st_wire, (void*)sig);
}


/** Add a 8-bit data bus to the VCD view
 *  @param name   display name of the bus
 *  @param bus    ptr to the data bus
 */
void vcd_add_dbus(const char *name, dbus_t *bus)
{
  new_vcdsig(name, st_dbus, (void*)bus);
}


/** Add a 16-bit address bus to the VCD view
 *  @param name   display name of the bus
 *  @param bus    ptr to the address bus
 */
void vcd_add_abus(const char *name, abus_t *bus)
{
  new_vcdsig(name, st_abus, (void*)bus);
}


/** Execute a new timestep with timescale resolution
 *  given in call to vcd_newfile.
 */
void vcd_timestep(void)
{
  vcdsig_t *v;
  char buf[40];
  int chng = 0;
  
  /* walk through the list and check if something has changed */
  for (v = signallist_root_g; v != NULL; v = v->next)
  {
    switch (v->type)
    {
      case st_wire:
      {
        if (v->last.wire != *((signal_t*)v->ptr))
        {
          v->last.wire = *((signal_t*)v->ptr);
          if (!chng)
          {
            fprintf(vcdfile_g, "#%d\n", timescale_ctr_g);
            chng = 1;
          }
          fprintf(vcdfile_g, "%d%s\n", v->last.wire & 1, v->id);
        }
        break;
      }

      case st_dbus:
      {
        if (v->last.dbus != *((dbus_t*)v->ptr))
        {
          v->last.dbus = *((dbus_t*)v->ptr);
          if (!chng)
          {
            fprintf(vcdfile_g, "#%d\n", timescale_ctr_g);
            chng = 1;
          }
          fprintf(vcdfile_g, "%s %s\n", bin2str(buf, v->last.dbus, 8), v->id);
        }
        break;
      }

      case st_abus:
      {
        if (v->last.abus != *((abus_t*)v->ptr))
        {
          v->last.abus = *((abus_t*)v->ptr);
          if (!chng)
          {
            fprintf(vcdfile_g, "#%d\n", timescale_ctr_g);
            chng = 1;
          }
          fprintf(vcdfile_g, "%s %s\n", bin2str(buf, v->last.abus, 16), v->id);
        }
        break;
      }
      default: break;
    }
  }

  timescale_ctr_g++;
}


/** Open a new VCD file for wave output
 *  @param filename   name of the new file
 *  @param comment    comment for the header
 *  @param timescale  timescale (in nanoseconds)
 *  @note Call vcd_add_wire / vcd_add_dbus / vcd_add_abus functions
 *        to register all signals that shall be recorded
 *        _before_ you call this function to create a new vcd file!
 */
int vcd_newfile(char *filename, char *comment, unsigned timescale)
{
  vcdsig_t *v;
  char buf[40];
#ifndef _WINDOWS
  time_t timer;
  struct tm* tm_info;
#endif

  vcdfile_g = fopen(filename, "w+");
  if (vcdfile_g == NULL)
  {
    printf("Failed to create new VCD output file '%s'\n", filename);
    return -1;
  }

#ifdef _WINDOWS
  // The strftime() implementation on Windows/MinGW is erroneous. So I insert the birthday of My4TH here :-)
  sprintf(buf, "Fri Nov  1 16:07:08 2019");
#else
  time(&timer);
  tm_info = localtime(&timer);
  strftime(buf, sizeof(buf), "%a %b %e %H:%M:%S %Y", tm_info);
#endif
  fprintf(vcdfile_g, "$date %s $end\n", buf);
  fprintf(vcdfile_g, "$version My4TH CPU Simulator v1.0 $end\n");
  fprintf(vcdfile_g, "$comment\n  %s\n$end\n", comment);
  fprintf(vcdfile_g, "$timescale %d ns $end\n", timescale);
  fprintf(vcdfile_g, "$scope module My4TH $end\n");

  for (v = signallist_root_g; v != NULL; v = v->next)
  {
    switch (v->type)
    {
      case st_wire: fprintf(vcdfile_g, "$var wire 1 %s %s $end\n", v->id, v->name); break;
      case st_dbus: fprintf(vcdfile_g, "$var wire 8 %s %s $end\n", v->id, v->name); break;
      case st_abus: fprintf(vcdfile_g, "$var wire 16 %s %s $end\n", v->id, v->name); break;
      default: break;
    }
  }
  
  fprintf(vcdfile_g, "$upscope $end\n");
  fprintf(vcdfile_g, "$enddefinitions $end\n");
  
  atexit(vcd_closefile);
  return 0;
}


/** Close / finish a VCD file
 */
void vcd_closefile(void)
{
  vcdsig_t *v;
  
  if (vcdfile_g != NULL)
  {
    fprintf(vcdfile_g, "#%d\n", timescale_ctr_g);
    fclose(vcdfile_g);
    vcdfile_g = NULL;
  }
  
  while (signallist_root_g != NULL)
  {
    v = signallist_root_g->next;
    free(signallist_root_g);
    signallist_root_g = v;
  }
}
