/*
 *  UART interface for the transfer tool
 */

#include <stdio.h>
#include <stdlib.h>
#include "uart.h"
#include "timing.h"
#include "opsystem.h"
#include "xfer-uart.h"

static volatile int xferuart_inited_g = 0;


/*-----------------------------------------------------------------------------
 *  LINUX Implementation
 *----------------------------------------------------------------------------*/
 
#ifdef _LINUX

//#define DEBUG_SERIAL

#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
//#include <termios.h>
#include <fcntl.h>
#include <poll.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <asm/termbits.h>


#ifdef DEBUG_SERIAL
static FILE *dumpfile_g = NULL;

static void closeDumpFile(void)
{
  if (dumpfile_g != NULL)
  {
    fprintf(dumpfile_g, "\n");
    fclose(dumpfile_g);
  }
}

static void dbgLineEnd(uint8_t *line, int len)
{
  uint8_t c;
  int i;

  if (len > 0)
  {
    for (i=len; i<16; i++)
    {
      fprintf(dumpfile_g, "   ");
    }
    fprintf(dumpfile_g, "  [");
    for (i=0; i<16; i++)
    {
      c = line[i];
      if ((c < 0x20) || (c > 0x7F))
        c = '.';
      if (i >= len)
        c = ' ';
      fprintf(dumpfile_g, "%c", (char)c);
    }
    fprintf(dumpfile_g, "]\n");
  }
}

/** Dump sent/received data to a file.
 *  @param rx  direction, true = received data, false = transmitted data
 *  @param data the data to dump.
 */
static void dbgData(bool rx, uint8_t data)
{
  static uint8_t line[16];
  static bool last_rx = false;
  static int xindex = 0;

  if (dumpfile_g == NULL)
  {
    dumpfile_g = fopen("serial-dump.txt", "w");
    if (dumpfile_g == NULL)
      return;
    atexit(closeDumpFile);
    last_rx = !rx;
  }

  if (last_rx != rx)
  {
    dbgLineEnd(line, xindex);
    if (rx) {
      fprintf(dumpfile_g, "\nReceived: ");
    } else {
      fprintf(dumpfile_g, "\nSent:     ");
    }
    last_rx = rx;
    xindex = 0;
  }
  else
  {
    if (xindex == 0)
      fprintf(dumpfile_g, "          ");
  }

  fprintf(dumpfile_g, " %02X", data);
  fflush(dumpfile_g);
  line[xindex++] = data;
  if (xindex >= 16)
  {
    dbgLineEnd(line, xindex);
    xindex = 0;
  }
}

#else
static void dbgData(bool rx, uint8_t data) { }
#endif

static int pipe_rxd_g = -1;
static int pipe_txd_g = -1;
static int rs232file_g = -1;
static int baudrate_g = -1;
static struct termios2 oldtermios_g;
static char uartdevice_g[500];
static int bytectr_g = 0;

//for 16MHz/14400
//static int send_delay1_g=60000000L;
//static int send_delay2_g=12000000L;

//for 20MHz/14400
static int send_delay1_g=30000000L;
static int send_delay2_g=6000000L;


void increase_send_delay(void){
  printf("Increase send delay\n"); 
  send_delay1_g=65000000L;
  send_delay2_g=13000000L;

  //send_delay1_g=send_delay1_g*2;
  //send_delay2_g=send_delay2_g*2;
}

/** Send a byte over RS232. This function blocks until
 *  the last bit of the data was sent out.
 *  @param data   data byte to send
 *  @return zero on success, any other value when
 *  the UART interface is no more available.
 */
int xfuart_send(uint8_t data)
{
  if (pipe_txd_g >= 0)
  {
    if (write(pipe_txd_g, &data, 1) != 1)
      return -1;
    fsync(pipe_txd_g);
    usleep(4160);  // 2400 baud
    dbgData(false, data);
    return 0; 
  }
  
  if (rs232file_g >= 0)
  {
    if (write(rs232file_g, &data, 1) != 1)
      return -1;
    if (++bytectr_g >= 7)
    {
      // timing-fix for USB->RS232-converter-cables
      bytectr_g = 0;
      usleep(send_delay1_g / baudrate_g);
    }
    else
    {
      usleep(send_delay2_g / baudrate_g);
    }
    dbgData(false, data);
    return 0;
  }

  fprintf(stderr,"ERROR: No UART configured\n");
  return -2;
}


/** Receives a byte over RS232. This function blocks
 *  until a byte was received or the timeout period
 *  has been elapsed.
 *  @param timeout    timeout in [ms]
 *  @return  positive values on success (the databyte).
 *           -1 is returned when the timeout was hit,
 *           -2 is returned when the UART interface
 *           is no more available.
 */
int xfuart_receive(int timeout)
{
  uint8_t rxbyte = 0;
  uint32_t timer = get_usec();
  uint32_t end, remain, rem100msecs;

  rem100msecs = timeout / 100;
  timeout %= 100;

  // busy-polling on the pipe, with timeout
  end = (uint32_t)(timeout * 1000);
  if ((end == 0) && (rem100msecs != 0))
  {
    end = 100000;
    rem100msecs--;
  }

  if (pipe_rxd_g >= 0)
  {
    for (;;)
    {
      if (read(pipe_rxd_g, &rxbyte, 1) > 0)
      {
        dbgData(true, rxbyte);
        return ((int)rxbyte) & 0xFF;
      }
      
      if (isTimeOver(&timer, end, get_usec(), &remain))
      {
        if (rem100msecs == 0)
          return -1;
        end = 100000;
        rem100msecs--;
      }

      usleep(1000);
    }
  }

  if (rs232file_g >= 0)
  {
    struct pollfd fds[1];
    int ret;
    
    for (;;)
    {
      fds[0].fd = rs232file_g;
      fds[0].events = POLLIN;
      
      ret = poll(fds, 1, timeout);
      if (ret < 0)
        return -2;
      if (ret > 0)
      {
        if (read(rs232file_g, &rxbyte, 1) > 0)
        {
          dbgData(true, rxbyte);
          bytectr_g = 5;
          return ((int)rxbyte) & 0xFF;
        }
      }

      if (isTimeOver(&timer, end, get_usec(), &remain))
      {
        if (rem100msecs == 0)
          return -1;
        end = 100000;
        rem100msecs--;
      }
      
      usleep(1000);
      timeout = 1;
    }
  }
  
  return -2;
}


/** This function gets called to set the baud rate
 *  of the RS232 connection. This function can be
 *  called at any time after the UART was initially
 *  set up (see function ::xfuart_setup)
 *  @param baud  new baud rate
 *  @return zero on success
 */
int xfuart_baudrate(int baud)
{
  struct termios2 newtio;
  int bd;

  if (!xferuart_inited_g)
    return -1;

  if (baudrate_g == baud)
    return 0;

  baudrate_g = -1;
  if (rs232file_g >= 0)
  {
    close(rs232file_g);
  }

  rs232file_g = open(uartdevice_g, O_RDWR | O_NOCTTY | O_NDELAY);
  if (rs232file_g < 0)
  {
    fprintf(stderr, "ERROR: Failed to open serial port '%s'\n", uartdevice_g);
    return -1;
  }

  /* save current port settings */
  if (ioctl(rs232file_g, TCGETS2, &oldtermios_g) != 0)
  {
    close(rs232file_g);
    rs232file_g = -1;
    fprintf(stderr, "ERROR: Failed to get device attributes of serial port '%s'\n", uartdevice_g);
    return -1;
  }

  baudrate_g = baud;
  newtio = oldtermios_g;
  
  bd=baud;
  newtio.c_cflag &= ~CBAUD;
  newtio.c_cflag |= CBAUDEX;
  newtio.c_cflag |= CREAD | CLOCAL;
  newtio.c_oflag &= ~OPOST;
  printf("Setting baud rate %d...\n", baud);
  /* set baud rate */
  newtio.c_ispeed=bd;
  newtio.c_ospeed=bd;  

  /* set mode 8N1 */
  newtio.c_cflag &= ~PARENB;
  newtio.c_cflag &= ~CSTOPB;
  newtio.c_cflag &= ~CSIZE;
  newtio.c_cflag |= CS8;

  /* no parity checking */
  newtio.c_iflag |= IGNPAR;
  
  /* no flow control */
  newtio.c_iflag &= ~(IXON | IXOFF | IXANY);
#ifdef CRTSCTS
  newtio.c_cflag &= ~CRTSCTS;
#endif
#ifdef CNEW_RTSCTS
  newtio.c_cflag &= ~CNEW_RTSCTS;
#endif

  /* set raw mode */
  newtio.c_lflag &= ~(ICANON | ECHO | ECHONL | ECHOE | ISIG | IEXTEN);
  newtio.c_iflag &= ~(IGNBRK | BRKINT | PARMRK | ISTRIP | INLCR | IGNCR | ICRNL);
  
  newtio.c_cc[VTIME] = 0;   /* inter-character timer unused */
  newtio.c_cc[VMIN]  = 0;   /* no blocking */

  if (ioctl(rs232file_g, TCFLSH, TCIOFLUSH) == -1)
  {
    close(rs232file_g);
    rs232file_g = -1;
    return -1;
  }

  if (ioctl(rs232file_g, TCSETS2, &newtio) == -1)
  {
    close(rs232file_g);
    rs232file_g = -1;
    return -1;
  }

  return 0;
}


/** Terminate this module
 */
void xfuart_term(void)
{
  if (xferuart_inited_g)
  {
    if (pipe_rxd_g >= 0)
      close(pipe_rxd_g);
    if (pipe_txd_g >= 0)
      close(pipe_txd_g);

    if (rs232file_g >= 0)
    {
      /* restore original port settings and close the device */
      ioctl(rs232file_g, TCFLSH, TCIFLUSH);
      ioctl(rs232file_g, TCSETS2, &oldtermios_g);
      close(rs232file_g);
    }
  }
  xferuart_inited_g = 0;
}


/** Initialize the UART / the RS232 interface.
 *  @param device    the name of the UART device
 *  @param baudrate  the baudrate
 *  @return zero on success
 */
int xfuart_init_rs232(char *device, int baudrate)
{
  if (xferuart_inited_g)
    return -1;

  strncpy(uartdevice_g, device, sizeof(uartdevice_g)-1);
  rs232file_g = -1;

  atexit(xfuart_term);
  xferuart_inited_g = 1;
  
  if (xfuart_baudrate(baudrate) != 0)
  {
    xferuart_inited_g = 0;
    fprintf(stderr, "ERROR: Failed to set baudrate for serial port '%s'\n", device);
    return -1;
  }
  
  return 0;
}


/** Initialize the UART / the local pipe interface.
 *  @param rxpipe   name of the receiver pipe file
 *  @param txpipe   name of the transmitter pipe file
 *  @return zero on success
 */
int xfuart_init_pipe(char *rxpipe, char *txpipe)
{
  if (xferuart_inited_g)
    return -1;

  pipe_rxd_g = -1;
  if (*rxpipe != 0)
    pipe_rxd_g = open(rxpipe, O_RDWR);
  if (pipe_rxd_g < 0)
  {
    fprintf(stderr, "ERROR: Failed to open input pipe '%s'\n", rxpipe);
    return -1;
  }

  pipe_txd_g = -1;
  if (*txpipe != 0)
    pipe_txd_g = open(txpipe, O_RDWR);
  if (pipe_txd_g < 0)
  {
    fprintf(stderr, "ERROR: Failed to open input pipe '%s'\n", txpipe);
    close(pipe_rxd_g);
    return -1;
  }
  
  if ((fcntl(pipe_rxd_g, F_SETFL, O_NONBLOCK) < 0) ||
      (fcntl(pipe_txd_g, F_SETFL, O_NONBLOCK) < 0))
  {
    fprintf(stderr, "ERROR: Failed to set pipes to non-blocking mode\n");
    close(pipe_rxd_g);
    close(pipe_txd_g);
    return -1;
  }

  xferuart_inited_g = 1;
  atexit(xfuart_term);
  return 0;
}


#else /* WINDOWS */


/*-----------------------------------------------------------------------------
 *  Windows Implementation
 *----------------------------------------------------------------------------*/

#include <windows.h>

static HANDLE     hCom_g = NULL;
static OVERLAPPED ovlRead_g;
static OVERLAPPED ovlWrite_g;
static uint8_t    pendWriteByte_g = 0;
static uint8_t    pendReadByte_g = 0;
static uint8_t    rdPendFlag_g = 0;
static int        baudrate_g = -1;
static char       uartdevice_g[MAX_PATH];


/** Send a byte over RS232. This function blocks until
 *  the last bit of the data was sent out.
 *  @param data   data byte to send
 *  @return zero on success, any other value when
 *  the UART interface is no more available.
 */
int xfuart_send(uint8_t data)
{
  DWORD written = 0, status;
  int waited = 0;

  /* Overlapped Write */
  pendWriteByte_g = data;
  ResetEvent(ovlWrite_g.hEvent);
  if (WriteFile(hCom_g, &pendWriteByte_g, 1, &written, &ovlWrite_g) != 0)
    return -2;

  status = GetLastError();
  if (status != ERROR_IO_PENDING)
    return (written == 1) ? 0 : -2;
      
  /* I/O operation is pending, so wait for completion */
  for(;;)
  {
    status = WaitForSingleObject(ovlWrite_g.hEvent, 0);
    if (status != WAIT_TIMEOUT)
      break;
    Sleep(waited ? 0 : (10000L / baudrate_g));
    waited = 1;
  }
  if (!GetOverlappedResult(hCom_g, &ovlWrite_g, &written, TRUE))
    return -1;

  return (written == 1) ? 0 : -1;
}


/** Receives a byte over RS232. This function blocks
 *  until a byte was received or the timeout period
 *  has been elapsed.
 *  @param timeout    timeout in [ms]
 *  @return  positive values on success (the databyte).
 *           -1 gets returned when the timeout was hit,
 *           -2 gets returned when the UART interface
 *           is no more available.
 */
int xfuart_receive(int timeout)
{
  DWORD read = 0, status;

  do
  {
    if (!rdPendFlag_g)
    {
      ResetEvent(ovlRead_g.hEvent);
      if (!ReadFile(hCom_g, &pendReadByte_g, 1, &read, &ovlRead_g))
      {
        status = GetLastError();
        if (status == ERROR_IO_PENDING)
        {
          rdPendFlag_g = 1;
        }
        else
        if (status || !read)
        {
          return -2;
        }
      }
    }

    if (rdPendFlag_g)
    {
      status = WaitForSingleObject(ovlRead_g.hEvent, timeout);
      if (status == WAIT_TIMEOUT)
        return -1;

      if (!GetOverlappedResult(hCom_g, &ovlRead_g, &read, TRUE))
        read = 0;

      if (read)
        rdPendFlag_g = 0;
    }
  }
  while (!read);

  return ((int)pendReadByte_g) & 0xFF;
}


/** This function gets called to set the baud rate
 *  of the RS232 connection. This function can be
 *  called at any time after the UART was initially
 *  set up (see function ::xfuart_setup)
 *  @param baud  new baud rate
 *  @return zero on success
 */
int xfuart_baudrate(int baud)
{
  if (!xferuart_inited_g)
    return -1;

  if (baud != baudrate_g)
  {
    Sleep(20);
    xfuart_term();
    Sleep(20);
    printf("Setting baud rate %d...\n", baud);    
    return xfuart_init_rs232(uartdevice_g, baud);
  }  
  return 0;
}


/** Terminate this module
 */
void xfuart_term(void)
{
  if (xferuart_inited_g)
  {
    CloseHandle(hCom_g);
    CloseHandle(ovlWrite_g.hEvent);
    CloseHandle(ovlRead_g.hEvent);
  }
  xferuart_inited_g = 0;
}


/** Initialize the UART / the RS232 interface.
 *  @param device   the name of the UART device
 *  @return zero on success
 */
int xfuart_init_rs232(char *device, int baudrate)
{
  COMMTIMEOUTS cto;
  DCB   dcb;
  DWORD br;
  int i;

  if (xferuart_inited_g)
    return -1;
  
  switch (baudrate)
  {
    case 2400 : br = CBR_2400; break;
    case 4800 : br = CBR_4800; break;
    case 9600 : br = CBR_9600; break;
    case 14400: br = CBR_14400; break;
    case 19200: br = CBR_19200; break;
    default: return -1;
  }

  if (device != uartdevice_g)
    strcpy(uartdevice_g, device);
  i = strlen(uartdevice_g);
  while ((i > 0) && ((uartdevice_g[i-1] == ':') || (uartdevice_g[i-1] == '/') || (uartdevice_g[i-1] <= ' '))) i--;
  uartdevice_g[i] = 0;

  hCom_g = CreateFile(uartdevice_g,
                      GENERIC_READ | GENERIC_WRITE,
                      0,                    // exclusive access
                      NULL,                 // no security attrs
                      OPEN_EXISTING,
                      FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED, // overlapped I/O
                      NULL );

  if ((hCom_g == NULL) || (hCom_g == INVALID_HANDLE_VALUE))
    goto reterr1;

  rdPendFlag_g = 0;
  memset(&ovlRead_g, 0, sizeof(ovlRead_g));
  ovlRead_g.hEvent  = CreateEvent(NULL, TRUE, FALSE, NULL); // set to manual reset
  if (!ovlRead_g.hEvent)
    goto reterr2;
  
  memset(&ovlWrite_g, 0, sizeof(ovlWrite_g));
  ovlWrite_g.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL); // set to manual reset
  if (!ovlWrite_g.hEvent)
    goto reterr3;

  ResetEvent(ovlRead_g.hEvent);
  ResetEvent(ovlWrite_g.hEvent);

  memset(&dcb, 0, sizeof(dcb));
  if (!GetCommState(hCom_g, &dcb))
    goto reterr3;

  dcb.BaudRate = br;
  dcb.StopBits = ONESTOPBIT;
  dcb.ByteSize = 8;
  dcb.Parity = NOPARITY;
  dcb.fBinary = TRUE;                     // binary mode, no EOF check
  dcb.fParity = FALSE;                    // disable parity checking
  dcb.fOutxCtsFlow = FALSE;               // CTS output flow control
  dcb.fOutxDsrFlow = FALSE;               // DSR output flow control
  dcb.fDtrControl = DTR_CONTROL_ENABLE;   // DTR flow control type (level of DTR output, alternativ DTR_CONTROL_DISABLE)
  dcb.fDsrSensitivity = FALSE;            // DSR sensitivity
  dcb.fTXContinueOnXoff = FALSE;          // XOFF continues Tx
  dcb.fOutX = FALSE;                      // XON/XOFF out flow control
  dcb.fInX = FALSE;                       // XON/XOFF in flow control
  dcb.fErrorChar = FALSE;                 // enable error replacement
  dcb.fNull = FALSE;                      // enable null stripping
  dcb.fRtsControl = RTS_CONTROL_DISABLE;  // RTS flow control (level of RTS output, alternativ RTS_CONTROL_ENABLE)
  dcb.fAbortOnError = FALSE;              // abort reads/writes on error

  if (!SetCommState(hCom_g, &dcb))
    goto reterr3;

  if (!SetupComm(hCom_g, 256 /*recbuf*/, 256 /*sendbuf*/))
    goto reterr3;

  if (!PurgeComm(hCom_g, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR))
    goto reterr3;

  if (!SetCommMask(hCom_g, EV_RXCHAR | EV_RXFLAG | EV_RLSD))
    goto reterr3;

  cto.ReadIntervalTimeout = 0;
  cto.ReadTotalTimeoutMultiplier = 0;
  cto.ReadTotalTimeoutConstant = 0;
  cto.WriteTotalTimeoutMultiplier = 0;
  cto.WriteTotalTimeoutConstant = 0;

  if (!SetCommTimeouts(hCom_g, &cto))
    goto reterr3;

  EscapeCommFunction(hCom_g, SETDTR);
  EscapeCommFunction(hCom_g, CLRRTS);
  
  baudrate_g = baudrate;
  xferuart_inited_g = 1;
  atexit(xfuart_term);
  return 0;

reterr3:
  CloseHandle(ovlWrite_g.hEvent);
reterr2:
  CloseHandle(ovlRead_g.hEvent);
reterr1:
  CloseHandle(hCom_g);
  return -1;
}


/** Initialize the UART / the local pipe interface.
 *  @param rxpipe   name of the receiver pipe file
 *  @param txpipe   name of the transmitter pipe file
 *  @return zero on success
 */
int xfuart_init_pipe(char *rxpipe, char *txpipe)
{
  printf("Pipes not supported, use Linux!\n");
  return -1;
}


#endif /* WINDOWS */

