/*
 *  UART interface for the transfer tool
 */

#ifndef _XFER_UART_H
#define _XFER_UART_H

#include <stdio.h>
#include <stdint.h>

//Increase send delay
void increase_send_delay(void);

/** Send a byte over RS232. This function blocks until
 *  the last bit of the data was sent out.
 *  @param data   data byte to send
 *  @return zero on success, any other value when
 *  the UART interface is no more available.
 */
int xfuart_send(uint8_t data);


/** Receives a byte over RS232. This function blocks
 *  until a byte was received or the timeout period
 *  has been elapsed.
 *  @param timeout    timeout in [ms]
 *  @return  positive values on success (the databyte).
 *           -1 is returned when the timeout was hit,
 *           -2 is returned when the UART interface
 *           is no more available.
 */
int xfuart_receive(int timeout);


/** This function gets called to set the baud rate
 *  of the RS232 connection. This function can be
 *  called at any time after the UART was initially
 *  set up (see function ::xfuart_setup)
 *  @param baud  new baud rate
 *  @return zero on success
 */
int xfuart_baudrate(int baud);


/** Initialize the UART / the RS232 interface.
 *  @param device    the name of the UART device
 *  @param baudrate  the baudrate
 *  @return zero on success
 */
int xfuart_init_rs232(char *device, int baudrate);


/** Initialize the UART / the local pipe interface.
 *  @param rxpipe   name of the receiver pipe file
 *  @param txpipe   name of the transmitter pipe file
 *  @return zero on success
 */
int xfuart_init_pipe(char *rxpipe, char *txpipe);


/** Terminate this module
 */
void xfuart_term(void);


#endif /* _XFER_UART_H */
