;------------------------------------------------------------------------------
; Data transfer (via RS232 to/from the host computer)
;------------------------------------------------------------------------------

SEG_LOWCODE

CMD_NOP       SET 0x00
CMD_LISTEN1   SET 0x01
CMD_LISTEN2   SET 0x06
CMD_EXECUTE   SET 0x02  ;execute text string
CMD_READ1K    SET 0x12  ;read 1kb from EEPROM
CMD_WRITE1K   SET 0x13  ;write 1kb text data to EEPROM

REP_NACK      SET 0x04
REP_ACK       SET 0x05
REP_COK       SET 0x0E
REP_CERR      SET 0x0F
REP_BUSY      SET 0x10
REP_CONT      SET 0x11
REP_BINMODE   SET 0x86

ESC_00_1F     SET 0x15  ;Escape for 0x00 - 0x1F, followed by an ASCII character where only bits 0-4 are used
ESC_80_9F     SET 0x16  ;Escape for 0x80 - 0x9F, followed by an ASCII character where only bits 0-4 are used
ESC_A0_FF     SET 0x17  ;Escape for 0xA0 - 0xFF, followed by an ASCII character in the range 0x20 - 0x7F


readkey_xfer:
          ; Read a key input from either UART or the keyboard.
          ; If a special character was received over UART, the data
          ; transfer mode will be entered. When the data transfer
          ; mode was entered in the meantime, this function will not return.
          ; Out: ACCU = key code (ASCII value)
          PHL
          LD   LIB_TFDATA+5,#0  ;flag: if set, forth must be restarted
_rkxf00
#if !defined(NO_LCD) || defined(WITH_I2C_LCD_KYBD)
          TST  LIB_BIGKEYB
          JNF  _rkxf01
#else
#ifdef MY4TH_NFD
          TST  LIB_BIGKEYB
          JNF  _rkxf01
#endif
#endif
          ;no keyboard installed, poll only the UART
_rkxf04   JSR  uart_recv
          CMP  #CMD_LISTEN1
          JPF  _binmode_enter
          CMP  #CMD_LISTEN2
          JPF  _binmode_enter
          TST
          JPF  _rkxf04
          RTS  ;return key code

readkey:  ; Same like readkey_xfer, but also decodes VT100 escape sequences.
          ; This function implements a keyboard abstraction layer, the
          ; returned key codes are the same for a physical keyboard and a
          ; serial VT100 terminal.
          ; Out: ACCU = key code (ASCII value)
          PHL
          LD   LIB_TFDATA+5,#0  ;flag: if set, forth must be restarted
#if !defined(NO_LCD) || defined(WITH_I2C_LCD_KYBD)
          TST  LIB_BIGKEYB
          JNF  _rkxf01  ;read key from keyboard (and poll UART for external control connection)
#else
#ifdef MY4TH_NFD
          TST  LIB_BIGKEYB
          JNF  _rkxf01
#endif
#endif
_rdkey01  JSR  uart_recv
_rdkey08  CMP  #27
          JPF  _rdkey07
          CMP  #CMD_LISTEN1
          JPF  _binmode_enter
          CMP  #CMD_LISTEN2
          JPF  _binmode_enter
          TST
          JPF  _rdkey01 
          CMP  #0x7F
          JNF  rts_opc
          LDA  #KCODE_BS
          RTS
_rdkey10  LDA  #KCODE_HOME
          RTS
_rdkey07  JSR  uart_recv
          CMP  #'['
          JNF  _rdkey02
          JSR  uart_recv
          CMP  #'1'
          JNF  _rdkey06
          JSR  uart_recv
          ;Accu = 0-9 or A-D or ~
          CMP  #0x7E  ; '~'
          JPF  _rdkey10 ; 1~
          STA  R2
          LDA  #0xC0
          AND  R2
          TST
          JNF  _rdkey09  ;jump if A-Z
          ;is number, receive ~
          JSR  uart_recv  ;receive (and drop) the ~
          LDA  R2
          ;numbers 1-9 = keys F1-F8
_rdkey12  LD   PAR1,#0x1F
          ;accu = 0x10 - 0x19, 0x1A - 0x1C
_rdkey11  AND  PAR1
          OR   #<tab_vt100keys
          STA  PTR_L
          LD   PTR_H,#>tab_vt100keys
          LAP
          TST
          JPF  _rdkey01
          RTS
_rdkey06  ;Accu = 2-4 (folowed by ~) or A-D
          STA  R2
          LDA  #0xC0
          AND  R2
          TST
          JNF  _rdkey09  ;jump if A-Z
          ;is number, receive ~
          JSR  uart_recv  ;receive (and drop) the ~
          LDA  R2
          ;numbers 2-4 = keys INSERT,DELETE,END
          AD   #8
          JMP  _rdkey12
_rdkey09  LDA  R2
          LD   PAR1,#0x0F
          JMP  _rdkey11
_rdkey02  ;assume we just received "O" as prefix for F1-F4 keys
          JSR  uart_recv
          ;receive P-S for F1-F4 key
          LD   PAR1,#0x13
          INC
          JMP  _rdkey11


_cont_readkey:
          ;continue with reading next key.
          ;check if FORTH environment must be reset before reading next key.                    
          TST  LIB_TFDATA+5
          JNF  restart_forth
          ;check if we should simply return to called after remote string execution
          LDA  INPUTDEV
          CMP  #INDEV_REMOTE
          JPF  rts_opc
          ;check if input is done over uart: if yes, do a clean restart of the input prompt
#if !defined(NO_LCD) || defined(WITH_I2C_LCD_KYBD)
          TST  LIB_BIGKEYB
          JPF  forth_restart_prompt
          ;loop
          JMP  _rkxf00
#else
#ifdef MY4TH_NFD
          TST  LIB_BIGKEYB
          JPF  forth_restart_prompt
          ;loop
          JMP  _rkxf00
#else
          JMP  forth_restart_prompt
#endif
#endif


_rkxf01   ;Read key from UART or keyboard. Wait until a key was pressed.
          ;1. poll UART
          IN
#ifndef PLATFORM_XS
          STA  FLAG
#endif
          JNF  _rkxf02
#if !defined(NO_LCD) || defined(WITH_I2C_LCD_KYBD)
          ;2. poll keyboard
          JSR  keyb_readkey
          TST
          JPF  _rkxf01
          RTS
#else
#ifdef MY4TH_NFD
          ;2. poll keyboard
          JSR  FDVECT_READKEY
          TST
          JPF  _rkxf01
          RTS
#else
          JMP  _rkxf01
#endif
#endif


test_for_io_activity:
          ;Test for I/O-activity on UART or keyboard
          ;returns with FLAG = 0 if there is activity.
          ;changes ACCU, R0, PTR
          PHL
          IN
#ifndef PLATFORM_XS
          STA  FLAG
#endif
#if !defined(NO_LCD) || defined(WITH_I2C_LCD_KYBD)
          JNF  rts_opc
          ;poll keyboard
          TST  LIB_BIGKEYB
          JPF  _tfioa01
          JSR  keyb_readkey
          TST
#else
#ifdef MY4TH_NFD
          JNF  rts_opc
          ;poll keyboard
          TST  LIB_BIGKEYB
          JPF  _tfioa01
          JSR  FDVECT_READKEY
          TST
#endif
#endif
          RTS  ;flag = 0 if there is I/O activity (keyboard or RS232)
_tfioa01  LD   R0,#18
_tfioa02  IN
#ifndef PLATFORM_XS
          STA  FLAG
#endif
          JNF  rts_opc
          JLP  _tfioa02
          RTS  ;flag = 1, no activity


;------------------------------------------------------
;NOTE: This code must be within one 256-byte page !!!!
transfer_functions Segment Code
ORG ROM_XFER


;UART routines for 8 MHz and 16 MHz CPU clock frequency:
#if (CPUCLOCK == 8) || (CPUCLOCK == 16)

#ifdef BAUD14400	; 14400 baud


_rkxf02   ;low-condition on UART detected. Try to receive a byte.
          ;1. wait for one bit-time to skip the first zero (if host baud rate is too low)
          ROL
          ;2. wait for a high-level on RXD
_rkxf07   IN
          STA  FLAG
          JNF  _rkxf07
          ;hopefully we skiped this byte, and the high-level is the stop bit / idle state
          ;3. wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_rkxf08   IN
          STA  FLAG
          JNF  _rkxf05
          JLP  _rkxf08
          IN
          STA  FLAG
          JNF  _rkxf05
          DEC  R1
          TST  R1
          JNF  _rkxf08
          ;timeout, continue with read
          JMP  _cont_readkey
_rkxf05   ;4. receive the byte
          ;delay
          ;receive 8 bits
          LD   R0,#8
_rkxf06   ;delay
          OR  ACCU
          INC
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _rkxf06
          ;all bits received
          LDA  PAR2
          ;check for special codes
          CMP  #CMD_LISTEN1
          JPF  _binmode_enter
          CMP  #CMD_LISTEN2
          JNF  _cont_readkey

_binmode_enter:  ;"LISTEN"-command received via UART
          ;send a reply to the host, showing him that we are in binary mode now.
          LDA  #REP_BINMODE
_binmode_loop_tx:
          JSR  uart_xmit
          
_binmode_loop:
          ;binary mode loop
          ;wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_bmlp01   IN
          STA  FLAG
          JNF  _bmlp02
          JLP  _bmlp01
          IN
          STA  FLAG
          JNF  _bmlp02
          DEC  R1
          TST  R1
          JNF  _bmlp01
          ;timeout, continue with read
          JMP  _cont_readkey
_bmlp02   ;receive the byte
          ;delay
          ;receive 8 bits
          LD   R0,#8
_bmlp03   ;delay
          OR  ACCU
          INC             
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _bmlp03
          ;all bits received
          LDA  PAR2
          JMP  _cmd_decode

_cmds_receive:
          ;Common code to all commands: receive the command string
          ;Out: R7 = bytes in the buffer,
          ;     PTR points to beginning of data
          PHL
          LD   PTR_L,#<UART_BUFFER
          LD   PTR_H,#>UART_BUFFER
          LDA  #REP_CONT
          JSR  uart_xmit
          ;uart receiver loop
_cmdrx2   IN
          STA  FLAG
          JPF  _cmdrx2
          ;delay
          ;receive 8 bits
          LD   R0,#8
_cmdrx3   ;delay
          OR  ACCU
          INC
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _cmdrx3
          ;store the received byte
          LDA  PAR2
          TST
          JPF  _cmdrx4  ; end of received string
          SAP
          INC  PTR_L
          TST  PTR_L
          JNF  _cmdrx1
          INC  PTR_H
_cmdrx1   ;wait until RXD is high again
          IN
          STA  FLAG
          JNF  _cmdrx1
          JMP  _cmdrx2


#else ; 9600 baud

_rkxf02   ;low-condition on UART detected. Try to receive a byte.
#ifdef PLATFORM_XS
          ;1. wait for one bit-time to skip the first zero (if host baud rate is too low)
          OR   ACCU
          XOR  ACCU
          ;2. wait for a high-level on RXD
_rkxf07   IN
          JNF  _rkxf07
          ;hopefully we skiped this byte, and the high-level is the stop bit / idle state
          ;3. wait for a low-level ("start condition") with timeout
          LD   R1,#32
          LD   R0,#0
_rkxf08   IN
          JNF  _rkxf05
          JLP  _rkxf08
          IN
          JNF  _rkxf05
          DEC  R1
          TST  R1
          JNF  _rkxf08
          ;timeout, continue with read
          JMP  _cont_readkey
_rkxf05   ;4. receive the byte
          ;delay
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_rkxf06   ;delay
          AND  ACCU
          CLC
          CLC
          INC
          INC
          ;receive next bit
          IN
          ROR  PAR2
          JLP  _rkxf06
#else     ;---------------------------
          ;1. wait for one bit-time to skip the first zero (if host baud rate is too low)
          XOR  ACCU
          XOR  ACCU
          ;2. wait for a high-level on RXD
_rkxf07   IN
          STA  FLAG
          JNF  _rkxf07
          ;hopefully we skiped this byte, and the high-level is the stop bit / idle state
          ;3. wait for a low-level ("start condition") with timeout
          LD   R1,#32
          LD   R0,#0
_rkxf08   IN
          STA  FLAG
          JNF  _rkxf05
          JLP  _rkxf08
          IN
          STA  FLAG
          JNF  _rkxf05
          DEC  R1
          TST  R1
          JNF  _rkxf08
          ;timeout, continue with read
          JMP  _cont_readkey
_rkxf05   ;4. receive the byte
          ;delay
          INC
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_rkxf06   ;delay
          XOR  ACCU
          ROL
          ROL
          ROL
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _rkxf06
#endif
          ;all bits received
          LDA  PAR2
          ;check for special codes
          CMP  #CMD_LISTEN1
          JPF  _binmode_enter
          CMP  #CMD_LISTEN2
          JNF  _cont_readkey

_binmode_enter:  ;"LISTEN"-command received via UART
          ;send a reply to the host, showing him that we are in binary mode now.
          LDA  #REP_BINMODE
_binmode_loop_tx:
          JSR  uart_xmit
          
_binmode_loop:
          ;binary mode loop
          ;wait for a low-level ("start condition") with timeout
          LD   R1,#32
          LD   R0,#0
#ifdef PLATFORM_XS
_bmlp01   IN
          JNF  _bmlp02
          JLP  _bmlp01
          IN
          JNF  _bmlp02
          DEC  R1
          TST  R1
          JNF  _bmlp01
          ;timeout, continue with read
          JMP  _cont_readkey
_bmlp02   ;receive the byte
          ;delay
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_bmlp03   ;delay
          AND  ACCU
          CLC
          CLC
          INC
          INC
          ;receive next bit
          IN
          ROR  PAR2
          JLP  _bmlp03
#else     ;----------------------
_bmlp01   IN
          STA  FLAG
          JNF  _bmlp02
          JLP  _bmlp01
          IN
          STA  FLAG
          JNF  _bmlp02
          DEC  R1
          TST  R1
          JNF  _bmlp01
          ;timeout, continue with read
          JMP  _cont_readkey
_bmlp02   ;receive the byte
          ;delay
          INC
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_bmlp03   ;delay
          XOR  ACCU
          ROL
          ROL
          ROL
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _bmlp03
#endif
          ;all bits received
          LDA  PAR2
          JMP  _cmd_decode

_cmds_receive:
          ;Common code to all commands: receive the command string
          ;Out: R7 = bytes in the buffer,
          ;     PTR points to beginning of data
          PHL
          LD   PTR_L,#<UART_BUFFER
          LD   PTR_H,#>UART_BUFFER
          LDA  #REP_CONT
          JSR  uart_xmit
          ;uart receiver loop
#ifdef PLATFORM_XS
_cmdrx2   IN
          JPF  _cmdrx2
          ;delay
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_cmdrx3   ;delay
          AND  ACCU
          CLC
          CLC
          INC
          INC
          ;receive next bit
          IN
          ROR  PAR2
          JLP  _cmdrx3
#else     ;----------------------
_cmdrx2   IN
          STA  FLAG
          JPF  _cmdrx2
          ;delay
          INC
          INC
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_cmdrx3   ;delay
          XOR  ACCU
          ROL
          ROL
          ROL
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _cmdrx3
#endif
          ;store the received byte
          LDA  PAR2
          TST
          JPF  _cmdrx4  ; end of received string
          SAP
          INC  PTR_L
          TST  PTR_L
          JNF  _cmdrx1
          INC  PTR_H
_cmdrx1   ;wait until RXD is high again
          IN
#ifndef PLATFORM_XS
          STA  FLAG
#endif
          JNF  _cmdrx1
          JMP  _cmdrx2

#endif ; FASTBAUD

#endif ; 8/16 MHz

;------------------------------------------------------------------------------

;UART routines for 10 MHz and 20 MHz CPU clock frequency:
#if (CPUCLOCK == 10) || (CPUCLOCK == 20)

#ifdef BAUD14400	; 14400 baud

_rkxf02   ;low-condition on UART detected. Try to receive a byte.
          ;1. wait for one bit-time to skip the first zero (if host baud rate is too low)
          LDA  ACCU
          ;CLC
          OR   ACCU
          ;2. wait for a high-level on RXD
_rkxf07   IN
          STA  FLAG
          JNF  _rkxf07
          ;hopefully we skiped this byte, and the high-level is the stop bit / idle state
          ;3. wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_rkxf08   IN
          STA  FLAG
          JNF  _rkxf05
          JLP  _rkxf08
          IN
          STA  FLAG
          JNF  _rkxf05
          DEC  R1
          TST  R1
          JNF  _rkxf08
          ;timeout, continue with read
          JMP  _cont_readkey
_rkxf05   ;4. receive the byte
          ;delay
          ;receive 8 bits
          LD   R0,#8
_rkxf06   ;delay
          LDA  ACCU
          CLC
          OR   ACCU
          INC
          INC          
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _rkxf06
          ;all bits received
          LDA  PAR2
          ;check for special codes
          CMP  #CMD_LISTEN1
          JPF  _binmode_enter
          CMP  #CMD_LISTEN2
          JNF  _cont_readkey

_binmode_enter:  ;"LISTEN"-command received via UART
          ;send a reply to the host, showing him that we are in binary mode now.
          LDA  #REP_BINMODE
_binmode_loop_tx:
          JSR  uart_xmit
          
_binmode_loop:
          ;binary mode loop
          ;wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_bmlp01   IN
          STA  FLAG
          JNF  _bmlp02
          JLP  _bmlp01
          IN
          STA  FLAG
          JNF  _bmlp02
          DEC  R1
          TST  R1
          JNF  _bmlp01
          ;timeout, continue with read
          JMP  _cont_readkey
_bmlp02   ;receive the byte
          ;delay
          ;receive 8 bits
          LD   R0,#8
_bmlp03   ;delay
          LDA  ACCU
          CLC
          OR   ACCU
          INC
          INC              
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _bmlp03
          ;all bits received
          LDA  PAR2
          JMP  _cmd_decode

_cmds_receive:
          ;Common code to all commands: receive the command string
          ;Out: R7 = bytes in the buffer,
          ;     PTR points to beginning of data
          PHL
          LD   PTR_L,#<UART_BUFFER
          LD   PTR_H,#>UART_BUFFER
          LDA  #REP_CONT
          JSR  uart_xmit
          ;uart receiver loop
_cmdrx2   IN
          STA  FLAG
          JPF  _cmdrx2
          ;delay
          ;receive 8 bits
          LD   R0,#8
_cmdrx3   ;delay
          LDA  ACCU
          CLC
          OR   ACCU
          INC
          INC              
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _cmdrx3
          ;store the received byte
          LDA  PAR2
          TST
          JPF  _cmdrx4  ; end of received string
          SAP
          INC  PTR_L
          TST  PTR_L
          JNF  _cmdrx1
          INC  PTR_H
_cmdrx1   ;wait until RXD is high again
          IN
          STA  FLAG
          JNF  _cmdrx1
          JMP  _cmdrx2



#else ; 9600 baud



_rkxf02   ;low-condition on UART detected. Try to receive a byte.
          ;1. wait for one bit-time to skip the first zero (if host baud rate is too low)
          XOR  ACCU
          XOR  ACCU
          ROL
          ;2. wait for a high-level on RXD
_rkxf07   IN
          STA  FLAG
          JNF  _rkxf07
          ;hopefully we skiped this byte, and the high-level is the stop bit / idle state
          ;3. wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_rkxf08   IN
          STA  FLAG
          JNF  _rkxf05
          JLP  _rkxf08
          IN
          STA  FLAG
          JNF  _rkxf05
          DEC  R1
          TST  R1
          JNF  _rkxf08
          ;timeout, continue with read
          JMP  _cont_readkey
_rkxf05   ;4. receive the byte
          ;delay
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_rkxf06   ;delay
          XOR  ACCU
          OR   ACCU
          LDA  ACCU
          INC
          CLC
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _rkxf06
          ;all bits received
          LDA  PAR2
          ;check for special codes
          CMP  #CMD_LISTEN1
          JPF  _binmode_enter
          CMP  #CMD_LISTEN2
          JNF  _cont_readkey

_binmode_enter:  ;"LISTEN"-command received via UART
          ;send a reply to the host, showing him that we are in binary mode now.
          LDA  #REP_BINMODE
_binmode_loop_tx:
          JSR  uart_xmit
          
_binmode_loop:
          ;binary mode loop
          ;wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_bmlp01   IN
          STA  FLAG
          JNF  _bmlp02
          JLP  _bmlp01
          IN
          STA  FLAG
          JNF  _bmlp02
          DEC  R1
          TST  R1
          JNF  _bmlp01
          ;timeout, continue with read
          JMP  _cont_readkey
_bmlp02   ;receive the byte
          ;delay
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_bmlp03   ;delay
          XOR  ACCU
          OR   ACCU
          LDA  ACCU
          INC
          CLC
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _bmlp03
          ;all bits received
          LDA  PAR2
          JMP  _cmd_decode

_cmds_receive:
          ;Common code to all commands: receive the command string
          ;Out: R7 = bytes in the buffer,
          ;     PTR points to beginning of data
          PHL
          LD   PTR_L,#<UART_BUFFER
          LD   PTR_H,#>UART_BUFFER
          LDA  #REP_CONT
          JSR  uart_xmit
          ;uart receiver loop
_cmdrx2   IN
          STA  FLAG
          JPF  _cmdrx2
          ;delay
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_cmdrx3   ;delay
          XOR  ACCU
          OR   ACCU
          LDA  ACCU
          INC
          CLC
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _cmdrx3
          ;store the received byte
          LDA  PAR2
          TST
          JPF  _cmdrx4  ; end of received string
          SAP
          INC  PTR_L
          TST  PTR_L
          JNF  _cmdrx1
          INC  PTR_H
_cmdrx1   ;wait until RXD is high again
          IN
          STA  FLAG
          JNF  _cmdrx1
          JMP  _cmdrx2

#endif ; FASTBAUD

#endif ; 10/20 MHz

;------------------------------------------------------------------------------

;UART routines for 22.1184 MHz CPU clock frequency:
#if (CPUCLOCK == 22)

#if defined(BAUD14400)	; 14400 baud


_rkxf02   ;low-condition on UART detected. Try to receive a byte.
          ;1. wait for one bit-time to skip the first zero (if host baud rate is too low)
          XOR  ACCU
          XOR  ACCU
          OR  ACCU
          ;2. wait for a high-level on RXD
_rkxf07   IN
          STA  FLAG
          JNF  _rkxf07
          ;hopefully we skiped this byte, and the high-level is the stop bit / idle state
          ;3. wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_rkxf08   IN
          STA  FLAG
          JNF  _rkxf05
          JLP  _rkxf08
          IN
          STA  FLAG
          JNF  _rkxf05
          DEC  R1
          TST  R1
          JNF  _rkxf08
          ;timeout, continue with read
          JMP  _cont_readkey
_rkxf05   ;4. receive the byte
          ;delay     
          INC      
          ;receive 8 bits
          LD   R0,#8
_rkxf06   ;delay
          XOR  ACCU
          ROR
          ROR                 
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _rkxf06
          ;all bits received
          LDA  PAR2
          ;check for special codes
          CMP  #CMD_LISTEN1
          JPF  _binmode_enter
          CMP  #CMD_LISTEN2
          JNF  _cont_readkey

_binmode_enter:  ;"LISTEN"-command received via UART
          ;send a reply to the host, showing him that we are in binary mode now.
          LDA  #REP_BINMODE
_binmode_loop_tx:
          JSR  uart_xmit
          
_binmode_loop:
          ;binary mode loop
          ;wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_bmlp01   IN
          STA  FLAG
          JNF  _bmlp02
          JLP  _bmlp01
          IN
          STA  FLAG
          JNF  _bmlp02
          DEC  R1
          TST  R1
          JNF  _bmlp01
          ;timeout, continue with read
          JMP  _cont_readkey
_bmlp02   ;receive the byte
          ;delay
          INC
          ;receive 8 bits
          LD   R0,#8
_bmlp03   ;delay         
          XOR  ACCU
          ROR
          ROR  
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _bmlp03
          ;all bits received
          LDA  PAR2
          JMP  _cmd_decode

_cmds_receive:
          ;Common code to all commands: receive the command string
          ;Out: R7 = bytes in the buffer,
          ;     PTR points to beginning of data
          PHL
          LD   PTR_L,#<UART_BUFFER
          LD   PTR_H,#>UART_BUFFER
          LDA  #REP_CONT
          JSR  uart_xmit
          ;uart receiver loop
_cmdrx2   IN
          STA  FLAG
          JPF  _cmdrx2
          ;delay
          INC
          ;receive 8 bits
          LD   R0,#8
_cmdrx3   ;delay
          XOR  ACCU
          ROR
          ROR                                  
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _cmdrx3
          ;store the received byte
          LDA  PAR2
          TST
          JPF  _cmdrx4  ; end of received string
          SAP
          INC  PTR_L
          TST  PTR_L
          JNF  _cmdrx1
          INC  PTR_H
_cmdrx1   ;wait until RXD is high again
          IN
          STA  FLAG
          JNF  _cmdrx1
          JMP  _cmdrx2

#elif defined(BAUD19200)

_rkxf02   ;low-condition on UART detected. Try to receive a byte.
          ;1. wait for one bit-time to skip the first zero (if host baud rate is too low)
          XOR ACCU
          ;XOR ACCU
          OR  ACCU
          ;2. wait for a high-level on RXD
_rkxf07   IN
          STA  FLAG
          JNF  _rkxf07
          ;hopefully we skiped this byte, and the high-level is the stop bit / idle state
          ;3. wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_rkxf08   IN
          STA  FLAG
          JNF  _rkxf05
          JLP  _rkxf08
          IN
          STA  FLAG
          JNF  _rkxf05
          DEC  R1
          TST  R1
          JNF  _rkxf08
          ;timeout, continue with read
          JMP  _cont_readkey
_rkxf05   ;4. receive the byte
          ;delay
          INC
          ;receive 8 bits
          LD   R0,#8
_rkxf06   ;delay
          OR  ACCU
          INC
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _rkxf06
          ;all bits received
          LDA  PAR2
          ;check for special codes
          CMP  #CMD_LISTEN1
          JPF  _binmode_enter
          CMP  #CMD_LISTEN2
          JNF  _cont_readkey

_binmode_enter:  ;"LISTEN"-command received via UART
          ;send a reply to the host, showing him that we are in binary mode now.
          LDA  #REP_BINMODE
_binmode_loop_tx:
          JSR  uart_xmit
          
_binmode_loop:
          ;binary mode loop
          ;wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_bmlp01   IN
          STA  FLAG
          JNF  _bmlp02
          JLP  _bmlp01
          IN
          STA  FLAG
          JNF  _bmlp02
          DEC  R1
          TST  R1
          JNF  _bmlp01
          ;timeout, continue with read
          JMP  _cont_readkey
_bmlp02   ;receive the byte
          ;delay
          INC
          ;receive 8 bits
          LD   R0,#8
_bmlp03   ;delay
          OR  ACCU
          INC             
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _bmlp03
          ;all bits received
          LDA  PAR2
          JMP  _cmd_decode

_cmds_receive:
          ;Common code to all commands: receive the command string
          ;Out: R7 = bytes in the buffer,
          ;     PTR points to beginning of data
          PHL
          LD   PTR_L,#<UART_BUFFER
          LD   PTR_H,#>UART_BUFFER
          LDA  #REP_CONT
          JSR  uart_xmit
          ;uart receiver loop
_cmdrx2   IN
          STA  FLAG
          JPF  _cmdrx2
          ;delay
          INC
          ;receive 8 bits
          LD   R0,#8
_cmdrx3   ;delay
          OR  ACCU
          INC
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _cmdrx3
          ;store the received byte
          LDA  PAR2
          TST
          JPF  _cmdrx4  ; end of received string
          SAP
          INC  PTR_L
          TST  PTR_L
          JNF  _cmdrx1
          INC  PTR_H
_cmdrx1   ;wait until RXD is high again
          IN
          STA  FLAG
          JNF  _cmdrx1
          JMP  _cmdrx2


#else ; 9600 baud

_rkxf02   ;low-condition on UART detected. Try to receive a byte.
          ;1. wait for one bit-time to skip the first zero (if host baud rate is too low)
          XOR  ACCU
          XOR  ACCU
          OR  ACCU
          ;2. wait for a high-level on RXD
_rkxf07   IN
          STA  FLAG
          JNF  _rkxf07
          ;hopefully we skiped this byte, and the high-level is the stop bit / idle state
          ;3. wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_rkxf08   IN
          STA  FLAG
          JNF  _rkxf05
          JLP  _rkxf08
          IN
          STA  FLAG
          JNF  _rkxf05
          DEC  R1
          TST  R1
          JNF  _rkxf08
          ;timeout, continue with read
          JMP  _cont_readkey
_rkxf05   ;4. receive the byte
          ;delay 
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_rkxf06   ;delay
          XOR  ACCU
          OR   ACCU
          OR   ACCU
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _rkxf06
          ;all bits received
          LDA  PAR2
          ;check for special codes
          CMP  #CMD_LISTEN1
          JPF  _binmode_enter
          CMP  #CMD_LISTEN2
          JNF  _cont_readkey

_binmode_enter:  ;"LISTEN"-command received via UART
          ;send a reply to the host, showing him that we are in binary mode now.
          LDA  #REP_BINMODE
_binmode_loop_tx:
          JSR  uart_xmit
          
_binmode_loop:
          ;binary mode loop
          ;wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_bmlp01   IN
          STA  FLAG
          JNF  _bmlp02
          JLP  _bmlp01
          IN
          STA  FLAG
          JNF  _bmlp02
          DEC  R1
          TST  R1
          JNF  _bmlp01
          ;timeout, continue with read
          JMP  _cont_readkey
_bmlp02   ;receive the byte
          ;delay
          INC
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_bmlp03   ;delay
          XOR  ACCU
          OR  ACCU
          OR  ACCU
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _bmlp03
          ;all bits received
          LDA  PAR2
          JMP  _cmd_decode

_cmds_receive:
          ;Common code to all commands: receive the command string
          ;Out: R7 = bytes in the buffer,
          ;     PTR points to beginning of data
          PHL
          LD   PTR_L,#<UART_BUFFER
          LD   PTR_H,#>UART_BUFFER
          LDA  #REP_CONT
          JSR  uart_xmit
          ;uart receiver loop
_cmdrx2   IN
          STA  FLAG
          JPF  _cmdrx2
          ;delay
          INC
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_cmdrx3   ;delay
          XOR  ACCU
          OR   ACCU
          OR   ACCU
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _cmdrx3
          ;store the received byte
          LDA  PAR2
          TST
          JPF  _cmdrx4  ; end of received string
          SAP
          INC  PTR_L
          TST  PTR_L
          JNF  _cmdrx1
          INC  PTR_H
_cmdrx1   ;wait until RXD is high again
          IN
          STA  FLAG
          JNF  _cmdrx1
          JMP  _cmdrx2

#endif ; FASTBAUD

#endif ; 22 MHz

;------------------------------------------------------------------------------

;UART routines for 12 MHz and 24 MHz CPU clock frequency:
#if (CPUCLOCK == 12) || (CPUCLOCK == 24)

_rkxf02   ;low-condition on UART detected. Try to receive a byte.
          ;1. wait for one bit-time to skip the first zero (if host baud rate is too low)
          XOR  ACCU
          XOR  ACCU
          XOR  ACCU
          ;2. wait for a high-level on RXD
_rkxf07   IN
          STA  FLAG
          JNF  _rkxf07
          ;hopefully we skiped this byte, and the high-level is the stop bit / idle state
          ;3. wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_rkxf08   IN
          STA  FLAG
          JNF  _rkxf05
          JLP  _rkxf08
          IN
          STA  FLAG
          JNF  _rkxf05
          DEC  R1
          TST  R1
          JNF  _rkxf08
          ;timeout, continue with read
          JMP  _cont_readkey
_rkxf05   ;4. receive the byte
          ;delay
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_rkxf06   ;delay
          XOR  ACCU
          XOR  ACCU
          OR   ACCU
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _rkxf06
          ;all bits received
          LDA  PAR2
          ;check for special codes
          CMP  #CMD_LISTEN1
          JPF  _binmode_enter
          CMP  #CMD_LISTEN2
          JNF  _cont_readkey

_binmode_enter:  ;"LISTEN"-command received via UART
          ;send a reply to the host, showing him that we are in binary mode now.
          LDA  #REP_BINMODE
_binmode_loop_tx:
          JSR  uart_xmit
          
_binmode_loop:
          ;binary mode loop
          ;wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_bmlp01   IN
          STA  FLAG
          JNF  _bmlp02
          JLP  _bmlp01
          IN
          STA  FLAG
          JNF  _bmlp02
          DEC  R1
          TST  R1
          JNF  _bmlp01
          ;timeout, continue with read
          JMP  _cont_readkey
_bmlp02   ;receive the byte
          ;delay
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_bmlp03   ;delay
          XOR  ACCU
          XOR  ACCU
          OR   ACCU
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _bmlp03
          ;all bits received
          LDA  PAR2
          JMP  _cmd_decode

_cmds_receive:
          ;Common code to all commands: receive the command string
          ;Out: R7 = bytes in the buffer,
          ;     PTR points to beginning of data
          PHL
          LD   PTR_L,#<UART_BUFFER
          LD   PTR_H,#>UART_BUFFER
          LDA  #REP_CONT
          JSR  uart_xmit
          ;uart receiver loop
_cmdrx2   IN
          STA  FLAG
          JPF  _cmdrx2
          ;delay
          INC
          INC
          ;receive 8 bits
          LD   R0,#8
_cmdrx3   ;delay
          XOR  ACCU
          XOR  ACCU
          OR   ACCU
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _cmdrx3
          ;store the received byte
          LDA  PAR2
          TST
          JPF  _cmdrx4  ; end of received string
          SAP
          INC  PTR_L
          TST  PTR_L
          JNF  _cmdrx1
          INC  PTR_H
_cmdrx1   ;wait until RXD is high again
          IN
          STA  FLAG
          JNF  _cmdrx1
          JMP  _cmdrx2

#endif ; 12/24 MHz


;------------------------------------------------------------------------------

;UART routines for 14 MHz and 28 MHz CPU clock frequency:
#if (CPUCLOCK == 14) || (CPUCLOCK == 28)

_rkxf02   ;low-condition on UART detected. Try to receive a byte.
          ;1. wait for one bit-time to skip the first zero (if host baud rate is too low)
          XOR  ACCU
          XOR  ACCU
          XOR  ACCU
          XOR  ACCU
          ;2. wait for a high-level on RXD
_rkxf07   IN
          STA  FLAG
          JNF  _rkxf07
          ;hopefully we skiped this byte, and the high-level is the stop bit / idle state
          ;3. wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_rkxf08   IN
          STA  FLAG
          JNF  _rkxf05
          JLP  _rkxf08
          IN
          STA  FLAG
          JNF  _rkxf05
          DEC  R1
          TST  R1
          JNF  _rkxf08
          ;timeout, continue with read
          JMP  _cont_readkey
_rkxf05   ;4. receive the byte
          ;delay
          XOR  ACCU
          ROR
          ;receive 8 bits
          LD   R0,#8
_rkxf06   ;delay
          XOR  ACCU
          XOR  ACCU
          XOR  ACCU
          INC
          CLC
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _rkxf06
          ;all bits received
          LDA  PAR2
          ;check for special codes
          CMP  #CMD_LISTEN1
          JPF  _binmode_enter
          CMP  #CMD_LISTEN2
          JNF  _cont_readkey

_binmode_enter:  ;"LISTEN"-command received via UART
          ;send a reply to the host, showing him that we are in binary mode now.
          LDA  #REP_BINMODE
_binmode_loop_tx:
          JSR  uart_xmit
          
_binmode_loop:
          ;binary mode loop
          ;wait for a low-level ("start condition") with timeout
          LD   R1,#40
          LD   R0,#0
_bmlp01   IN
          STA  FLAG
          JNF  _bmlp02
          JLP  _bmlp01
          IN
          STA  FLAG
          JNF  _bmlp02
          DEC  R1
          TST  R1
          JNF  _bmlp01
          ;timeout, continue with read
          JMP  _cont_readkey
_bmlp02   ;receive the byte
          ;delay
          XOR  ACCU
          ROR
          ;receive 8 bits
          LD   R0,#8
_bmlp03   ;delay
          XOR  ACCU
          XOR  ACCU
          XOR  ACCU
          INC
          CLC
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _bmlp03
          ;all bits received
          LDA  PAR2
          JMP  _cmd_decode

_cmds_receive:
          ;Common code to all commands: receive the command string
          ;Out: R7 = bytes in the buffer,
          ;     PTR points to beginning of data
          PHL
          LD   PTR_L,#<UART_BUFFER
          LD   PTR_H,#>UART_BUFFER
          LDA  #REP_CONT
          JSR  uart_xmit
          ;uart receiver loop
_cmdrx2   IN
          STA  FLAG
          JPF  _cmdrx2
          ;delay
          XOR  ACCU
          ROR
          ;receive 8 bits
          LD   R0,#8
_cmdrx3   ;delay
          XOR  ACCU
          XOR  ACCU
          XOR  ACCU
          INC
          CLC
          ;receive next bit
          IN
          STA  FLAG
          ROR  PAR2
          JLP  _cmdrx3
          ;store the received byte
          LDA  PAR2
          TST
          JPF  _cmdrx4  ; end of received string
          SAP
          INC  PTR_L
          TST  PTR_L
          JNF  _cmdrx1
          INC  PTR_H
_cmdrx1   ;wait until RXD is high again
          IN
          STA  FLAG
          JNF  _cmdrx1
          JMP  _cmdrx2

#endif ; 14/28 MHz

;rom_xfer_end: ;-----------------

#ifdef WITH_I2C_LCD_KYBD
#include "m4-i2c-lcd-kybd-fast/lcd-kybd-utils4.asm"
#endif



;----------------------------------------
; timing un-critical code follows here:
SEG_PRGCODE

#ifdef EXCLUDE_TRANSFER
_cmd_decode:
          LDA  #REP_CERR
          JMP  _binmode_loop_tx
txferr_errorback:
txferr_okback:
_cmdrx4   RTS
#else

_cmdrx4   ;finished. check if forth code got overwritten.
          LDA  PTR_H
          SU   #>FORTH_START
          JNF  _cmdrx5
          LD   LIB_TFDATA+5,#1  ; set flag that forth code might by overwritten now
_cmdrx5   ;calculate number of bytes in buffer
          LDA  PTR_L
          SU   #<UART_BUFFER
          STA  R7_L
          LDA  PTR_H
          SUB  #>UART_BUFFER
          STA  R7_H
          ;init buffer pointer
          LD   PTR_L,#<UART_BUFFER
          LD   PTR_H,#>UART_BUFFER
          RTS


_cmd_decode:
          ;decode the command
          CMP  #CMD_LISTEN1
          JPF  _binmode_enter
          CMP  #CMD_LISTEN2
          JPF  _binmode_enter
          CMP  #CMD_EXECUTE
          JPF  _cmd_exec
          CMP  #CMD_READ1K
          JPF  _cmd_read1k
          CMP  #CMD_WRITE1K
          JPF  _cmd_write1k
          TST
          JPF  _binmode_loop  ;NOP
          ;unknown command
          LDA  #REP_CERR
          JMP  _binmode_loop_tx


_cmd_exec:
          ;execute a text string
          JSR  _cmds_receive
          TST  LIB_TFDATA+5
          JPF  _cmdex05
          ; received string too long, FORTH program memory was overwritten
          LDA  #REP_NACK
          JSR  uart_xmit
          JMP  restart_forth
_cmdex05  ;get CRC-sum, store it in LIB_TFDATA+2/+3
          JSR  read_hex4
          JPF  _cmd_ret_nak
          ;PTR points to data, R7 = length of data
          ;calculate the checksum
          LD   R4_L,#0     ;checksum
          LD   R4_H,#0     ;checksum
          LD   R5_L,PTR_L
          LD   R5_H,PTR_H
          ;R7 contains the number of received bytes
_cmdex01  ;loop over all the data
          TST  R7_L
          JNF  _cmdex02
          TST  R7_H
          JPF  _cmdex03
          DEC  R7_H
_cmdex02  DEC  R7_L
          ;read next byte
          JSR  ld_ptr_r5
          LAP
          INC  R5_L
          TST  R5_L
          JNF  _cmdex04
          INC  R5_H
_cmdex04  ;update the checksum
          XOR  R4_H
          STA  PTR_L
          LD   PTR_H,#>TAB_CRC16HI
          LAP
          XOR  R4_L
          STA  R4_H
          LD   PTR_H,#>TAB_CRC16LO
          LAP
          STA  R4_L
          JMP  _cmdex01
_cmdex03  ;test the checksum
          LDA  R4_L
          CMP  LIB_TFDATA+2
          JNF  _cmd_ret_nak
          LDA  R4_H
          CMP  LIB_TFDATA+3
          JNF  _cmd_ret_nak
          ;checksum correct
          LDA  #REP_ACK
          JSR  uart_xmit
          ;store terminating zero
          JSR  ld_ptr_r5
          LDA  #0
          SAP
          ;set pointer to beginning of the buffer
          LD   PTR_L,#<(UART_BUFFER+4)
          LD   PTR_H,#>(UART_BUFFER+4)
          ;execute the string
          JMP  execRemoteString

txferr_errorback:
          ;this function is called from execRemoteString when
          ;the string execution failed
          PHL  ;push LR to stack so transfer can RTS to the calling forth module (instead of polling the keyboard)
          JMP  _cmd_ret_cerr

txferr_okback:
          ;this function is called from execRemoteString when
          ;the string execution succeeded
          PHL  ;push LR to stack so transfer can RTS to the calling forth module (instead of polling the keyboard)
          JMP  _cmd_ret_cok


_cmd_read1k:
          ;read 1kB from EEPROM and transmit it over UART
          JSR  _cmds_receive
          JSR  read_blocknbr
          JPF  _binmode_loop_tx  ; error, the error-code is in ACCU
          ;PTR = pointer to next data
          ;R7 = remaining data length
          ;R4 = block number
          LDA  R7_L
          OR   R7_H
          TST
          JNF  _cmd_ret_nak
          ;send ACK, command successfully received
          LDA  #REP_ACK
          JSR  uart_xmit
          ;start reading from EEPROM
          JSR  fblockToEepromAddr
          JPF  _cmd_ret_cerr
          JSR  eeprom_start_read
          JPF  _cmd_ret_cerr
          ;initialize checksum accu
          LD   R4_L,#0
          LD   R4_H,#0
          ;load counter for 1kB (remaining bytes)
          LD   R6_L,#0xFF
          LD   R6_H,#0x04
#ifdef PLATFORM_XS
          ;On the XS-platform it is not possible to use RS232 and I2C at the same time.
          ;So the data must be read from EEPROM first, before it can be sent over RS232.
          LD   LIB_TFDATA+5,#1  ; set flag that forth code might by overwritten now
          ;loop: read 1024 bytes into temporary buffer
          LD   PTR_L,#<HEAP_START
          LD   PTR_H,#>HEAP_START
          LD   R1,#0
_cmdrd02  DEC  R6_H
_cmdrd01  DEC  R6_L
          JSR  i2c_recv
          JSR  sap_inc_ptr
          TST  R6_L
          JNF  _cmdrd01
          TST  R6_H
          JNF  _cmdrd02
          LD   R1,#1
          JSR  i2c_recv
          SAP
          JSR  i2c_stop
          ;loop: send 1024 bytes over RS232
          LD   R6_H,#0x04
          LD   PTR_L,#<HEAP_START
          LD   PTR_H,#>HEAP_START
_cmdrd03  DEC  R6_H
_cmdrd04  DEC  R6_L
          JSR  lap_inc_ptr
          JSR  psh_ptr
          JSR  _cmdr_snd
          JSR  pop_ptr
          TST  R6_L
          JNF  _cmdrd04
          TST  R6_H
          JNF  _cmdrd03
#else     ;-------------------------------
          ;send 1023 bytes
          LD   R1,#0
_cmdrd02  DEC  R6_H
_cmdrd01  DEC  R6_L
          JSR  i2c_recv
          JSR  _cmdr_snd
          TST  R6_L
          JNF  _cmdrd01
          TST  R6_H
          JNF  _cmdrd02
          ;send last byte
          LD   R1,#1
          JSR  i2c_recv
          JSR  _cmdr_snd
          JSR  i2c_stop
#endif
          ;send checksum as hex-16
          JSR  send_hex4
          ;send the terminating zero
          LDA  #0
          JSR  uart_xmit
          JMP  _binmode_loop

_cmdr_snd ;send one byte
          STA  R3
          ;update the checksum
          XOR  R4_H
          STA  PTR_L
          LD   PTR_H,#>TAB_CRC16HI
          LAP
          XOR  R4_L
          STA  R4_H
          LD   PTR_H,#>TAB_CRC16LO
          LAP
          STA  R4_L
          ;escape the data (which is in R3)
          LDA  R3
          ROL
          JPF  _cmdrs01
          ;the value is < 0x80
          LDA  #0x60
          AND  R3
          TST
          LDA  R3
          JNF  uart_xmit  ; no need to escape the data, send it now
          ;the value is < 0x20
          PHL
          LDA  #ESC_00_1F
          JSR  uart_xmit
          LDA  #0x40
          OR   R3
          JSR  uart_xmit
          RTS
_cmdrs01  ;the value is >= 0x80
          PHL
          LDA  #0x60
          AND  R3
          TST
          JPF  _cmdrs02  ;the value is in the range 0x80 - 0x9F
          ;the value is in the range 0xA0 - 0xFF
          LDA  #ESC_A0_FF
          JSR  uart_xmit
          LDA  R3
          ROL
          CLC
          ROR
          JSR  uart_xmit
          RTS
_cmdrs02  LDA  #ESC_80_9F
          JSR  uart_xmit
          LDA  #0x1F
          AND  R3
          OR   #0x40
          JSR  uart_xmit
          RTS


_cmd_write1k:
          ;receive 1kB over UART and write it into the EEPROM   
          JSR  _cmds_receive
          LDA  #REP_BUSY
          JSR  uart_xmit
          JSR  read_blocknbr
          JPF  _binmode_loop_tx  ; error, the error-code is in ACCU
          ;PTR = pointer to next data
          ;R7 = remaining data length
          ;R4 = block number
          ;Move block number to LIB_TFDATA+0/+1
          LD   LIB_TFDATA+0,R4_L
          LD   LIB_TFDATA+1,R4_H
          ;get CRC-sum, store it in LIB_TFDATA+2/+3
          JSR  read_hex4
          JPF  _cmd_ret_nak
          ;de-escape the data and calculate the checksum
          LD   R4_L,#0     ;checksum
          LD   R4_H,#0     ;checksum
          LD   R5_L,PTR_L  ;source ptr
          LD   R5_H,PTR_H  ;source ptr
          LD   R6_L,#<UART_BUFFER   ;destination pointer
          LD   R6_H,#>UART_BUFFER   ;destination pointer
          ;R7 contains the number of remaining bytes to decode
_cmdwr01  ;loop over all the data
          TST  R7_L
          JNF  _cmdwr02
          TST  R7_H
          JPF  _cmdwr03
          DEC  R7_H
_cmdwr02  DEC  R7_L
          ;read next byte
          LD   PTR_L,R5_L
          LD   PTR_H,R5_H
          LAP
          INC  R5_L
          TST  R5_L
          JNF  _cmdwr04
          INC  R5_H
_cmdwr04  STA  R1
          LDA  #0x60
          AND  R1
          TST
          JNF  _cmdwr05
          ;R1 is a control character, get next byte from buffer
          TST  R7_L
          JNF  _cmdwr09
          TST  R7_H
          JPF  _cmd_ret_nak
          DEC  R7_H
_cmdwr09  DEC  R7_L
          JSR  ld_ptr_r5
          LAP
          INC  R5_L
          TST  R5_L
          JNF  _cmdwr10
          INC  R5_H
_cmdwr10  STA  R2
          ;decode the control character, the next byte is already in R2.
          LDA  R1
          CMP  #ESC_A0_FF
          JPF  _cmdwr07
          CMP  #ESC_80_9F
          JPF  _cmdwr08
          CMP  #ESC_00_1F
          JNF  _cmd_ret_nak
          ;the byte value is in the range 0x00-0x1F
          LDA  #0x1F
          AND  R2
          JMP  _cmdwr11
_cmdwr07  ;the byte value is in the range 0xA0-0xFF
          LDA  #0x80
          OR   R2
          JMP  _cmdwr11
_cmdwr08  ;the byte value is in the range 0x80-0x9F
          LDA  #0x1F
          AND  R2
          OR   #0x80
          STA  R1
_cmdwr05  ;R1 = de-escaped data byte
          LDA  R1
_cmdwr11  ;write data in ACCU to RAM
          LD   PTR_L,R6_L
          LD   PTR_H,R6_H
          SAP
          INC  R6_L
          TST  R6_L
          JNF  _cmdwr06
          INC  R6_H
_cmdwr06  ;update the checksum
          XOR  R4_H
          STA  PTR_L
          LD   PTR_H,#>TAB_CRC16HI
          LAP
          XOR  R4_L
          STA  R4_H
          LD   PTR_H,#>TAB_CRC16LO
          LAP
          STA  R4_L
          ;next byte
          JMP  _cmdwr01

_cmdwr03  ;test checksum
          LDA  R4_L
          CMP  LIB_TFDATA+2
          JNF  _cmd_ret_nak
          LDA  R4_H
          CMP  LIB_TFDATA+3
          JNF  _cmd_ret_nak
          LDA  #REP_ACK
          JSR  uart_xmit
          ;write data to EEPROM
          LD   R4_L,LIB_TFDATA+0
          LD   R4_H,LIB_TFDATA+1
          LD   R5_L,#<UART_BUFFER
          LD   R5_H,#>UART_BUFFER
          JSR  eeprom_write_forth_block
          JPF  _cmd_ret_nak
_cmd_ret_cok:
          ;all ok
          LDA  #REP_COK
          DB   OPCODE_LDI  ; trick: jumps over next two bytes, but overwrites register TEMP5
_cmd_ret_nak:
          LDA  #REP_NACK
          DB   OPCODE_LDI  ; trick: jumps over next two bytes, but overwrites register TEMP5
_cmd_ret_cerr:
          LDA  #REP_CERR
          JMP  _binmode_loop_tx



read_blocknbr:
          ;Read a hex-coded block number from UART buffer.
          ;Checks the amount of bytes in the buffer.
          ;In: PTR = pointer to begin of received data, R7 = number of received bytes
          ;Out: PTR points behind number, R4 = converted number, FLAG = 0 on success
          ;     R7 = remaining bytes in the buffer, ACCU = error code if FLAG = 1
          ;Changes: ACCU, R0, R1
          PHL
          ;read hex value
          JSR  read_hex4
          JPF  _rbnbr2
          ;test checksum
          LDA  R4_L
          XOR  R4_H
          STA  PAR2
          LD   R0,#4
_rbnbr1   CLC
          ROR
          JLP  _rbnbr1
          XOR  PAR2
          LD   PAR1,#0x0F
          AND  PAR1
          TST
          JNF  _rbnbr2
          LDA  R4_H
          AND  PAR1
          STA  R4_H
          ;check range of block number
          LDA  R4_L
          SU   LIB_NUMPAGES
          LDA  R4_H
          SUB  LIB_NUMPAGES+1
          LDA  #REP_CERR
          RTS  ;return FLAG=0 on success, otherwise FLAG=1 and ACCU = error code
_rbnbr2   LDA  #REP_NACK
          SEC
          RTS


read_hex4:
          ;Convert a 4-character hex-number
          ;In: PTR = pointer to hex number, R7 = length of input buffer
          ;Out: PTR points behind number, R4 and LIB_TFDATA+2/3 = converted number,
          ;     R7 = length - 4 bytes, FLAG = 0 on success
          ;Changes: ACCU, R0, R1
          PHL
          LDA  R7_L
          SU   #4
          STA  R7_L
          LDA  R7_H
          SUB  #0
          STA  R7_H
          JNF  return_flag_1
          LD   R1,#4
_rdhx0    LD   R0,#4
_rdhx1    CLC
          RWL  R4
          JLP  _rdhx1
          JSR  lap_inc_ptr
          JSR  is_hexchar
          JPF  rts_opc
          OR   R4_L
          STA  R4_L
          DEC  R1
          TST  R1
          JNF  _rdhx0
          LD   LIB_TFDATA+2,R4_L
          LD   LIB_TFDATA+3,R4_H
          CLC
          RTS


send_hex4:
          ;Send a word (16-bit) in hexadecimal over uart
          ;In: word in R4
          ;Changes: ACCU
          LDA  R4_H
          PHL
          JSR  _snhxw3
          POP  LR_H
          POP  LR_L
          LDA  R4_L
_snhxw3   PSH  ACCU
          CLC
          ROR
          CLC
          ROR
          CLC
          ROR
          CLC
          ROR
          PHL
          JSR  _snhxw2
          POP  LR_H
          POP  LR_L
          POP  ACCU
          AND  #0Fh
_snhxw2   SU   #10
          JPF  _snhxw1
          ADD  #3Ah
          JMP  uart_xmit
_snhxw1   ADD  #40h
          JMP  uart_xmit

#endif

SEG_PRGCODE
